<?php

/**
 * This file contains PCSG\Api\Ebay
 */

namespace PCSG\Api;

use DTS\eBaySDK\Finding\Services\FindingService;
use DTS\eBaySDK\Finding\Types\FindItemsAdvancedRequest;
use DTS\eBaySDK\Finding\Types\FindItemsByKeywordsRequest;
use DTS\eBaySDK\Finding\Types\FindItemsBySellerRequest;
use DTS\eBaySDK\Finding\Types\ItemFilter;
use DTS\eBaySDK\Finding\Types\PaginationInput;
use DTS\eBaySDK\OAuth\Services\OAuthService;
use DTS\eBaySDK\OAuth\Types\GetAppTokenRestRequest;
use DTS\eBaySDK\Shopping\Services\ShoppingService;
use DTS\eBaySDK\Shopping\Types\FindItemsByKeywordsRequestType;
use DTS\eBaySDK\Shopping\Types\GetSingleItemRequestType;
use DTS\eBaySDK\Trading\Services\TradingService;
use DTS\eBaySDK\Trading\Types\CustomSecurityHeaderType;
use DTS\eBaySDK\Trading\Types\GetUserRequestType;
use QUI;
use QUI\Exception;
use Stash\Driver\Apc;
use Stash\Driver\FileSystem;
use Stash\Pool;

/**
 * PCSG eBay API
 * Mini API access
 *
 * @author www.pcsg.de (Henning Leutz)
 * @author www.pcsg.de (Patrick Müller)
 *
 * API Status:
 * https://go.developer.ebay.com/api-status
 *
 * Release Notes:
 * http://developer.ebay.com/devzone/xml/docs/releasenotes.html#957
 */
class Ebay extends QUI\QDOM
{
    protected FindingService $service;
    protected ShoppingService $shoppingService;

    protected OAuthService $oAuthService;
    protected mixed $oAuthToken;

    /**
     * Internal Session-ID
     *
     * @var string
     */
    protected string $Session;

    /**
     * Constructor
     *
     * @param array $attributes - optional
     *
     * @throws \Exception
     */
    public function __construct(array $attributes = [])
    {
        // Default attributes
        $this->setAttributes([
            'SECURITY-APPNAME' => '',
            'GLOBAL-ID' => 'EBAY-DE',
            'OPERATION-NAME' => 'findItemsAdvanced',
            'SERVICE-VERSION' => '1.11.0',
            'REST-PAYLOAD' => '',
            'affiliate.networkId' => 9,
            'affiliate.trackingId' => '5336210971',
            'affiliate.customId' => '5336210971',
            'sandbox' => false,
            'debug' => false,
            'DEV-NAME' => '',
            'CERT-NAME' => '',
            'COMPATIBILITY-LEVEL' => 681,
            'CALL-NAME' => 'FetchToken',
            'RuName' => '',
            'authToken' => '',
            'oauthUserToken' => '',
        ]);

        // Set user-provided attributes
        $this->setAttributes($attributes);

        // check app id
        if (!$this->getAttribute('SECURITY-APPNAME')) {
            throw new \Exception(
                'No Access. No eBay SECURITY-APPNAME given',
                401
            );
        }

        $this->oAuthService = new OAuthService([
            'credentials' => [
                'appId' => $this->getAttribute('SECURITY-APPNAME'),
                'certId' => $this->getAttribute('CERT-NAME'),
                'devId' => $this->getAttribute('DEV-NAME')
            ],
            'sandbox' => $this->getAttribute('sandbox'),
            'ruName' => $this->getAttribute('RuName')
        ]);

        $getAppToken = new GetAppTokenRestRequest();
        $getAppToken->grant_type = 'client_credentials';

        $response = $this->oAuthService->getAppToken($getAppToken);
        $this->oAuthToken = $response->access_token;

        // Initialize eBay services using attributes
        $this->service = new FindingService([
            'credentials' => [
                'appId' => $this->getAttribute('SECURITY-APPNAME'),
                'certId' => $this->getAttribute('CERT-NAME'),
                'devId' => $this->getAttribute('DEV-NAME')
            ],
            'sandbox' => $this->getAttribute('sandbox'),
            'ruName' => $this->getAttribute('RuName'),
            'authorization' => $this->oAuthToken
        ]);

        $this->shoppingService = new ShoppingService([
            'credentials' => [
                'appId' => $this->getAttribute('SECURITY-APPNAME'),
                'certId' => $this->getAttribute('CERT-NAME'),
                'devId' => $this->getAttribute('DEV-NAME')
            ],
            'sandbox' => $this->getAttribute('sandbox'),
            'ruName' => $this->getAttribute('RuName'),
            'authorization' => $this->oAuthToken
        ]);
    }

    /**
     * Make a normal eBay request with the set attributes
     *
     * @return array
     */
    public function search(): array
    {
        $attr = $this->getFindingServiceAttributes();
        $request = new FindItemsByKeywordsRequest();
        $request->keywords = $attr['keywords'];
        $response = $this->service->findItemsByKeywords($request);

        return $this->parseItems($response);
    }

    /**
     * Return the Articles from the seller
     *
     * @param string|boolean $sellerId - seller id, seller name
     * @param bool $noCache - should the result be cached?
     *
     * @return array
     * @throws \Exception
     */
    public function getItemsBySeller(string | bool $sellerId = false, bool $noCache = false): array
    {
        if (empty($sellerId)) {
            throw new QUI\Exception('$sellerid is empty');
        }

        $cachePool = self::getStash();
        $cacheKey = 'findItemsAdvanced/' . $sellerId;
        $cacheItem = $cachePool->getItem($cacheKey);

        if (!$cacheItem->isMiss()) {
            return $cacheItem->get();
        }

        $appId = $this->getAttribute('SECURITY-APPNAME');
        $globalId = $this->getAttribute('GLOBAL-ID');

        $endpoint = $this->getAttribute('sandbox') ?
            'https://svcs.sandbox.ebay.com/services/search/FindingService/v1' :
            'https://svcs.ebay.com/services/search/FindingService/v1';

        $headers = [
            'X-EBAY-SOA-SECURITY-APPNAME: ' . $appId,
            'X-EBAY-SOA-OPERATION-NAME: findItemsAdvanced',
            'X-EBAY-SOA-SERVICE-VERSION: 1.13.0',
            'X-EBAY-SOA-GLOBAL-ID: ' . $globalId,
            'X-EBAY-SOA-RESPONSE-DATA-FORMAT: JSON',
            'Content-Type: text/xml'
        ];

        $xmlBody = '<?xml version="1.0" encoding="utf-8"?>' .
            '<findItemsAdvancedRequest xmlns="http://www.ebay.com/marketplace/search/v1/services">' .
            '  <paginationInput><entriesPerPage>10</entriesPerPage></paginationInput>' .
            '  <itemFilter><name>Seller</name><value>' . htmlspecialchars($sellerId) . '</value></itemFilter>' .
            '  <itemFilter><name>LocatedIn</name><value>Worldwide</value></itemFilter>' .
            '</findItemsAdvancedRequest>';


        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $endpoint);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlBody);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            throw new \Exception('cURL Error: ' . curl_error($ch));
        }

        curl_close($ch);

        $response = json_decode($response, true);

        // Fehlerprüfung
        if (isset($response['errorMessage'])) {
            foreach ($response['errorMessage'] as $errors) {
                foreach ($errors as $error) {
                    $errorId = is_array($error[0]['errorId']) ? implode(
                        ', ',
                        $error[0]['errorId']
                    ) : $error[0]['errorId'];
                    $message = is_array($error[0]['message']) ? implode(
                        ', ',
                        $error[0]['message']
                    ) : $error[0]['message'];

                    throw new \Exception($message, $errorId);
                }
            }
        }


        // Daten verarbeiten
        $data = $this->parseItems($response);

        $cacheItem->set($data)->expiresAfter(1800);
        $cachePool->save($cacheItem);

        return $data;
    }

    /**
     * Returns the number of Articles from a seller
     *
     * @param string|boolean $sellerID - seller id, seller name
     * @return integer
     * @throws \Exception
     */
    public function getNumberItemsBySeller(string | bool $sellerID = false): int
    {
        $pool = self::getStash();
        $cacheKey = 'numberItemsBySellerId/' . $sellerID;
        $item = $pool->getItem($cacheKey);
        $data = $item->get();

        if ($item->isMiss()) {
            $items = $this->getItemsBySeller($sellerID);
            $data = count($items);
            $item->set($data)->expiresAfter(1800);
            $pool->save($item);
        }

        return $data;
    }

    /**
     * Returns information about an article
     *
     * @param string $articleId
     * @return array
     */
    public function getItemById(string $articleId): array
    {
        $sideId = $this->globalIDToSiteID($this->getAttribute('GLOBAL-ID'));
        $pool = self::getStash();
        $cacheKey = 'itemsById/' . $articleId . '/' . $sideId;
        $cache = $pool->getItem($cacheKey);
        $data = $cache->get();

        if ($cache->isMiss()) {
            $cache->lock();

            $request = new GetSingleItemRequestType();
            $request->ItemID = $articleId;
            $request->IncludeSelector = 'ItemSpecifics,Details';

            $response = $this->shoppingService->getSingleItem($request);
            $item = $response->Item ?? null;
            $data = $item->toArray();

            // Falls Artikel nicht gefunden, speichere Fehler dauerhaft
            if (!$data) {
                $cache->set($data);
            } else {
                $cache->set($data)->expiresAfter(600);
            }

            $pool->save($cache);
        }

        if (!is_array($data)) {
            $data = [];
        }

        return $data;
    }

    /**
     * Get number of results for a specific search term
     *
     * @param string $searchStr - Search string
     * @return int
     */
    public function getItemSearchResultCount(string $searchStr): int
    {
        $pool = self::getStash();
        $cacheKey = 'searchResultCount/' . md5($searchStr);
        $item = $pool->getItem($cacheKey);
        $data = $item->get();

        if ($item->isMiss()) {
            $attr = $this->getFindingServiceAttributes();
            $request = new FindItemsByKeywordsRequest();
            $request->keywords = $searchStr;

            $pagination = new \DTS\eBaySDK\Finding\Types\PaginationInput();
            $pagination->entriesPerPage = 1;
            $request->paginationInput = $pagination;

            $response = $this->service->findItemsByKeywords($request);
            $data = $response->paginationOutput->totalEntries ?? 0;

            $item->set($data)->expiresAfter(1800);
            $pool->save($item);
        }

        return $data;
    }

    /**
     * Check if an eBay username is taken
     *
     * @param string $username
     * @return bool
     */
    public function existsUsername(string $username): bool
    {
        $tradingService = new TradingService([
            'credentials' => [
                'appId' => $this->getAttribute('SECURITY-APPNAME'),
                'certId' => $this->getAttribute('CERT-NAME'),
                'devId' => $this->getAttribute('DEV-NAME')
            ],
            'sandbox' => $this->getAttribute('sandbox'),
            'siteId' => $this->globalIDToSiteID($this->getAttribute('GLOBAL-ID')),
            'authorization' => $this->getAttribute('userToken')
        ]);

        $request = new GetUserRequestType();
        $request->RequesterCredentials = new CustomSecurityHeaderType();
        $request->RequesterCredentials->eBayAuthToken = $this->oAuthToken;
        $request->UserID = $username;

        try {
            $response = $tradingService->getUser($request);

            if ($response->Ack !== 'Success') {
                $message = "Fehler bei GetUser:\n" .
                    "Fehlercode: " . ($response->Errors[0]->ErrorCode ?? 'Unbekannt') . "\n" .
                    "Kurzbeschreibung: " . ($response->Errors[0]->ShortMessage ?? 'Keine') . "\n" .
                    "Lange Beschreibung: " . ($response->Errors[0]->LongMessage ?? 'Keine');

                throw new \Exception(
                    $message,
                    $response->Errors[0]->ErrorCode
                );
            }

            return $response->Ack === 'Success';
        } catch (\Exception) {
            return false;
        }
    }

    /**
     * Return the
     *
     * @return array
     */
    protected function getFindingServiceAttributes(): array
    {
        $attr = $this->getAttributes();

        // old api compatiblity
        $attr['RESPONSE-DATA-FORMAT'] = 'JSON';

        if (!isset($attr['paginationInput.entriesPerPage'])) {
            $attr['paginationInput.entriesPerPage'] = 10;
        }

        if (!isset($attr['outputSelector'])) {
            $attr['outputSelector'] = 'PictureURLLarge';
        }

        if (!isset($attr['keywords'])) {
            $attr['keywords'] = '';
        }

        return $attr;
    }

    /**
     * Parse the return of a FindingService request
     *
     * @param array $data
     * @return array
     */
    protected function parseItems(array $data): array
    {
        if (!isset($data->searchResult) || empty($data->searchResult->item)) {
            return [];
        }

        $result = [];

        foreach ($data->searchResult->item as $item) {
            $result[] = [
                'itemId' => $item->itemId,
                'title' => $item->title,
                'endTime' => $item->listingInfo->endTime ?? '',
                'url' => $item->viewItemURL,
                'listingType' => $item->listingInfo->listingType ?? '',
                'imageUrl' => isset($item->pictureURLLarge) ? str_replace(
                    'http://',
                    'https://',
                    $item->pictureURLLarge
                ) : '',
                'galleryURL' => isset($item->galleryURL) ? str_replace('http://', 'https://', $item->galleryURL) : '',
                'primaryCategoryId' => $item->primaryCategory->categoryId ?? '',
                'primaryCategoryName' => $item->primaryCategory->categoryName ?? '',
                'price' => [
                    'value' => $item->sellingStatus->currentPrice->value ?? 0,
                    'currency' => $item->sellingStatus->currentPrice->currencyId ?? ''
                ],
                'unitPrice' => $item->unitPrice ?? null,
                'timeLeft' => isset($item->sellingStatus->timeLeft) ? $this->getPrettyTimeFromEbayTime(
                    $item->sellingStatus->timeLeft
                ) : '',
                'shippingCost' => [
                    'value' => isset($item->shippingInfo->shippingServiceCost) ? (float)$item->shippingInfo->shippingServiceCost->value : 0,
                    'currency' => $item->shippingInfo->shippingServiceCost->currencyId ?? ''
                ],
                'seller' => [
                    'username' => $item->sellerInfo->sellerUserName ?? '',
                    'feedbackScore' => $item->sellerInfo->feedbackScore ?? 0,
                    'positiveFeedbackPercent' => $item->sellerInfo->positiveFeedbackPercent ?? 0,
                    'topRatedSeller' => $item->sellerInfo->topRatedSeller ?? false
                ]
            ];
        }

        return $result;
    }

    /**
     * Convert the eBay timestamp to a pretty format
     *
     * @param string $eBayTimeString Input is of form 'PT12M25S'
     * @return string
     */
    public static function getPrettyTimeFromEbayTime(string $eBayTimeString): string
    {
        if ($eBayTimeString == 'PT0S') {
            $data = "Auktion beendet.";

            return $data;
        }

        $matchAry = [];
        $pattern = "#P([0-9]{0,3}D)?T([0-9]?[0-9]H)?([0-9]?[0-9]M)?([0-9]?[0-9]S)#msiU";
        preg_match($pattern, $eBayTimeString, $matchAry);

        $days = false;
        $hours = false;
        $min = false;
        $sec = false;

        if (isset($matchAry[1])) {
            $days = (int)$matchAry[1];
        }

        if (isset($matchAry[2])) {
            $hours = (int)$matchAry[2];
        }

        if (isset($matchAry[3])) {
            $min = (int)$matchAry[3];
        }

        if (isset($matchAry[4])) {
            $sec = (int)$matchAry[4];
        }

        $retnStr = '';

        if ($days) {
            $retnStr .= "$days Tag(e),";
        }

        if ($hours) {
            $retnStr .= " $hours Stunde(n),";
        }

        if ($min) {
            $retnStr .= " $min Minute(n),";
        }

        if ($sec) {
            $retnStr .= " $sec Sekunde(n)";
        }

        return $retnStr;
    }

    /**
     * If a session already exist you can set the internal Session-ID
     *
     * @param string $SessID
     */
    public function setSession(string $SessID): void
    {
        $this->Session = $SessID;
    }

    /**
     * Create a ebay session, if a session is set, this session would be return
     *
     * @throws Exception
     * @deprecated use oAuth tokens
     */
    public function getSession(): string
    {
        throw new QUI\Exception('Please use getToken. getSession do not work anymore');
    }

    /**
     * Ask for a token
     * - returns the oAuth token
     */
    public function getToken(): string
    {
        return $this->oAuthToken;
    }

    /**
     * Fetch the sign in form for the session
     *
     * @return string
     */
    public function signInForm(): string
    {
        return file_get_contents($this->signInUrl());
    }

    /**
     * Return the sign in Url
     *
     * @return string
     * @throws Exception
     */
    public function signInUrl(): string
    {
        $url = 'https://signin.ebay.com/ws/eBayISAPI.dll?';

        if ($this->getAttribute('sandbox')) {
            $url = 'https://signin.sandbox.ebay.com/ws/eBayISAPI.dll?';
        }

        $attr = [
            'RuName' => $this->getAttribute('RuName'),
            'SessID' => $this->getSession()
        ];

        return $url . http_build_query($attr);
    }

    /**
     * Return the difference between two dates
     *
     * @param string $start - start date, ebay date
     * @param string|boolean $end - end date, ebay date
     * @return string
     */
    public function dateDiff(string $start, string | bool $end = false): string
    {
        try {
            $start = new \DateTime($start);
            $end = new \DateTime($end);

            $form = $start->diff($end);
        } catch (\Exception $Exception) {
            return $Exception->getMessage();
        }

        $hour = sprintf("%02d", $form->h);
        $min = sprintf("%02d", $form->i);
        $sec = sprintf("%02d", $form->s);

        $year = sprintf("%02d", $form->y);
        $month = sprintf("%02d", $form->m);
        $days = sprintf("%02d", $form->d);

        $result = '';

        if ($form->y || $form->m || $form->d) {
            if (!$form->y || !$form->m || $form->d) {
                $result = "{$days} Tage, ";
            } else {
                $result = "{$days} Tage, {$month} Monate, {$year} Jahre ";
            }
        }

        $result .= "{$hour}:{$min}:{$sec}";

        return $result;
    }

    /**
     * Build the ebay request header for authentication
     *
     * @return array
     */
    protected function buildHeaders(): array
    {
        return [
            // Regulates versioning of the XML interface for the API
            'X-EBAY-API-COMPATIBILITY-LEVEL: ' . $this->getAttribute('COMPATIBILITY-LEVEL'),
            //set the keys
            'X-EBAY-API-DEV-NAME: ' . $this->getAttribute('DEV-NAME'),
            'X-EBAY-API-APP-NAME: ' . $this->getAttribute('SECURITY-APPNAME'),
            'X-EBAY-API-CERT-NAME: ' . $this->getAttribute('CERT-NAME'),
            'X-EBAY-API-CALL-NAME: ' . $this->getAttribute('CALL-NAME'),
            'X-EBAY-API-SITEID: ' . $this->globalIDToSiteID(
                $this->getAttribute('GLOBAL-ID')
            )
        ];
    }

    /**
     * Return the eBay Site-ID of the eBay Global-ID
     *
     * @param string $globalID
     *
     * @return integer
     */
    public static function globalIDToSiteID(string $globalID): int
    {
        $ids = [
            'EBAY-AT' => 16,
            'EBAY-AU' => 15,
            'EBAY-CH' => 193,
            'EBAY-DE' => 77,
            'EBAY-ENCA' => 2,
            'EBAY-ES' => 186,
            'EBAY-FR' => 71,
            'EBAY-FRBE' => 23,
            'EBAY-FRCA' => 210,
            'EBAY-GB' => 3,
            'EBAY-HK' => 201,
            'EBAY-IE' => 205,
            'EBAY-IN' => 203,
            'EBAY-IT' => 101,
            'EBAY-MOTOR' => 100,
            'EBAY-MY' => 207,
            'EBAY-NL' => 146,
            'EBAY-NLBE' => 123,
            'EBAY-PH' => 211,
            'EBAY-PL' => 212,
            'EBAY-SG' => 216,
            'EBAY-US' => 0
        ];

        return $ids[$globalID] ?? 77;
    }

    /**
     * Send an http request to eBay
     *
     * @param string $requestBody - XML request
     * @return mixed
     */
    protected function sendHttpRequest(string $requestBody): mixed
    {
        $serverUrl = 'https://api.ebay.com/ws/api.dll';
//        $shoppingURL = 'http://open.api.ebay.com/shopping';
//        $findingURL  = 'http://svcs.ebay.com/services/search/FindingService/v1';

        if ($this->getAttribute('sandbox')) {
            $serverUrl = 'https://api.sandbox.ebay.com/ws/api.dll';
//            $shoppingURL = 'http://open.api.sandbox.ebay.com/shopping';
//            $findingURL  = 'http://svcs.sandbox.ebay.com/services/search/FindingService/v1';
        }

        $Curl = curl_init();
        curl_setopt($Curl, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($Curl, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($Curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($Curl, CURLOPT_POST, 1);
        curl_setopt($Curl, CURLOPT_URL, $serverUrl);
        curl_setopt($Curl, CURLOPT_HTTPHEADER, $this->buildHeaders());
        curl_setopt($Curl, CURLOPT_POSTFIELDS, $requestBody);
        $Response = curl_exec($Curl);

        curl_close($Curl);

        return $Response;
    }

    /**
     * Execute a GET request via cURL
     *
     * @param string $url
     *
     * @return string|false
     */
    protected function executeGETRequest(string $url): string | false
    {
        $Curl = curl_init($url);
        curl_setopt($Curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($Curl, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($Curl, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($Curl, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($Curl, CURLOPT_TIMEOUT, 10);
        $response = curl_exec($Curl);

        curl_close($Curl);

        return $response;
    }

    /**
     * Log a message via error_log
     * only if debug is true
     *
     * @param string $message - the messsage that would be log
     */
    public function debug(string $message): void
    {
        if ($this->getAttribute('debug')) {
            error_log("\n\n" . $message);
        }
    }

    /**
     * @return Pool
     */
    public static function getStash(): Pool
    {
        if (class_exists('\QUI\Cache\Manager')) {
            QUI\Cache\Manager::getStash();
            return QUI\Cache\Manager::$Stash;
        }

        if (Apc::isAvailable()) {
            $driver = new Apc();
            return new Pool($driver);
        }

        $driver = new FileSystem();
        return new Pool($driver);
    }
}
