# How to Create an MCP Tool in Your QUIQQER Module/Plugin

This guide explains how to provide your own Model Context Protocol (MCP) tool in a QUIQQER module or plugin, so that it
is accessible via the QUIQQER AI MCP Server and the MCP API.

## Providing an MCP Tool in Your Module

To make a functionality available to the MCP server, you need to implement a "provider" in your module or plugin. A
provider is a PHP class that registers one or more tools (functionalities) with the MCP server.

### Example: `GeneralMcpProvider`

Let's look at the `GeneralMcpProvider` as an example. This provider exposes general-purpose tools to the MCP server.

#### 1. Create a Provider Class

Your provider class should implement the necessary interface (`QUI\AI\MCP\ProviderInterface`) and
define the tools it provides. Each tool is usually represented by a class extending `AbstractTool` or by using
`CallbackTool` for simple callbacks.

**Example:**

```php
namespace QUI\AI\MCP\General;

use QUI\AI\MCP\Tool\AbstractTool;
use QUI\AI\MCP\Tool\CallbackTool;

class GeneralMcpProvider implements ProviderInterface
{
}
```

#### 2. Register Your Provider in `package.xml`

To make your provider discoverable, register it in your module's `package.xml`:

```xml

<ai-mcp>
    <provider>QUI\AI\MCP\General\GeneralMcpProvider</provider>
</ai-mcp>
```

Replace the class path with your own provider class.

#### 3. Create and Register Your Tools in the Provider

To make your tool available, your provider class must implement the `registerTools` method. In this method, you add your tool(s) to the provided `ToolRepository` using `$toolRepository->addTool(...)`.

**Example:**
```php
use QUI\AI\MCP\Tool\ToolRepository;

class GeneralMcpProvider implements ProviderInterface
{
    public function registerTools(ToolRepository $toolRepository): void
    {
        $toolRepository->addTool(new SomeCustomTool());
        $toolRepository->addTool(
            new CallbackTool([
                'name' => 'simpleTool',
                'description' => 'A simple callback tool',
                'parameters' => [
                    // parameter definitions
                ],
                'callback' => function ($params) {
                    // tool logic
                }
            ])
        );
    }
}
```

You can define your tools as separate methods or inline. Each tool must be added to the repository in `registerTools`.

#### 4. Define Tool Parameters

Each tool can define parameters that it accepts. Parameter types include:

- `string`
- `int`
- `float`
- `bool`
- `array`
- `object`
- `file`

Parameters are defined as an array with keys like `name`, `type`, `description`, and optionally `required` or `default`.

**Example:**

```php
'parameters' => [
    [
        'name' => 'username',
        'type' => 'string',
        'description' => 'The name of the user',
        'required' => true
    ],
    [
        'name' => 'age',
        'type' => 'int',
        'description' => 'The age of the user',
        'required' => false
    ]
]
```

#### 5. Creating a Tool Class

For more complex tools, create a class that extends `AbstractTool`. You must set the `name`, `description`, and `properties` attributes (usually in the constructor or via setters), and implement the `execute(array $params = []): mixed` method.

**Example:**
```php
use QUI\AI\MCP\Tool\AbstractTool;
use QUI\AI\MCP\Parameter\ParameterType;

class SomeCustomTool extends AbstractTool
{
    public string $name = 'someCustomTool';
    public string $description = 'Does something useful';
    public array $properties = [
        'project' => [
            'type' => ParameterType::STRING,
            'description' => 'The project name to create the page in',
            'required' => true
        ],
        'lang' => [
            'type' => ParameterType::STRING,
            'description' => 'The language of the project to create the page in',
            'required' => true
        ]
    ];

    public function execute(array $params = []): mixed
    {
        // Tool logic
    }
}
```

#### 6. Using `CallbackTool`

For simple tools, you can use `CallbackTool` and provide a closure for the logic.

```php
new CallbackTool([
    'name' => 'simpleTool',
    'description' => 'A simple callback tool',
    'parameters' => [
        // parameter definitions
    ],
    'callback' => function ($params) {
        // tool logic
    }
])
```

---

## Summary

1. **Implement a Provider**: Create a provider class that returns your tools.
2. **Register Provider**: Add your provider to `package.xml` under `<ai-mcp>`.
3. **Create and Register Tools**: Implement the `registerTools` method to add your tools to the `ToolRepository`.
4. **Define Parameters**: Specify parameter types and requirements for each tool.
5. **Implement Tools**: Use `AbstractTool` for complex tools, or `CallbackTool` for simple logic.

For more details about MCP and the protocol, see the [main documentation](README.md).
