<?php

namespace QUI\AI\MCP;

use Mcp\Server as MCPServer;
use Mcp\Server\Session\FileSessionStore;
use Mcp\Server\Transport\StreamableHttpTransport;
use Mcp\Server\Transport\TransportInterface;
use Mcp\Server\Transport\StdioTransport;
use Http\Discovery\Psr17Factory;
use Psr\Log\AbstractLogger;
use Psr\Log\LoggerInterface;
use QUI;
use QUI\Exception;
use QUI\ExceptionStack;
use Stringable;

class Server
{
    public static function run(): mixed
    {
        $mcpLogger = new class extends AbstractLogger implements LoggerInterface {
            public function log(mixed $level, string | Stringable $message, array $context = []): void
            {
                if ($level === 'info' || $level === 'debug') {
                    return;
                }

                // todo better logging
                error_log(
                    "[$level] $message " . print_r($context, true) . PHP_EOL,
                    3,
                    VAR_DIR . 'log/mcp.log'
                );
                //Log::write($message, (int)$level, $context, 'mcp');
            }
        };

        $server = MCPServer::builder();
        $server->setServerInfo('QUIQQER MCP', '1.0.0');
        //$server->setDiscovery(__DIR__);
        //$server->setContainer(container());

        // logging
        $server->setLogger($mcpLogger);

        // session
        $server->setSession(
            new FileSessionStore(
                VAR_DIR . '/sessions'
            )
        );

        // provider API
        $providerList = self::getProvider();

        foreach ($providerList as $provider) {
            $provider->register($server);
        }

        // API
        try {
            QUI::getEvents()->fireEvent('quiqqerMcpInit', [$server]);
        } catch (\Exception $e) {
            QUI\System\Log::addError($e->getMessage(), [
                'place' => 'mcp-server'
            ]);
        }

        // run the server
        $transport = function () use ($mcpLogger): TransportInterface | StreamableHttpTransport {
            if ('cli' === PHP_SAPI) {
                return new StdioTransport(STDIN, STDOUT, $mcpLogger);
            }

            $psr17 = new Psr17Factory();
            $request = $psr17->createServerRequestFromGlobals();

            return new StreamableHttpTransport(
                $request,
                $psr17,
                $psr17,
                [
                    'Access-Control-Allow-Origin' => '*',
                    // weitere Header
                ],
                $mcpLogger
            );
        };

        return $server->build()->run($transport());
    }

    /**
     * @throws Exception
     * @throws ExceptionStack
     * @throws QUI\Database\Exception
     */
    public static function getRequestUser(): QUI\Interfaces\Users\User
    {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? null;

        if (!$authHeader || stripos($authHeader, 'Bearer ') !== 0) {
            return new QUI\Users\Nobody();
        }

        $accessToken = trim(substr($authHeader, 7)); // "Bearer " entfernen
        $clientData = QUI\OAuth\Clients\Handler::getOAuthClientByAccessToken($accessToken);

        if (!$clientData) {
            return new QUI\Users\Nobody();
        }

        return QUI::getUsers()->get($clientData['user_id']);
    }

    /**
     * @return array<ProviderInterface>
     */
    public static function getProvider(): array
    {
        $cacheName = 'quiqqer/mcp/providerList';
        $packages = QUI::getPackageManager()->getInstalled();
        $result = [];

        try {
            $providerList = QUI\Cache\Manager::get($cacheName);
        } catch (QUI\Cache\Exception) {
            $providerList = [];

            foreach ($packages as $package) {
                try {
                    $Package = QUI::getPackage($package['name']);
                } catch (QUI\Exception $Exception) {
                    QUI\System\Log::writeException($Exception);
                    continue;
                }

                $provider = $Package->getProvider();
                $provider = array_filter($provider, function ($key) {
                    return $key === 'mcp';
                }, ARRAY_FILTER_USE_KEY);

                if (isset($provider['mcp'])) {
                    $providerList = array_merge($providerList, $provider['mcp']);
                }
            }

            QUI\Cache\Manager::set($cacheName, $providerList);
        }

        // initialize the instances
        foreach ($providerList as $provider) {
            if (!class_exists($provider)) {
                continue;
            }

            $instance = new $provider();

            if ($instance instanceof ProviderInterface) {
                $result[$instance::class] = $instance;
            }
        }

        return $result;
    }
}
