<?php

/**
 * This file contains QUI\LinkedIn\Auth
 */

namespace QUI\LinkedIn;

use QUI;
use QUI\Exception;
use QUI\Interfaces\Users\User;
use QUI\Users\AbstractAuthenticator;
use QUI\Locale;

/**
 * Class Auth
 *
 * Authentication handler for LinkedIn authentication
 *
 * @package QUI\LinkedIn\Auth
 */
class Auth extends AbstractAuthenticator
{
    protected QUI\Interfaces\Users\User $User;

    /**
     * @param array<string, mixed>|int|string|User|null $user
     */
    public function __construct(array | int | string | User | null $user = '')
    {
        $this->User = QUI::getUsers()->getNobody();

        if (!empty($user) && is_string($user)) {
            try {
                $this->User = QUI::getUsers()->getUserByName($user);
            } catch (\Exception) {
            }
        }
    }

    public function isSecondaryAuthentication(): bool
    {
        return false;
    }

    /**
     * @param array<string, mixed>|int|string $authParams
     * @throws Exception
     * @throws QUI\Exception
     */
    public function auth(array | int | string $authParams): void
    {
        if (!is_array($authParams) || !isset($authParams['token'])) {
            throw new QUI\Exception([
                'quiqqer/authlinkedin',
                'exception.auth.wrong.data'
            ], 401);
        }

        $token = $authParams['token'];
        $payload = LinkedIn::validateAccessToken($token);

        if (!LinkedIn::existsQuiqqerAccount($token, $payload)) {
            throw new Exception('LinkedIn user does not exist in QUIQQER', 401);
        }

        $linkedInSub = $payload['sub'] ?? null;
        $Users = QUI::getUsers();

        if (empty($linkedInSub)) {
            throw new Exception('LinkedIn user does not exist in QUIQQER', 401);
        }

        $connectionProfile = LinkedIn::getConnectedAccountByToken($token, $payload);

        if (empty($connectionProfile['userId'])) {
            throw new Exception('LinkedIn user does not exist in QUIQQER', 401);
        }

        try {
            $User = $Users->get($connectionProfile['userId']);
            $this->User = $User;
        } catch (QUI\Exception) {
            throw new Exception('LinkedIn user does not exist in QUIQQER', 401);
        }
    }

    public function getUser(): User
    {
        return $this->User;
    }

    public function getTitle(?Locale $Locale = null): string
    {
        if (is_null($Locale)) {
            $Locale = QUI::getLocale();
        }

        return $Locale->get('quiqqer/authlinkedin', 'authlinkedin.title');
    }

    public function getDescription(?Locale $Locale = null): string
    {
        if (is_null($Locale)) {
            $Locale = QUI::getLocale();
        }

        return $Locale->get('quiqqer/authlinkedin', 'authlinkedin.description');
    }

    public static function getLoginControl(): ?QUI\Control
    {
        return new QUI\LinkedIn\Controls\Button();
    }

    public function getIcon(): string
    {
        return 'fa fa-brands fa-linkedin';
    }
}
