/**
 * Main CAPTCHA control that displays the current default CAPTCHA
 *
 * @module package/quiqqer/captcha/bin/controls/CaptchaDisplay
 * @author www.pcsg.de (Patrick Müller)
 */
define('package/quiqqer/captcha/bin/controls/CaptchaDisplay', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/loader/Loader',
    'utils/Controls',

    'package/quiqqer/captcha/bin/controls/Captcha',

    'Ajax'

], function (QUI, QUIControl, QUILoader, QUIControlUtils, BaseCaptchaControl, QUIAjax) {
    "use strict";

    return new Class({

        Extends: QUIControl,
        Type   : 'package/quiqqer/captcha/bin/controls/CaptchaDisplay',

        Binds: [
            '$onInject',
            '$getCurrentCaptchaControl'
        ],

        options: {
            module_has_js_control: false // flag that indicates if the used captcha module has its own js control
        },

        initialize: function (options) {
            this.parent(options);

            this.Loader          = new QUILoader();
            this.$CaptchaControl = null;
            this.$ResponseInput  = null;
            this.$loaded = false;

            this.addEvents({
                onInject: this.$onInject,
                onImport: this.$onImport
            });
        },

        /**
         * event: onInject
         */
        $onInject: function () {
            this.Loader.inject(this.$Elm);
            this.Loader.show();

            this.$getCurrentCaptchaControl().then((captchaControlHtml) => {
                this.$Elm.set('html', captchaControlHtml);

                if (this.$Elm.get('data-quiid')) {
                    this.setAttribute('module_has_js_control', true);
                }

                QUI.parse(this.$Elm).then(() => {
                    this.Loader.hide();
                    this.$loaded = true;
                });
            });
        },

        /**
         * event: onImport
         */
        $onImport: function () {
            const CaptchaResponseInput = this.$Elm.getElement('input[name="quiqqer-captcha-response"]');

            if (!CaptchaResponseInput) {
                return;
            }

            this.$ResponseInput = CaptchaResponseInput;

            this.$loaded = true;
            this.Loader.show();

            this.getCaptchaControl().then((CaptchaControl) => {
                this.Loader.show();

                if (this.getAttribute('module_has_js_control')) {
                    CaptchaControl.addEvents({
                        onSuccess: function (response) {
                            CaptchaResponseInput.value = response;
                        },
                        onExpired: function () {
                            CaptchaResponseInput.value = '';
                        }
                    });

                    return;
                }

                // Fallback: Use default captcha module if current module has no own js control
                CaptchaResponseInput.addEvent('keyup', () => {
                    if (CaptchaResponseInput.value.trim() !== '') {
                        CaptchaControl.$onCaptchaSuccess(CaptchaResponseInput.value);
                    }
                });
            });
        },

        /**
         * Get input for CAPTCHA response
         *
         * @return {null|HTMLInputElement}
         */
        $getCaptchaResponseInput: function () {
            return this.$ResponseInput;
        },

        /**
         * Get Captcha control
         *
         * @return {Promise}
         */
        getCaptchaControl: async function () {
            await this.$waitForLoaded();

            if (this.$CaptchaControl) {
                return Promise.resolve(this.$CaptchaControl);
            }

            if (this.getAttribute('module_has_js_control')) {
                return QUIControlUtils.getControlByElement(
                    this.$Elm.getElement('.quiqqer-captcha-control')
                ).then((CaptchaControl) => {
                    this.$CaptchaControl = CaptchaControl;
                    return CaptchaControl;
                });
            }

            this.$CaptchaControl = new BaseCaptchaControl();
            return Promise.resolve(this.$CaptchaControl);
        },

        /**
         * Wait for this control to be loaded
         * @return {Promise<void>}
         */
        $waitForLoaded: async function() {
            if (this.$loaded) {
                return;
            }

            return new Promise((resolve) => {
                const waitForLoaded = setInterval(() => {
                    if (!this.$loaded) {
                        return;
                    }

                    clearInterval(waitForLoaded);
                    resolve();
                }, 200);
            });
        },

        /**
         * Set settings to input
         */
        $getCurrentCaptchaControl: function () {
            return new Promise(function (resolve, reject) {
                QUIAjax.get(
                    'package_quiqqer_captcha_ajax_getCurrentCaptchaControl', resolve, {
                        'package': 'quiqqer/captcha',
                        onError  : reject
                    }
                );
            });
        }
    });
});
