<?php

/**
 * This file contains QUI\Composer
 */

namespace QUI\Composer;

use QUI;
use Symfony\Component\Console\Output\OutputInterface;

use function rtrim;

/**
 * Class Composer
 */
class Composer implements QUI\Composer\Interfaces\ComposerInterface
{
    /**
     * @var int
     */
    const MODE_CLI = 0;

    /**
     * @var int
     */
    const MODE_WEB = 1;

    /**
     * @var QUI\Composer\Interfaces\ComposerInterface
     */
    protected Interfaces\ComposerInterface $Runner;

    /**
     * @var int
     */
    protected int $mode;

    /**
     * @var bool
     */
    protected bool $mute = false;

    /**
     * @var string
     */
    protected string $workingDir;

    /**
     * @var string
     */
    protected string $composerDir;

    /**
     * @var Utils\Events
     */
    protected Utils\Events $Events;

    /**
     * Composer constructor.
     *
     * Can be used as a general access point to composer.
     * Will use CLI composer if shell_exec is available
     *
     * @param string $workingDir
     * @throws Exception
     */
    public function __construct(string $workingDir)
    {
        $this->workingDir = rtrim($workingDir, '/') . '/';
        $this->composerDir = $this->workingDir;
        $this->Events = new QUI\Composer\Utils\Events();

        if (QUI\Utils\System::isShellFunctionEnabled('shell_exec')) {
            $this->setMode(self::MODE_CLI);
        } else {
            $this->setMode(self::MODE_WEB);
        }
    }

    /**
     * Sets the output interface.
     *
     * @param OutputInterface $Output The output interface to be set.
     * @return void
     */
    public function setOutput(OutputInterface $Output): void
    {
        $this->Runner->setOutput($Output);
    }

    /**
     * Executes a Composer command with given options and returns the result as an array.
     *
     * @param string $command The Composer command to execute.
     * @param array<string, mixed> $options (Optional) The options to be passed to the Composer command.
     * @return array<int, string> The result of executing the Composer command as an array.
     */
    public function executeComposer(string $command, array $options = []): array
    {
        return $this->Runner->executeComposer($command, $options);
    }

    /**
     * Set the composer mode
     * CLI, or web
     *
     * @param int $mode - self::MODE_CLI, self::MODE_WEB
     * @throws Exception
     */
    public function setMode(int $mode): void
    {
        $self = $this;

        switch ($mode) {
            case self::MODE_CLI:
                $this->Runner = new CLI($this->workingDir);
                $this->mode = self::MODE_CLI;
                break;

            case self::MODE_WEB:
                $this->Runner = new Web($this->workingDir);
                $this->mode = self::MODE_WEB;
                break;
        }

        if (method_exists($this->Runner, 'addEvent')) {
            $this->Runner->addEvent('onOutput', function ($Runner, $output, $type) use ($self) {
                $self->Events->fireEvent('output', [$self, $output, $type]);
            });
        }

        if ($this->mute) {
            $this->Runner->mute();
        } else {
            $this->Runner->unmute();
        }
    }

    /**
     * Return the internal composer runner
     *
     * @return Interfaces\ComposerInterface
     */
    public function getRunner(): Interfaces\ComposerInterface
    {
        return $this->Runner;
    }

    /**
     * @return array<string, string>
     */
    public function getVersions(): array
    {
        if (method_exists($this->Runner, 'getVersions')) {
            return $this->Runner->getVersions();
        }

        return [];
    }

    /**
     * @return string
     */
    public function getWorkingDir(): string
    {
        return $this->workingDir;
    }

    /**
     * Executes composers update command
     *
     * @param array<string, mixed> $options
     * @return array<int, string>
     */
    public function update(array $options = []): array
    {
        return $this->Runner->update($options);
    }

    /**
     * Executes composers install command
     *
     * @param array<string, mixed> $options
     * @return array<int, string>
     */
    public function install(array $options = []): array
    {
        return $this->Runner->install($options);
    }

    /**
     * Executes composers require command
     *
     * @param array<string>|string $package - Name of the package: i.E. 'quiqqer/core' or list of packages
     * @param string $version
     * @param array<string, mixed> $options
     * @return array<int, string>
     */
    public function requirePackage(array | string $package, string $version = "", array $options = []): array
    {
        return $this->Runner->requirePackage($package, $version, $options);
    }

    /**
     * Executes composers' outdated command
     *
     * @param bool $direct
     * @param array<string, mixed> $options
     * @return array<int, array{package: string, version: string}>
     */
    public function outdated(bool $direct = false, array $options = []): array
    {
        return $this->Runner->outdated($direct, $options);
    }

    /**
     * Checks whether updates are available
     *
     * @param bool $direct
     * @return bool - true if updates are available
     */
    public function updatesAvailable(bool $direct): bool
    {
        return $this->Runner->updatesAvailable($direct);
    }

    /**
     * Return the packages which could be updated
     *
     * @return array<int, array{package: string, version: string, oldVersion: string}>
     * @throws Exception
     */
    public function getOutdatedPackages(): array
    {
        return $this->Runner->getOutdatedPackages();
    }

    /**
     * Returns the current mode (Web or CLI) as int
     *
     * @return int
     */
    public function getMode(): int
    {
        return $this->mode;
    }

    /**
     * Generates the autoloader files again without downloading anything
     *
     * @param array<string, mixed> $options
     *
     * @return bool - true on success
     */
    public function dumpAutoload(array $options = []): bool
    {
        return $this->Runner->dumpAutoload($options);
    }

    /**
     * Searches the repositories for the given needle
     *
     * @param string $needle
     * @param array<string, mixed> $options
     * @return array<string, string> - Returns an array in the format: array( packagename => description)
     */
    public function search(string $needle, array $options = []): array
    {
        return $this->Runner->search($needle, $options);
    }

    /**
     * Lists all installed packages
     *
     * @param string $package
     * @param array<string, mixed> $options
     * @return array<int, string> - returns an array with all installed packages
     */
    public function show(string $package = "", array $options = []): array
    {
        return $this->Runner->show($package, $options);
    }

    /**
     * Clears the composer cache
     *
     * @return bool - true on success; false on failure
     */
    public function clearCache(): bool
    {
        return $this->Runner->clearCache();
    }

    /**
     * Mute the execution
     */
    public function mute(): void
    {
        $this->mute = true;
        $this->Runner->mute();
    }

    /**
     * Unmute the execution
     */
    public function unmute(): void
    {
        $this->mute = false;
        $this->Runner->unmute();
    }

    /**
     * Executes the composer why command.
     * This commands displays why the package has been installed and which packages require it.
     *
     * @param string $package
     *
     * @return array<int, array{package: string, version: string, constraint: string}>
     */
    public function why(string $package): array
    {
        return $this->Runner->why($package);
    }

    //region events

    /**
     * Add an event
     *
     * @param string $event - The type of event (e.g. 'output').
     * @param callable $fn - The function to execute.
     * @param int $priority - optional, Priority of the event
     */
    public function addEvent(string $event, callable $fn, int $priority = 0): void
    {
        $this->Events->addEvent($event, $fn, $priority);
    }

    //endregion
}
