/**
 * Manage Invite Codes
 */
define('package/quiqqer/coupons/bin/backend/controls/Manager', [

    'qui/QUI',
    'qui/controls/desktop/Panel',
    'qui/controls/loader/Loader',
    'qui/controls/windows/Popup',
    'qui/controls/windows/Confirm',
    'qui/controls/buttons/Button',
    'qui/controls/buttons/Separator',

    'controls/grid/Grid',
    'qui/utils/Form',

    'package/quiqqer/coupons/bin/backend/CouponCodes',

    'Locale',
    'Mustache',

    'text!package/quiqqer/coupons/bin/backend/controls/Manager.html',
    'text!package/quiqqer/coupons/bin/backend/controls/Manager.Create.html',
    'text!package/quiqqer/coupons/bin/backend/controls/Manager.Usages.html',
    'css!package/quiqqer/coupons/bin/backend/controls/Manager.css'

], function (QUI, QUIPanel, QUILoader, QUIPopup, QUIConfirm, QUIButton, QUISeparator,
             Grid, QUIFormUtils, CouponCodes, QUILocale, Mustache, template, templateCreate,
             templateUsages) {
    "use strict";

    const lg = 'quiqqer/coupons';

    return new Class({

        Extends: QUIPanel,
        Type: 'package/quiqqer/coupons/bin/backend/controls/Manager',

        Binds: [
            '$onCreate',
            '$onResize',
            '$listRefresh',
            '$onRefresh',
            '$load',
            '$setGridData',
            '$showDetails',
            '$toggleActiveStatus',
            '$managePackages',
            '$delete',
            '$editBundle',
            'refresh',
            '$openUserPanel',
            '$sendMail'
        ],

        options: {
            title: QUILocale.get(lg, 'controls.manager.title')
        },

        initialize: function (options) {
            this.parent(options);

            this.Loader = new QUILoader();
            this.$User = null;
            this.$Grid = null;
            this.$GridParent = null;
            this.$Panel = null;

            this.addEvents({
                onCreate: this.$onCreate,
                onRefresh: this.$onRefresh,
                onResize: this.$onResize
            });
        },

        /**
         * Event: onCreate
         */
        $onCreate: function () {
            const self = this;

            this.Loader.inject(this.$Elm);

            this.addButton({
                name: 'create',
                text: QUILocale.get(lg, 'controls.manager.tbl.btn.create'),
                textimage: 'fa fa-plus',
                events: {
                    onClick: function () {
                        self.$showDetails();
                    }
                }
            });

            this.addButton(new QUISeparator());

            this.addButton({
                name: 'edit',
                text: QUILocale.get(lg, 'controls.manager.tbl.btn.edit'),
                textimage: 'fa fa-edit',
                events: {
                    onClick: function () {
                        self.$showDetails(Object.clone(self.$Grid.getSelectedData()[0]));
                    }
                }
            });

            this.addButton({
                name: 'delete',
                text: QUILocale.get(lg, 'controls.manager.tbl.btn.delete'),
                textimage: 'fa fa-trash',
                events: {
                    onClick: this.$delete
                }
            });

            this.$load();
        },

        /**
         * Refresh data
         */
        refresh: function () {
            if (this.$Grid) {
                this.$Grid.refresh();
            }
        },

        /**
         * event: onResize
         */
        $onResize: function () {
            if (this.$GridParent && this.$Grid) {
                const size = this.$GridParent.getSize();

                this.$Grid.setHeight(size.y);
                this.$Grid.resize();
            }
        },

        /**
         * Load Grid
         */
        $load: function () {
            const self = this;

            this.setContent(Mustache.render(template));
            const Content = this.getContent();

            this.$GridParent = Content.getElement(
                '.quiqqer-coupons-manager-table'
            );

            this.$Grid = new Grid(this.$GridParent, {
                columnModel: [
                    {
                        header: QUILocale.get('quiqqer/system', 'id'),
                        dataIndex: 'id',
                        dataType: 'number',
                        width: 50
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.code'),
                        dataIndex: 'code',
                        dataType: 'string',
                        width: 150
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.title'),
                        dataIndex: 'title',
                        dataType: 'string',
                        width: 200
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.status'),
                        dataIndex: 'status',
                        dataType: 'node',
                        width: 200,
                        className: 'clickable'
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.validUntilDate'),
                        dataIndex: 'validUntilDateText',
                        dataType: 'string',
                        width: 150
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.reusable'),
                        dataIndex: 'maxUsageLabel',
                        dataType: 'string',
                        width: 150
                    },
                    {
                        header: QUILocale.get(lg, 'controls.manager.tbl.header.createDate'),
                        dataIndex: 'createDate',
                        dataType: 'string',
                        width: 150
                    }
                ],
                pagination: true,
                serverSort: true,
                selectable: true,
                multipleSelection: true
            });

            this.$Grid.addEvents({
                onDblClick: function () {
                    self.$showDetails(Object.clone(self.$Grid.getSelectedData()[0]));
                },
                onClick: function (event) {
                    const selected = self.$Grid.getSelectedData();

                    self.getButtons('delete').enable();

                    if (selected.length === 1) {
                        self.getButtons('edit').enable();
                    } else {
                        self.getButtons('edit').disable();
                    }

                    if (!event.cell.hasClass('clickable')) {
                        return;
                    }

                    const Row = selected[0];
                    self.$showUsages(Row);
                },
                onRefresh: this.$listRefresh
            });

            this.resize();
            this.$Grid.refresh();
        },

        /**
         * Event: onRefresh
         */
        $onRefresh: function () {
            if (this.$Grid) {
                this.$Grid.refresh();
            }
        },

        /**
         * Refresh bundle list
         *
         * @param {Object} Grid
         */
        $listRefresh: function (Grid) {
            if (!this.$Grid) {
                return;
            }

            const self = this;

            self.getButtons('delete').disable();
            self.getButtons('edit').disable();

            const GridParams = {
                sortOn: Grid.getAttribute('sortOn'),
                sortBy: Grid.getAttribute('sortBy'),
                perPage: Grid.getAttribute('perPage'),
                page: Grid.getAttribute('page')
            };

            switch (GridParams.sortOn) {
                case 'status':
                    GridParams.sortOn = 'useDate';
                    break;

                case 'user':
                    GridParams.sortOn = 'userId';
                    break;
            }

            this.Loader.show();

            CouponCodes.getList(GridParams).then(function (ResultData) {
                self.Loader.hide();
                self.$setGridData(ResultData);
            });
        },

        /**
         * Set license data to grid
         *
         * @param {Object} GridData
         */
        $setGridData: function (GridData) {
            const textUnlimited = QUILocale.get(lg, 'controls.manager.tbl.validUntil.unlimited');

            for (let i = 0, len = GridData.data.length; i < len; i++) {
                const Row = GridData.data[i];

                const StatusElm = new Element('span', {
                    'class': 'quiqqer-coupons-manager-tbl-status'
                });

                const usageCount = Row.usages.length;

                if (!Row.isValid) {
                    StatusElm.set('html', QUILocale.get(lg, 'controls.manager.tbl.status.invalid', {
                        usageCount: usageCount
                    }));

                    StatusElm.addClass('quiqqer-coupons-manager-tbl-status-invalid');
                } else {
                    StatusElm.set('html', QUILocale.get(lg, 'controls.manager.tbl.status.valid', {
                        usageCount: usageCount
                    }));

                    StatusElm.addClass('quiqqer-coupons-manager-tbl-status-used');
                }

                Row.status = StatusElm;

                if (!Row.validUntilDate) {
                    Row.validUntilDateText = textUnlimited;
                } else {
                    Row.validUntilDateText = Row.validUntilDate;
                }

                if (!Row.title) {
                    Row.title = '-';
                }

                Row.maxUsageLabel = QUILocale.get(lg, 'controls.manager.tbl.maxUsageLabel.' + Row.maxUsages);
            }

            this.$Grid.setData(GridData);
        },

        /**
         * Create new CouponCode or edit an existing one
         *
         * @param {Object} [CouponData] - If omitted create new CouponCode
         */
        $showDetails: function (CouponData) {
            let Form;
            let DiscountSelect;

            CouponData = CouponData || false;

            const FuncSubmit = () => {
                if (!Form.reportValidity()) {
                    return;
                }

                Popup.Loader.show();

                if (CouponData) {
                    CouponCodes.edit(CouponData.id, QUIFormUtils.getFormData(Form)).then((couponCodeId) => {
                        if (!couponCodeId) {
                            Popup.Loader.hide();
                            return;
                        }

                        this.refresh();
                        Popup.close();
                    });

                    return;
                }

                CouponCodes.create(QUIFormUtils.getFormData(Form)).then((couponCodeId) => {
                    if (!couponCodeId) {
                        Popup.Loader.hide();
                        return;
                    }

                    require([
                        'package/quiqqer/translator/bin/classes/Translator'
                    ], (Translator) => {
                        new Translator().publish(lg).then(() => {
                            this.refresh();
                            Popup.close();
                        }).then(() => {
                            this.refresh();
                            Popup.close();
                        });
                    });
                }).catch(() => {
                    Popup.Loader.hide();
                });
            };

            const openDiscount = () => {
                Popup.Loader.show();

                require([
                    'package/quiqqer/discount/bin/controls/Discounts',
                    'utils/Panels'
                ], function (DiscountsManagerPanel, PanelUtils) {
                    const Panel = new DiscountsManagerPanel();

                    PanelUtils.openPanelInTasks(Panel).then((PanelOpened) => {
                        PanelOpened.editChild(parseInt(DiscountSelect.getValue()));
                        Popup.close();
                    });
                });
            };

            // open popup
            const lgPrefix = 'controls.manager.create.template.';

            const Popup = new QUIPopup({
                icon: 'fa fa-plus',
                title: CouponData ?
                    QUILocale.get(lg, 'controls.manager.details.popup.title_edit', {code: CouponData.code}) :
                    QUILocale.get(lg, 'controls.manager.details.popup.title_new'),
                maxHeight: 1000,
                maxWidth: 1000,
                events: {
                    onOpen: (Win) => {
                        const Content = Popup.getContent();
                        Form = Content.getElement('form');

                        Form.addEvent('submit', function (event) {
                            event.stop();
                            FuncSubmit();
                        });

                        const Amount = Content.getElement('input[name="amount"]');
                        const Code = Content.getElement('input[name="code"]');

                        Code.addEvent('keyup', function () {
                            if (Code.value !== '') {
                                Amount.disabled = true;
                                Amount.value = 1;
                            } else {
                                Amount.disabled = false;
                            }
                        });

                        let EditDiscountBtn = false;

                        if (CouponData) {
                            CouponData.discountIds = CouponData.discountIds.join(',');
                            QUIFormUtils.setDataToForm(CouponData, Form);

                            EditDiscountBtn = new QUIButton({
                                'class': 'optional',
                                textimage: 'fa fa-percent',
                                text: QUILocale.get(lg, 'controls.manager.details.popup.btn.open_discount'),
                                title: QUILocale.get(lg, 'controls.manager.details.popup.btn.open_discount'),
                                events: {
                                    onClick: openDiscount
                                },
                                styles: {
                                    float: 'right'
                                }
                            }).inject(
                                Content.getElement('.quiqqer-coupons-manager-create-advanced-discount-buttons')
                            );
                        }

                        Win.Loader.show();

                        QUI.parse(Content).then(() => {
                            Win.Loader.hide();

                            if (!CouponData) {
                                return;
                            }

                            DiscountSelect = QUI.Controls.getById(
                                Content.getElement('input[name="discountIds"]').get('data-quiid')
                            );

                            DiscountSelect.addEvents({
                                onAddItem: () => {
                                    EditDiscountBtn.enable();
                                },
                                onRemoveItem: () => {
                                    EditDiscountBtn.disable();
                                }
                            });
                        });
                    }
                },
                closeButton: true,
                content: Mustache.render(templateCreate, {
                    labelTitle: QUILocale.get(lg, lgPrefix + 'labelTitle'),
                    labelCode: QUILocale.get(lg, lgPrefix + 'labelCode'),
                    labelUsers: QUILocale.get(lg, lgPrefix + 'labelUsers'),
                    labelGroups: QUILocale.get(lg, lgPrefix + 'labelGroups'),
                    labelDate: QUILocale.get(lg, lgPrefix + 'labelDate'),
                    labelAmount: QUILocale.get(lg, lgPrefix + 'labelAmount'),
                    labelReusable: QUILocale.get(lg, lgPrefix + 'labelReusable'),
                    labelReusableOncePerUser: QUILocale.get(lg, lgPrefix + 'labelReusableOncePerUser'),
                    labelReusableOnce: QUILocale.get(lg, lgPrefix + 'labelReusableOnce'),
                    labelReusableUnlimited: QUILocale.get(lg, lgPrefix + 'labelReusableUnlimited'),
                    labelDiscount: QUILocale.get(lg, lgPrefix + 'labelDiscount'),
                    headerBasics: QUILocale.get(lg, lgPrefix + 'headerBasics'),
                    headerDiscount: QUILocale.get(lg, lgPrefix + 'headerDiscount'),
                    headerRestrictions: QUILocale.get(lg, lgPrefix + 'headerRestrictions'),
                    descCode: QUILocale.get(lg, lgPrefix + 'descCode'),
                    descTitle: QUILocale.get(lg, lgPrefix + 'descTitle'),
                    labelDiscountAmount: QUILocale.get(lg, lgPrefix + 'labelDiscountAmount'),
                    labelDiscountType: QUILocale.get(lg, lgPrefix + 'labelDiscountType'),
                    labelDiscountTypeFlat: QUILocale.get(lg, lgPrefix + 'labelDiscountTypeFlat'),
                    labelDiscountTypePercentage: QUILocale.get(lg, lgPrefix + 'labelDiscountTypePercentage'),

                    isCreate: !CouponData
                })
            });

            Popup.open();

            Popup.addButton(new QUIButton({
                name: 'submit',
                text: CouponData ?
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_text_edit') :
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_text_new'),
                alt: CouponData ?
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_edit') :
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_new'),
                title: CouponData ?
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_edit') :
                    QUILocale.get(lg, 'controls.manager.details.popup.btn.confirm_new'),
                events: {
                    onClick: FuncSubmit
                }
            }));
        },

        /**
         * Remove all selected licenses
         */
        $delete: function () {
            const self = this;
            const deleteData = [];
            const deleteIds = [];
            const rows = this.$Grid.getSelectedData();

            for (const i = 0, len = rows.length; i < len; i++) {
                deleteData.push(
                    rows[i].title + ' (ID: #' + rows[i].id + ')'
                );

                deleteIds.push(rows[i].id);
            }

            // open popup
            const Popup = new QUIConfirm({
                'maxHeight': 300,
                'autoclose': false,

                'information': QUILocale.get(
                    lg,
                    'controls.manager.delete.popup.info', {
                        codes: deleteData.join('<br/>')
                    }
                ),
                'title': QUILocale.get(lg, 'controls.manager.delete.popup.title'),
                'texticon': 'fa fa-trash',
                text: QUILocale.get(lg, 'controls.manager.delete.popup.title'),
                'icon': 'fa fa-trash',

                cancel_button: {
                    text: false,
                    textimage: 'icon-remove fa fa-remove'
                },
                ok_button: {
                    text: false,
                    textimage: 'icon-ok fa fa-check'
                },
                events: {
                    onSubmit: function () {
                        Popup.Loader.show();

                        CouponCodes.delete(deleteIds).then(function (success) {
                            if (!success) {
                                Popup.Loader.hide();
                                return;
                            }

                            Popup.close();
                            self.refresh();
                        });
                    }
                }
            });

            Popup.open();
        },

        /**
         * Open user panel
         *
         * @param {Object} RowData
         */
        $showUsages: function (RowData) {
            new QUIConfirm({
                maxHeight: 550,
                maxWidth: 600,
                autoclose: false,

                title: QUILocale.get(lg, 'controls.Manager.usages.title', {code: RowData.code}),
                icon: 'fa fa-user',

                cancel_button: false,
                ok_button: {
                    text: QUILocale.get(lg, 'controls.Manager.usages.btn_ok'),
                    textimage: 'icon-ok fa fa-check'
                },
                events: {
                    onSubmit: function (Popup) {
                        Popup.close();
                    },
                    onOpen: function (Popup) {
                        const lgPrefix = 'controls.Manager.usages.template.';

                        Popup.setContent(Mustache.render(templateUsages, {
                            headerUser: QUILocale.get(lg, lgPrefix + 'headerUser'),
                            headerDate: QUILocale.get(lg, lgPrefix + 'headerDate'),
                            headerOrderPrefixedId: QUILocale.get(lg, lgPrefix + 'headerOrderPrefixedId')
                        }));

                        const Content = Popup.getContent();
                        const TableBody = Content.getElement('tbody');

                        for (const i = 0, len = RowData.usages.length; i < len; i++) {
                            const usage = RowData.usages[i];

                            const Row = new Element('tr', {
                                html: '<td>' + usage.date + '</td>' +
                                    '<td>' + usage.userId + ' (' + usage.userName + ')</td>'
                            }).inject(TableBody);

                            new Element('td', {
                                html: usage.orderPrefixedId ? usage.orderPrefixedId : '-'
                            }).inject(Row);
                        }
                    }
                }
            }).open();
        }
    });
});
