/**
 * CouponCodeInput
 */
define('package/quiqqer/coupons/bin/frontend/controls/CouponCodeInput', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/loader/Loader',
    'utils/Session',

    'package/quiqqer/coupons/bin/frontend/CouponCodes',

    'Ajax',
    'Locale',
    'Mustache',

    'text!package/quiqqer/coupons/bin/frontend/controls/CouponCodeInput.html',
    'css!package/quiqqer/coupons/bin/frontend/controls/CouponCodeInput.css'

], function (QUI, QUIControl, QUILoader, Sessions, CouponCodes, QUIAjax, QUILocale, Mustache, template) {
    'use strict';

    const lg = 'quiqqer/coupons';

    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/coupons/bin/frontend/controls/CouponCodeInput',

        Binds: [
            '$submit',
            'handleError'
        ],

        initialize: function (options) {
            this.parent(options);

            this.$Input = null;
            this.$ErrorMsgContainer = null;
            this.Loader = new QUILoader();
            this.$running = false;


            this.addEvents({
                onInject: this.$onInject
            });
        },

        /**
         * Event: onInject
         */
        $onInject: function () {
            const self = this;
            const lgPrefix = 'controls.frontend.CouponCodeInput.template.';

            this.$Elm.addClass('quiqqer-coupons-field');

            if (this.$Elm.getParent('[data-name="quiqqer-simple-checkout"]')) {
                this.$Elm.addClass('quiqqer-coupons-field-simpleCheckout');
            }

            this.$Elm.set('html', Mustache.render(template, {
                labelInput: QUILocale.get(lg, lgPrefix + 'labelInput'),
                labelInputPlaceholder: QUILocale.get(lg, lgPrefix + 'labelInputPlaceholder'),
                submitBtnText: QUILocale.get(lg, lgPrefix + 'submitBtnText')
            }));

            this.$ErrorMsgContainer = this.$Elm.querySelector('[data-ref="errorMsg"]');

            this.Loader.inject(this.$Elm);

            this.$Input = this.$Elm.getElement('[data-ref="code-input"]');

            let timeoutId = null;
            let previousValue = '';
            let currentValue = '';

            const keyUpFunc = function (event) {
                currentValue = self.$Input.value.trim();

                if (event.code === 13) {
                    clearTimeout(timeoutId);
                    previousValue = currentValue;
                    self.$submit();
                } else {
                    clearTimeout(timeoutId);

                    timeoutId = setTimeout(function () {
                        if (
                            (currentValue !== previousValue || currentValue === '') &&
                            self.$ErrorMsgContainer.getAttribute('data-show')
                        ) {
                            self.hideErrorMessage();
                            self.resetInvalidInput();
                        }
                        previousValue = currentValue;
                    }, 400);
                }
            };

            this.$Input.addEvent('keyup', keyUpFunc);

            this.$Elm.getElement('[data-ref="submit-coupon-code"]').addEvent('click', function (event) {
                event.stop();
                self.$submit();
            });

            if (this.isInOrder()) {
                const OrderProcess = this.getOrderProcess();

                new Promise(function (resolve) {
                    if (!OrderProcess.isLoaded()) {
                        return OrderProcess.addEvent('onLoad', resolve);
                    }

                    resolve();
                }).then(function () {
                    if (OrderProcess.getAttribute('basketEditable') === false) {
                        // remove coupon codes button
                        new Element('button', {
                            'class': 'quiqqer-coupons-remove',
                            'html': '<span class="fa fa-close"></span>',
                            title: QUILocale.get(lg, 'remove.coupons.button'),
                            events: {
                                click: function (e) {
                                    e.stop();
                                    self.removeCouponsFromOrder().catch(function (e) {
                                        console.error(e);
                                    });
                                }
                            }
                        }).inject(self.$Elm.getElement('.quiqqer-coupons-couponcodeinput'));
                    }
                });
            }
        },

        /**
         * Submit a CouponCode
         */
        $submit: function () {
            if (this.$running) {
                return;
            }

            const self = this;
            const code = this.$Input.value.trim();

            if (code === '') {
                this.$Input.focus();
                return;
            }

            this.$running = true;
            this.Loader.show();

            // simple order
            const SimpleCheckoutNode = this.getElm().getParent('[data-name="quiqqer-simple-checkout"]');

            if (SimpleCheckoutNode) {
                const SimpleCheckout = QUI.Controls.getById(SimpleCheckoutNode.get('data-quiid'));
                let orderHash = window.location.hash.substring(1);

                if (!orderHash || orderHash === '') {
                    orderHash = SimpleCheckout.getAttribute('orderHash');
                }

                QUIAjax.get('package_quiqqer_order-simple-checkout_ajax_frontend_getOrder', (response) => {
                    CouponCodes.addCouponCodeToBasket(code, response.order.uuid).then(() => {
                        if (SimpleCheckout) {
                            const refreshPayment = () =>
                                SimpleCheckout.$Payment
                                    ? SimpleCheckout.$Payment.refresh()
                                    : Promise.resolve();

                            refreshPayment().then(() => {
                                return SimpleCheckout.$refreshBasket()
                            }).then(() => {
                                this.$running = false;
                                this.Loader.hide();
                            });

                            return;
                        }

                        this.$running = false;
                        this.Loader.hide();
                    }).catch(self.handleError);
                }, {
                    'package': 'quiqqer/order-simple-checkout',
                    orderHash: orderHash
                });

                return;
            }

            // get order process
            if (!this.isInOrder()) {
                QUIAjax.get('package_quiqqer_order_ajax_frontend_basket_getLastOrder', function (order) {
                    CouponCodes.addCouponCodeToBasket(code, order.hash).then(function (redeemed) {
                        if (!redeemed) {
                            self.$running = false;
                            self.Loader.hide();
                            return;
                        }

                        self.$addCouponCodeToSession(code).then(function () {
                            QUIAjax.get('package_quiqqer_order_ajax_frontend_basket_getOrderProcessUrl', function (url) {
                                window.location = url + '?coupon=' + code;
                            }, {
                                'package': 'quiqqer/order'
                            });
                        });
                    }).catch(self.handleError);
                }, {
                    'package': 'quiqqer/order'
                });

                return;
            }

            const OrderProcess = this.getOrderProcess();

            OrderProcess.Loader.show();

            OrderProcess.getOrder().then(function (orderHash) {
                CouponCodes.addCouponCodeToBasket(code, orderHash).then(() => {
                    self.$running = false;

                    self.$addCouponCodeToSession(code).then(function () {
                        OrderProcess.reload();
                    });
                }).catch((err) => {
                    OrderProcess.Loader.hide();
                    self.handleError(err);
                });
            });
        },

        /**
         *
         * @param code
         * @return {Promise}
         */
        $addCouponCodeToSession: function (code) {
            return Sessions.get('quiqqer-coupons').then(function (coupons) {
                if (!coupons || typeOf(coupons) !== 'array') {
                    coupons = [];
                }

                coupons.push(code);

                coupons = coupons.filter(function (value, index, self) {
                    return self.indexOf(value) === index;
                });

                return Sessions.set('quiqqer-coupons', JSON.encode(coupons));
            });
        },

        /**
         * Is the coupon in the order process?
         * @return {boolean}
         */
        isInOrder: function () {
            const OrderProcessNode = this.getElm().getParent(
                '[data-qui="package/quiqqer/order/bin/frontend/controls/OrderProcess"]'
            );

            return !!OrderProcessNode;
        },

        /**
         * Return the order process
         *
         * @return {Object}
         */
        getOrderProcess: function () {
            const OrderProcessNode = this.getElm().getParent(
                '[data-qui="package/quiqqer/order/bin/frontend/controls/OrderProcess"]'
            );

            return window.QUI.Controls.getById(OrderProcessNode.get('data-quiid'));
        },

        /**
         * Remove all coupons from the order
         *
         * @return {Promise}
         */
        removeCouponsFromOrder: function () {
            if (!this.isInOrder()) {
                return Promise.resolve();
            }

            const self = this;

            return new Promise(function (resolve) {
                const OrderProcess = self.getOrderProcess();

                OrderProcess.Loader.show();
                OrderProcess.getOrder().then(function (orderHash) {
                    QUIAjax.get('package_quiqqer_coupons_ajax_frontend_removeCoupons', function () {
                        resolve();
                        OrderProcess.reload();
                    }, {
                        'package': 'quiqqer/coupon',
                        orderHash: orderHash
                    });
                });
            });
        },

        /**
         * Show error message container
         * @param msg
         */
        showErrorMessage: function (msg) {
            if (!msg) {
                console.log("no error message provided");

                return;
            }

            if (!this.$ErrorMsgContainer) {
                return;
            }

            this.$ErrorMsgContainer.set('html', msg);
            this.$ErrorMsgContainer.setAttribute('data-show', '1');
        },

        /**
         * Hide error message container
         */
        hideErrorMessage: function () {
            if (!this.$ErrorMsgContainer) {
                return;
            }

            this.$ErrorMsgContainer.removeAttribute('data-show');
        },

        /**
         * Set data-invalid attribute to the input
         */
        highlightInvalidInput: function () {
            this.$Input.setAttribute('data-invalid', '1');
        },

        /**
         * Remove data-invalid attribute from the input
         */
        resetInvalidInput: function () {
            this.$Input.removeAttribute('data-invalid');
        },

        /**
         * Handle error message (show error message and highlight invalid input)
         *
         * @param err
         */
        handleError: function (err) {
            console.error(err);

            if (err.options !== undefined && err.options.message) {
                const msg = err.options.message;
                this.showErrorMessage(msg);
                this.highlightInvalidInput();
            }

            this.$running = false;
            this.Loader.hide();
        }

    });
});
