<?php

namespace QUI\ERP\Coupons\Products;

use Exception;
use DateTime;
use QUI;
use QUI\ERP\Coupons\CouponCode;
use QUI\ERP\Coupons\Handler as CouponsHandler;
use QUI\ERP\Customer\CustomerFiles;
use QUI\ERP\Discount\Discount;
use QUI\ERP\Discount\Handler as DiscountHandler;
use QUI\ERP\Products\Handler\Products as ProductsHandler;
use QUI\ERP\Products\Product\Product;
use QUI\ExceptionStack;
use QUI\HtmlToPdf\Document;
use QUI\Interfaces\Users\User;
use QUI\Translator;

use function array_merge;
use function basename;
use function date;
use function is_numeric;
use function mb_substr;
use function rename;
use function str_replace;

/**
 * Class Handler
 *
 * Handles coupon code generation from orders / products.
 */
class Handler
{
    /**
     * Special fields for coupon products
     */
    const PRODUCT_FIELD_ID_TRANSFERABLE = 670;
    const PRODUCT_FIELD_ID_GENERATE_PDF = 671;
    const PRODUCT_FIELD_ID_COUPON_AMOUNT = 672;
    const PRODUCT_FIELD_ID_DAYS_VALID = 673;
    const PRODUCT_FIELD_ID_COUPON_DESCRIPTION = 674;
    const PRODUCT_FIELD_ID_IS_SINGLE_PURPOSE_COUPON = 675;
    const PRODUCT_FIELD_ID_SEND_MAIL = 676;
    const PRODUCT_FIELD_ID_USER_DELIVERY_TYPE_SELECT = 678;
    const PRODUCT_FIELD_ID_USER_DELIVERY_TYPE_SELECT_ALLOW = 679;

    /**
     * Create coupons from all coupon products of an order.
     *
     * Each coupon created is sent via e-mail (either as PDF or text).
     *
     * @param QUI\ERP\Order\AbstractOrder $Order
     * @return void
     */
    public static function createCouponCodesFromOrder(QUI\ERP\Order\AbstractOrder $Order): void
    {
        $Customer = $Order->getCustomer();
        $Currency = $Order->getCurrency();

        foreach ($Order->getArticles() as $Article) {
            try {
                // Do not parse coupon codes / discounts
                if (empty($Article->getId()) || !is_numeric($Article->getId())) {
                    continue;
                }

                $Product = ProductsHandler::getProduct((int)$Article->getId());

                // Only parse coupon products
                if (
                    !($Product instanceof DigitalCouponProductType) &&
                    !($Product instanceof PhysicalCouponProductType)
                ) {
                    continue;
                }

                if (!$Customer instanceof User) {
                    QUI\System\Log::addWarning(
                        'Could not generate coupon code for order #' . $Order->getUUID()
                        . ': customer is missing or invalid (-).'
                    );
                    continue;
                }

                $CouponCode = self::createCouponCodeFromProduct($Product, $Order, $Customer);
                $couponFilePathCustomerDir = null;

                $couponAmount = $Product->getFieldValue(self::PRODUCT_FIELD_ID_COUPON_AMOUNT);

                if (!is_numeric($couponAmount)) {
                    $couponAmount = (string)$Product->getPrice()->getValue();
                }

                $Order->addHistory(
                    QUI::getLocale()->get(
                        'quiqqer/coupons',
                        'DownloadProduct.Order.history.coupon_code',
                        [
                            'couponCode' => $CouponCode->getCode(),
                            'productTitle' => $Product->getTitle(),
                            'productId' => $Product->getId(),
                            'couponAmount' => $Currency->format((string)$couponAmount)
                        ]
                    )
                );

                /**
                 * Physical coupons are not sent via mail or PDF
                 */
                if ($Product instanceof PhysicalCouponProductType) {
                    continue;
                }

                // Generate coupon PDF
                if ($Product->getFieldValue(self::PRODUCT_FIELD_ID_GENERATE_PDF)) {
                    $couponPdfFile = self::createCouponCodePdf($CouponCode, $Product);

                    // Rename file
                    $productTitelSanitized = preg_replace("/[^\p{L}\p{N} ]/ui", '', $Product->getTitle());
                    $productTitelSanitized = trim($productTitelSanitized ?? '');

                    if ($productTitelSanitized === '') {
                        $productTitelSanitized = 'coupon';
                    }

                    $fileName = QUI::getLocale()->get(
                        'quiqqer/coupons',
                        'DownloadProduct.pdf.filename',
                        [
                            'productTitle' => str_replace(' ', '_', $productTitelSanitized),
                            'date' => date('Y_m_d')
                        ]
                    );

                    $fileName .= '__' . mb_substr($Order->getUUID(), 0, 6);

                    $newCouponPdfFile = str_replace(basename($couponPdfFile, '.pdf'), $fileName, $couponPdfFile);

                    rename($couponPdfFile, $newCouponPdfFile);

                    // Add PDF file to customer files
                    CustomerFiles::addFileToCustomer($Customer->getUUID(), $newCouponPdfFile);
                    CustomerFiles::addFileToDownloadEntry($Customer->getUUID(), basename($newCouponPdfFile));

                    $customerDir = CustomerFiles::getFolderPath($Customer);
                    $couponFilePathCustomerDir = $customerDir . DIRECTORY_SEPARATOR . basename($newCouponPdfFile);
                }

                // Send coupon via email

                /*
                 * Check if customer explicitly chose "send by email"
                 */
                $deliveryTypeFieldData = $Article->getCustomField(
                    self::PRODUCT_FIELD_ID_USER_DELIVERY_TYPE_SELECT
                );

                if ($deliveryTypeFieldData) {
                    $sendMail = match ($deliveryTypeFieldData['value']) {
                        1 => false,
                        default => true,
                    };
                } else {
                    $sendMail = $Product->getFieldValue(self::PRODUCT_FIELD_ID_SEND_MAIL);
                }

                if ($sendMail) {
                    self::sendCouponMail($CouponCode, $Product, $Customer, $couponFilePathCustomerDir);

                    $Order->addHistory(
                        QUI::getLocale()->get(
                            'quiqqer/coupons',
                            'DownloadProduct.Order.history.coupon_code_send_mail',
                            [
                                'couponCode' => $CouponCode->getCode(),
                                'customerMail' => QUI\ERP\Customer\Utils::getInstance()->getEmailByCustomer($Customer)
                            ]
                        )
                    );
                }
            } catch (Exception $Exception) {
                if ($Exception->getCode() === 404) {
                    QUI\System\Log::writeDebugException($Exception);
                } else {
                    QUI\System\Log::writeException($Exception);
                }
            }
        }
    }

    /**
     * Create coupon codes from an order
     *
     * @param Product $Product
     * @param QUI\ERP\Order\AbstractOrder $Order
     * @param QUI\Interfaces\Users\User $User - The user that bought the product
     * @return CouponCode
     *
     * @throws QUI\Exception
     * @throws Exception
     */
    protected static function createCouponCodeFromProduct(
        Product $Product,
        QUI\ERP\Order\AbstractOrder $Order,
        QUI\Interfaces\Users\User $User
    ): CouponCode {
        $Discount = self::createDiscountFromProduct($Product);
        $couponAttributes = [
            'title' => QUI::getSystemLocale()->get(
                'quiqqer/coupons',
                'ProductCoupon.coupon_title',
                [
                    'orderId' => $Order->getPrefixedNumber(),
                    'productId' => $Product->getId()
                ]
            ),
            'maxUsages' => CouponsHandler::MAX_USAGE_ONCE
        ];

        // Valid until
        $daysValid = $Product->getFieldValue(self::PRODUCT_FIELD_ID_DAYS_VALID);
        $daysValid = is_numeric($daysValid) ? (int)$daysValid : 0;

        if ($daysValid <= 0) {
            $daysValid = 1095; // 3 years;
        }

        $ValidUntil = new DateTime('+ ' . $daysValid . ' days');
        $couponAttributes['validUntilDate'] = $ValidUntil->format('Y-m-d');

        // Transferable (=usable by other users or guests)
        if ($Product->hasField(self::PRODUCT_FIELD_ID_TRANSFERABLE)) {
            $isTransferable = $Product->getFieldValue(self::PRODUCT_FIELD_ID_TRANSFERABLE);

            if (empty($isTransferable)) {
                $couponAttributes['userIds'] = [$User->getUUID()];
            }
        }

        return CouponsHandler::createCouponCode([$Discount->getId()], $couponAttributes);
    }

    /**
     * Creates a new discount for all coupon codes generated by the product.
     *
     * @param Product $Product
     * @return Discount
     *
     * @throws QUI\Exception
     */
    protected static function createDiscountFromProduct(Product $Product): Discount
    {
        $Handler = DiscountHandler::getInstance();

        $discountAmount = $Product->getFieldValue(self::PRODUCT_FIELD_ID_COUPON_AMOUNT);

        if (empty($discountAmount)) {
            $discountAmount = $Product->getPrice()->getValue();
        }

        // Determine discount calculation basis based on coupon type
        $isProductSpecificCoupon = $Product->getFieldValue(
            self::PRODUCT_FIELD_ID_IS_SINGLE_PURPOSE_COUPON
        );

        // @todo für Einzweck-Gutscheine muss ein spezieller Rabatt-Typ verwendet werden

        /** @var Discount $NewDiscount */
        $NewDiscount = $Handler->createChild([
            'active' => 1,
            'discount' => $discountAmount,
            'discount_type' => DiscountHandler::DISCOUNT_TYPE_CURRENCY,
            'hidden' => 1,
            'scope' => DiscountHandler::DISCOUNT_SCOPE_GRAND_TOTAL
        ]);

        $L = new QUI\Locale();
        $discountTitle = [];

        foreach (QUI::availableLanguages() as $lang) {
            $L->setCurrent($lang);

            $discountTitle[$lang] = $L->get('quiqqer/coupons', 'Discount.default_title.product', [
                'productId' => $Product->getId(),
                'productTitle' => $Product->getTitle($L)
            ]);
        }

        Translator::update(
            'quiqqer/discount',
            'discount.' . $NewDiscount->getId() . '.title',
            'quiqqer/discount',
            $discountTitle
        );

        Translator::publish('quiqqer/discount');

        return $NewDiscount;
    }

    /**
     * @param CouponCode $CouponCode - The coupon code
     * @param Product $Product - The product the coupon code was created from
     * @return string - PDF file path
     * @throws QUI\Exception
     */
    public static function createCouponCodePdf(CouponCode $CouponCode, Product $Product): string
    {
        $Document = new Document();

//        $Document->setAttribute('foldingMarks', true);
        $Document->setAttribute('disableSmartShrinking', true);
        $Document->setAttribute('headerSpacing', 0);
        $Document->setAttribute('marginTop', 85);
        $Document->setAttribute('marginBottom', 40);
        $Document->setAttribute('marginLeft', 0);
        $Document->setAttribute('marginRight', 0);
        $Document->setAttribute('showPageNumbers', false);

        $Engine = QUI::getTemplateManager()->getEngine();

        $Engine->assign(
            array_merge(
                [
                    'CouponCode' => $CouponCode,
                    'Product' => $Product->getViewFrontend(),
                ],
                self::getCouponViewData($CouponCode, $Product)
            )
        );

        $tplDir = QUI::getPackage('quiqqer/coupons')->getDir() . 'templates/';

        try {
            $Document->setHeaderHTML($Engine->fetch($tplDir . 'CouponCode.header.html'));
            $Document->setContentHTML($Engine->fetch($tplDir . 'CouponCode.body.html'));
            $Document->setFooterHTML($Engine->fetch($tplDir . 'CouponCode.footer.html'));
        } catch (Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }

        return $Document->createPDF();
    }

    /**
     * Send coupon code via e-mail to a customer.
     *
     * @param CouponCode $CouponCode - The coupon code
     * @param Product $Product - The product the coupon code was created from
     * @param QUI\Interfaces\Users\User $Customer
     * @param string|null $couponPdfFile (optional) - Coupon PDF that is attached to the email
     *
     * @throws Exception
     */
    public static function sendCouponMail(
        CouponCode $CouponCode,
        Product $Product,
        QUI\Interfaces\Users\User $Customer,
        ?string $couponPdfFile
    ): void {
        $recipient = QUI\ERP\Customer\Utils::getInstance()->getEmailByCustomer($Customer);

        if (empty($recipient)) {
            QUI\System\Log::addWarning(
                'Cannot send coupon code e-mail to customer #' . $Customer->getUUID() . ' because user has no'
                . ' email address!'
            );

            return;
        }

        $Mailer = QUI::getMailManager()->getMailer();
        $Mailer->addRecipient($recipient);
        $Mailer->setSubject(QUI::getLocale()->get('quiqqer/coupons', 'ProductCoupon.mail.subject'));

        $couponViewData = self::getCouponViewData($CouponCode, $Product);

        if ($couponViewData['isTransferable']) {
            $transferableInfo = QUI::getLocale()->get(
                'quiqqer/coupons',
                'ProductCoupon.mail.transferable_info.is_transferable'
            );
        } else {
            $transferableInfo = QUI::getLocale()->get(
                'quiqqer/coupons',
                'ProductCoupon.mail.transferable_info.is_not_transferable'
            );
        }

        $profileDownloadInfo = '';

        if ($couponPdfFile) {
            $Mailer->addAttachment($couponPdfFile);

            $profileDownloadInfo = QUI::getLocale()->get(
                'quiqqer/coupons',
                'ProductCoupon.mail.profile_download_info'
            );
        }

        $Mailer->setBody(
            QUI::getLocale()->get(
                'quiqqer/coupons',
                'ProductCoupon.mail.body',
                array_merge(
                    $couponViewData,
                    [
                        'customerName' => $Customer->getName(),
                        'company' => QUI\ERP\Defaults::conf('company', 'name'),
                        'transferableInfo' => $transferableInfo,
                        'profileDownloadInfo' => $profileDownloadInfo
                    ]
                )
            )
        );

        $Mailer->send();
    }

    /**
     * Get view data of a coupon.
     *
     * @param CouponCode $CouponCode
     * @param Product $Product
     * @return array<string, mixed>
     *
     * @throws QUI\Database\Exception
     * @throws QUI\ERP\Products\Product\Exception
     * @throws QUI\Exception
     * @throws ExceptionStack
     */
    protected static function getCouponViewData(CouponCode $CouponCode, Product $Product): array
    {
        $discounts = $CouponCode->getDiscounts();
        $Discount = $discounts[0];
        $Currency = QUI\ERP\Defaults::getCurrency();
        $DiscountAmount = new QUI\ERP\Money\Price($Discount->getAttribute('discount'), $Currency);

        $Locale = QUI::getLocale();

        // Check if coupon is restricted to specific user
        $User = false;
        $restrictedToUserIds = $CouponCode->getUserIds();

        if (!empty($restrictedToUserIds)) {
            $User = QUI::getUsers()->get($restrictedToUserIds[0]);
        }

        // Parse valid until date
        $dateFormat = QUI\ERP\Defaults::getDateFormat();
        $ValidUntilDate = $CouponCode->getValidUntilDate();
        $validUntilDateFormatted = '-';

        if ($ValidUntilDate) {
            $validUntilDateFormatted = $Locale->formatDate(
                $ValidUntilDate->getTimestamp(),
                $dateFormat
            );
        }

        return [
            'isTransferable' => empty($User),
            'couponCode' => $CouponCode->getCode(),
            'productTitle' => $Product->getTitle(),
            'userName' => $User ? $User->getName() : false,
            'couponDescription' => $Product->getFieldValueByLocale(
                self::PRODUCT_FIELD_ID_COUPON_DESCRIPTION
            ),
            'discountAmountFormatted' => $DiscountAmount->getDisplayPrice(),
            'createDateFormatted' => $Locale->formatDate(
                $CouponCode->getCreateDate()->getTimestamp(),
                $dateFormat
            ),
            'validUntilDateFormatted' => $validUntilDateFormatted,
        ];
    }

    /**
     * Get all tax types of which all tax entries have 0% vat!
     *
     * These tax types are relevant for unspecific coupons (german: "Mehrzweck-Gutschein")
     * which are NOT taxed at checkout but only when they are redeemed for a product.
     *
     * @return QUI\ERP\Tax\TaxType[]
     * @throws QUI\Exception
     */
    public static function getNoVatTaxTypes(): array
    {
        $noVatTaxTypes = [];
        $TaxHandler = QUI\ERP\Tax\Handler::getInstance();
        $taxTypes = $TaxHandler->getTaxTypes();

        /** @var QUI\ERP\Tax\TaxType $TaxType */
        foreach ($taxTypes as $TaxType) {
            $taxEntries = $TaxHandler->getChildren([
                'where' => [
                    'taxTypeId' => $TaxType->getId()
                ]
            ]);

            /** @var QUI\ERP\Tax\TaxEntry $TaxEntry */
            foreach ($taxEntries as $TaxEntry) {
                if ($TaxEntry->getValue() > 0) {
                    continue 2;
                }
            }

            $noVatTaxTypes[] = $TaxType;
        }

        return $noVatTaxTypes;
    }
}
