<?php

namespace QUI\ERP\Accounting\Template;

use QUI;
use QUI\ERP\BankAccounts\Handler as BankAccounts;
use QUI\ERP\Output\OutputTemplateProviderInterface;
use QUI\Interfaces\Template\EngineInterface;
use QUI\Locale;

use function file_exists;
use function file_get_contents;
use function strtotime;

class Template implements OutputTemplateProviderInterface
{
    /**
     * Entity types
     */
    const ENTITY_TYPE_CANCELLED = 'Canceled';
    const ENTITY_TYPE_CONTRACT = 'Contract';
    const ENTITY_TYPE_CONTRACT_CANCEL_CONFIRM = 'ContractCancelConfirm';
    const ENTITY_TYPE_CREDIT_NOTE = 'CreditNote';
    const ENTITY_TYPE_INVOICE = 'Invoice';
    const ENTITY_TYPE_ORDER = 'Order';
    const ENTITY_TYPE_OFFER = 'Offer';
    const ENTITY_TYPE_DUNNING = 'Dunning';
    const ENTITY_TYPE_OPEN_ITEMS_LIST = 'OpenItemsList';
    const ENTITY_TYPE_SEPA_DIRECT_DEBIT_AGREEMENT = 'SEPADirectDebitAgreement';
    const ENTITY_TYPE_PURCHASING_PROCESS = 'PurchasingProcess';
    const ENTITY_TYPE_CASHBOOK_JOURNAL = 'CashbookJournal';
    const ENTITY_TYPE_SALES_ORDER = 'SalesOrder';
    const ENTITY_TYPE_BOOKING_CONFIRMATION = 'BookingConfirmation';

    /**
     * Get all output types the template package provides templates for
     *
     * @return string[]
     */
    public static function getEntityTypes(): array
    {
        return [
            self::ENTITY_TYPE_CANCELLED,
            self::ENTITY_TYPE_CONTRACT,
            self::ENTITY_TYPE_CREDIT_NOTE,
            self::ENTITY_TYPE_INVOICE,
            self::ENTITY_TYPE_OFFER,
            self::ENTITY_TYPE_ORDER,
            self::ENTITY_TYPE_DUNNING,
            self::ENTITY_TYPE_OPEN_ITEMS_LIST,
            self::ENTITY_TYPE_SEPA_DIRECT_DEBIT_AGREEMENT,
            self::ENTITY_TYPE_PURCHASING_PROCESS,
            self::ENTITY_TYPE_CASHBOOK_JOURNAL,
            self::ENTITY_TYPE_CONTRACT_CANCEL_CONFIRM,
            self::ENTITY_TYPE_SALES_ORDER,
            self::ENTITY_TYPE_BOOKING_CONFIRMATION
        ];
    }

    /**
     * Get all available templates for $entityType
     *
     * @param string $entityType
     * @return string[]|int[] - Collection of templateIds
     */
    public static function getTemplates(string $entityType): array
    {
        return match ($entityType) {
            self::ENTITY_TYPE_CANCELLED, self::ENTITY_TYPE_CONTRACT, self::ENTITY_TYPE_CREDIT_NOTE, self::ENTITY_TYPE_INVOICE, self::ENTITY_TYPE_OFFER, self::ENTITY_TYPE_ORDER, self::ENTITY_TYPE_DUNNING, self::ENTITY_TYPE_OPEN_ITEMS_LIST, self::ENTITY_TYPE_SEPA_DIRECT_DEBIT_AGREEMENT, self::ENTITY_TYPE_PURCHASING_PROCESS, self::ENTITY_TYPE_CASHBOOK_JOURNAL, self::ENTITY_TYPE_CONTRACT_CANCEL_CONFIRM, self::ENTITY_TYPE_SALES_ORDER, self::ENTITY_TYPE_BOOKING_CONFIRMATION => ['system_default'],
            default => [],
        };
    }

    /**
     * Get title of Template
     *
     * @param int|string $templateId
     * @param Locale|null $Locale (optional) - If omitted use \QUI::getLocale()
     * @return string
     */
    public static function getTemplateTitle(int | string $templateId, null | Locale $Locale = null): string
    {
        if (empty($Locale)) {
            $Locale = QUI::getLocale();
        }

        return $Locale->get('quiqqer/erp-accounting-templates', 'template.title');
    }

    /**
     * Get HTML for document header area
     *
     * @param int|string $templateId
     * @param string $entityType
     * @param EngineInterface $Engine
     * @param mixed $Entity - The entity the output is created for
     * @return string|false
     *
     * @throws QUI\Exception
     */
    public static function getHeaderHtml(
        int | string $templateId,
        string $entityType,
        EngineInterface $Engine,
        mixed $Entity
    ): bool | string {
        $tplDir = self::getTemplateDir();
        $tplTypeDir = $tplDir . $entityType . '/';

        if (file_exists($tplTypeDir . 'header.html')) {
            $htmlFile = $tplTypeDir . 'header.html';
        } else {
            $htmlFile = $tplDir . 'header.html';
        }

        $output = '';

        if (file_exists($tplTypeDir . 'header.css')) {
            $output .= '<style>' . file_get_contents($tplTypeDir . 'header.css') . '</style>';
        } else {
            $output .= '<style>' . file_get_contents($tplDir . 'header.css') . '</style>';
        }

        $Instance = $Engine->getTemplateVariable('this');
        $Formatter = QUI::getLocale()->getDateFormatter();

        $Engine->assign('entityType', $entityType);

        switch ($entityType) {
            case self::ENTITY_TYPE_DUNNING:
            case self::ENTITY_TYPE_OFFER:
                break;

            case self::ENTITY_TYPE_CANCELLED:
            case self::ENTITY_TYPE_CREDIT_NOTE:
            case self::ENTITY_TYPE_INVOICE:
                $date = $Instance->getInvoice()->getAttribute('date');
                $date = strtotime($date);
                $date = $Formatter->format($date);

                $Engine->assign('dateFormatted', $date);

                //@todo sub header

                break;

            case self::ENTITY_TYPE_ORDER:
                /* @var $Instance QUI\ERP\Order\Order */
                $Engine->assign('dateFormatted', $Instance->getCreateDate());
                break;

            case self::ENTITY_TYPE_CONTRACT:
                $Engine->assign('dateFormatted', $Instance->getDate());
                break;

            case self::ENTITY_TYPE_OPEN_ITEMS_LIST:
                $OpenItemsList = $Engine->getTemplateVariable('OpenItemsList');
                $Engine->assign('dateFormatted', $OpenItemsList->getDateFormatted());
                break;

            case self::ENTITY_TYPE_SEPA_DIRECT_DEBIT_AGREEMENT:
                $Package = QUI::getPackage('quiqqer/erp');
                $Engine->assign('company', $Package->getConfig()->getSection('company'));
                break;

            case self::ENTITY_TYPE_PURCHASING_PROCESS:
                $Engine->assign('Customer', $Instance->getSupplier());

                $data = $Instance->toArray();
                $Engine->assign('dateFormatted', $data['date']);

                if (!empty($data['payment_method']) && class_exists('QUI\ERP\Accounting\Payments\Payments')) {
                    $Engine->assign(
                        'Payment',
                        QUI\ERP\Accounting\Payments\Payments::getInstance()->getPayment(
                            $data['payment_method']
                        )
                    );
                } else {
                    $Engine->assign('Payment', false);
                }

                break;
        }

        $Engine->assign('Entity', $Entity);

        $output .= $Engine->fetch($htmlFile);

        return $output;
    }

    /**
     * Get HTML for document body area
     *
     * @param int|string $templateId
     * @param string $entityType
     * @param EngineInterface $Engine
     * @param mixed $Entity - The entity the output is created for
     * @return string|false
     *
     * @throws QUI\Exception
     */
    public static function getBodyHtml(
        int | string $templateId,
        string $entityType,
        EngineInterface $Engine,
        mixed $Entity
    ): bool | string {
        $tplDir = self::getTemplateDir();
        $tplTypeDir = self::getTemplateDir() . $entityType . '/';
        $htmlFile = $tplTypeDir . 'body.html';

        if (!file_exists($htmlFile)) {
            throw new QUI\Exception('Template file ' . $htmlFile . ' not found.');
        }

        $output = '';

        $Engine->assign('entityType', $entityType);

        if (file_exists($tplTypeDir . 'body.css')) {
            $output .= '<style>' . file_get_contents($tplTypeDir . 'body.css') . '</style>';
        } else {
            $output .= '<style>' . file_get_contents($tplDir . 'body.css') . '</style>';
        }

        $output .= $Engine->fetch($htmlFile);

        return $output;
    }

    /**
     * Get HTML for document footer area
     *
     * @param int|string $templateId
     * @param string $entityType
     * @param EngineInterface $Engine
     * @param mixed $Entity - The entity the output is created for
     * @return string|false
     *
     * @throws QUI\Exception
     */
    public static function getFooterHtml(
        int | string $templateId,
        string $entityType,
        EngineInterface $Engine,
        mixed $Entity
    ): bool | string {
        $tplDir = self::getTemplateDir();
        $tplTypeDir = $tplDir . $entityType . '/';

        if (file_exists($tplTypeDir . 'footer.html')) {
            $htmlFile = $tplTypeDir . 'footer.html';
        } else {
            $htmlFile = $tplDir . 'footer.html';
        }

        $output = '';

        $Engine->assign('entityType', $entityType);

        if (file_exists($tplTypeDir . 'footer.css')) {
            $output .= '<style>' . file_get_contents($tplTypeDir . 'footer.css') . '</style>';
        } else {
            $output .= '<style>' . file_get_contents($tplDir . 'footer.css') . '</style>';
        }

        try {
            $Config = QUI::getPackage('quiqqer/erp')->getConfig();

            $Engine->assign('Config', $Config);
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeDebugException($Exception);
        }

        // Company bank account data
        $Engine->assign('bankAccount', BankAccounts::getCompanyBankAccount());
        $Engine->assign('Entity', $Entity);

        $output .= $Engine->fetch($htmlFile);

        return $output;
    }

    /**
     * Get main template directory
     *
     * @return string
     *
     * @throws QUI\Exception
     */
    protected static function getTemplateDir(): string
    {
        return QUI::getPackage('quiqqer/erp-accounting-templates')->getDir() . 'template/';
    }
}
