<?php

/**
 * This file contains QUI\FontAwesome\EventHandler
 */

namespace QUI\FontAwesome;

use QUI;
use Smarty;
use SmartyException;

use function is_dir;
use function rtrim;

/**
 * Class EventHandler
 * @package QUI\FontAwesome
 */
class EventHandler
{
    /**
     * Event : on smarty init
     * @param Smarty $Smarty - \Smarty
     * @throws SmartyException
     */
    public static function onSmartyInit(Smarty $Smarty): void
    {
        // {fontawesome}
        if (
            !isset($Smarty->registered_plugins['function']) ||
            !isset($Smarty->registered_plugins['function']['fontawesome'])
        ) {
            $Smarty->registerPlugin("function", "fontawesome", "\\QUI\\FontAwesome\\EventHandler::fontawesome");
        }
    }

    /**
     * Smarty fontawesome function {fontawesome}
     *
     * @return string
     */
    public static function fontawesome(): string
    {
        return FontAwesome::getLinkHtml();
    }

    /**
     * event : on icons init
     *
     * @param QUI\Icons\Handler $Icons
     */
    public static function onIconsInit(QUI\Icons\Handler $Icons): void
    {
        // css files
        $files = FontAwesome::getFiles();

        foreach ($files as $file) {
            $Icons->addCSSFile($file);
        }

        // css classes
        $cssList = QUI\FontAwesome\Util::getCSSClassList();

        foreach ($cssList as $key => $value) {
            $cssList[$key] = 'fa ' . $value;
        }

        $Icons->addIcons($cssList);
    }

    /**
     * @param QUI\Package\Package $Package
     * @return void
     */
    public static function onPackageInstall(QUI\Package\Package $Package): void
    {
        if ($Package->getName() !== 'quiqqer/fontawesome') {
            return;
        }

        self::moveToBin();
    }

    /**
     * @param QUI\Package\Package $Package
     * @return void
     */
    public static function onPackageSetup(QUI\Package\Package $Package): void
    {
        if ($Package->getName() !== 'quiqqer/fontawesome') {
            return;
        }

        // quiqqer/package-fontawesome#8
        $packagesBin = OPT_DIR . 'bin/fontawesome/';

        if (!is_dir($packagesBin)) {
            self::moveToBin();
        }
    }

    /**
     * @param QUI\Package\Package $Package
     * @return void
     */
    public static function onPackageUpdate(QUI\Package\Package $Package): void
    {
        if ($Package->getName() !== 'quiqqer/fontawesome') {
            return;
        }

        self::moveToBin();
    }

    /**
     * moves the fortawesome/font-awesome plugin to the packages/bin folder
     *
     * @return void
     */
    protected static function moveToBin(): void
    {
        // move fontawesome 6 php plugin to packages/bin
        $packagesBin = OPT_DIR . 'bin/fontawesome/';
        $faDir = OPT_DIR . 'fortawesome/font-awesome/';

        if (!is_dir($faDir)) {
            return;
        }

        try {
            if (is_dir($packagesBin)) {
                QUI\Utils\System\File::unlink($packagesBin);
            }

            QUI\Utils\System\File::mkdir($packagesBin);

            self::copyDirectory($faDir, $packagesBin);

            // remove unwanted stuff
            self::deleteFilesWithExtension($packagesBin, 'md');
            self::deleteFilesWithExtension($packagesBin, 'json');
            self::deleteFilesWithExtension($packagesBin, 'txt');

            $unwantedFolders = ['.github', 'js-packages', 'less', 'metadata'];

            foreach ($unwantedFolders as $folder) {
                if (is_dir($packagesBin . $folder)) {
                    QUI\Utils\System\File::unlink($packagesBin . $folder);
                }
            }
        } catch (QUI\Exception $exception) {
            QUI\System\Log::addError($exception->getMessage(), [
                'plugin' => 'quiqqer/fontawesome',
                'method' => 'moveToBin'
            ]);
        }
    }

    /**
     * @param string $src
     * @param string $dst
     * @return void
     */
    protected static function copyDirectory(string $src, string $dst): void
    {
        $dir = opendir($src);

        if (!file_exists($dst)) {
            mkdir($dst);
        }

        while (($file = readdir($dir)) !== false) {
            if (($file === '.') || ($file === '..')) {
                continue;
            }

            if (is_dir($src . '/' . $file)) {
                self::copyDirectory(
                    $src . '/' . $file,
                    $dst . '/' . $file
                );
            } else {
                copy($src . '/' . $file, $dst . '/' . $file);
            }
        }

        closedir($dir);
    }

    /**
     * @param string $dir
     * @param string $extension
     * @return void
     */
    protected static function deleteFilesWithExtension(string $dir, string $extension): void
    {
        $dir = rtrim($dir, '/');
        $files = scandir($dir);

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            $filePath = $dir . '/' . $file;

            if (is_dir($filePath)) {
                self::deleteFilesWithExtension($filePath, $extension);
            } elseif (pathinfo($filePath, PATHINFO_EXTENSION) === $extension) {
                unlink($filePath);
            }
        }
    }
}
