<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\GDPR\Controls;

use QUI;
use QUI\Control;
use QUI\Exception;
use QUI\GDPR\CookieInterface;
use QUI\GDPR\CookieManager;
use QUI\Projects\Project;
use QUI\Projects\Site;

use function in_array;

/**
 * Class CookieConsent
 *
 * @package QUI\GDPR\Controls
 */
class CookieConsent extends Control
{
    /**
     * @throws Exception
     */
    public function __construct($attributes = [])
    {
        $Project = QUI::getRewrite()->getProject();

        parent::__construct($attributes);

        $this->setJavaScriptControlOption('position', $this::getPosition($Project));
        $this->setJavaScriptControlOption('privacy-url', $this);
        $this->setJavaScriptControlOption('imprint-url', $this::getImprintUrl($Project));
        $this->setJavaScriptControlOption('blocksite', $Project->getConfig('gdpr.cookieconsent.blocksite'));
        $this->setJavaScriptControlOption(
            'show-accept-all-button',
            $Project->getConfig('gdpr.cookieconsent.showAcceptAllButton')
        );

        $this->setJavaScriptControl('package/quiqqer/gdpr/bin/controls/CookieConsent');

        $this->addCSSFile(dirname(__FILE__) . '/CookieConsent.css');
    }

    public function getBody(): string
    {
        $Project = QUI::getRewrite()->getProject();
        $cacheName = $this->getTemplateCacheName($Project);

        try {
            return QUI\Cache\Manager::get($cacheName);
        } catch (QUI\Cache\Exception) {
        }


        $TemplateEngine = QUI::getTemplateManager()->getEngine();
        $TemplateEngine->assign([
            'Control' => $this,
            'Project' => $Project,
            'cookiesGroupedByCategory' => CookieManager::getInstance()->getAllRegisteredCookiesGroupedByCategory(
                $Project
            ),
            'requiredCookieCategory' => CookieInterface::COOKIE_CATEGORY_ESSENTIAL,
            'preselectedCookieCategories' => CookieManager::getPreselectedCookieCategories($Project)
        ]);

        $result = $TemplateEngine->fetch(dirname(__FILE__) . '/CookieConsent.html');

        QUI\Cache\Manager::set($cacheName, $result);

        return $result;
    }


    /**
     * Returns the cookie banner's position for the given project
     *
     * @param Project $Project
     * @return string
     */
    public static function getPosition(Project $Project): string
    {
        $position = $Project->getConfig('gdpr.cookieconsent.position');
        $allowedPositions = ['top', 'bottom', 'topSlide', 'popup'];

        if (!in_array($position, $allowedPositions)) {
            $position = 'bottom';
        }

        return $position;
    }


    /**
     * Returns the URL of the first site with the legal notes sitetype for a given project.
     *
     * @param Project $Project
     *
     * @return false|string
     * @throws QUI\Database\Exception
     */
    public static function getImprintUrl(Project $Project): bool|string
    {
        return self::getSiteUrlBySitetype($Project, 'quiqqer/sitetypes:types/legalnotes');
    }


    /**
     * Returns the URL of the first site with the privacypolicy sitetype for a given project.
     *
     * @param Project $Project
     *
     * @return false|string
     * @throws QUI\Database\Exception
     */
    public static function getPrivacyUrl(Project $Project): bool|string
    {
        return self::getSiteUrlBySitetype($Project, 'quiqqer/sitetypes:types/privacypolicy');
    }

    /**
     * Returns the URL of the first site of a given sitetype for a given project.
     *
     * @param Project $Project
     * @param string $type
     *
     * @return false|string
     * @throws QUI\Database\Exception
     */
    protected static function getSiteUrlBySitetype(Project $Project, string $type): bool|string
    {
        /* @var $result Site[] */
        $result = $Project->getSites([
            'where' => [
                'type' => $type
            ],
            'limit' => 1
        ]);

        if (!isset($result[0])) {
            return false;
        }

        try {
            return $result[0]->getUrlRewritten();
        } catch (Exception) {
            return false;
        }
    }

    /**
     * Returns the cookie-banner text for the given project.
     *
     * @param Project $Project
     * @return string
     */
    public static function getText(Project $Project): string
    {
        $lg = 'quiqqer/gdpr';
        $locale = QUI::getLocale();

        $text = $locale->get($lg, "setting.text.project.{$Project->getName()}");

        if ($locale->isLocaleString($text) || empty($text)) {
            $text = $locale->get($lg, 'setting.text.default');
        }

        return $text;
    }

    /**
     * Returns the cookie-banner text for the given project.
     *
     * @param Project $Project
     * @return string
     */
    public static function getButtonText(Project $Project): string
    {
        $lg = 'quiqqer/gdpr';
        $locale = QUI::getLocale();

        $text = $locale->get($lg, "setting.buttontext.project.{$Project->getName()}");

        if ($locale->isLocaleString($text) || empty($text)) {
            $text = $locale->get($lg, 'setting.buttontext.default');
        }

        return $text;
    }

    /**
     * Returns the cookie-banner text for the given project.
     *
     * @param Project $Project
     * @return string
     */
    public static function getAcceptAllButtonText(Project $Project): string
    {
        $lg = 'quiqqer/gdpr';
        $locale = QUI::getLocale();

        $text = $locale->get($lg, "setting.buttontext.accept.all.project.{$Project->getName()}");

        if ($locale->isLocaleString($text) || empty($text)) {
            $text = $locale->get($lg, 'setting.buttontext.accept.all.default');
        }

        return $text;
    }

    /**
     * Returns the "Accept selected"-button text for the given project.
     *
     * @param Project $Project
     * @return string
     */
    public static function getAcceptSelectedButtonText(Project $Project): string
    {
        $lg = 'quiqqer/gdpr';
        $locale = QUI::getLocale();

        $text = $locale->get($lg, "setting.buttontext.accept.selected.project.{$Project->getName()}");

        if ($locale->isLocaleString($text) || empty($text)) {
            $text = $locale->get($lg, 'setting.buttontext.accept.selected.default');
        }

        return $text;
    }

    /**
     * Returns the name of the cache key used to store the control's body/template
     *
     * @param Project $Project
     *
     * @return string
     */
    protected function getTemplateCacheName(Project $Project): string
    {
        return "quiqqer/gdpr/controls/cookieConsent/template/{$Project->getName()}/{$Project->getLang()}";
    }
}
