/**
 * @module package/quiqqer/gdpr/bin/classes/CookieManager
 * @author www.pcsg.de (Jan Wennrich)
 */
define('package/quiqqer/gdpr/bin/classes/CookieManager', [
    'qui/controls/Control',
    'Ajax'
], function (QUIControl, QUIAjax) {
    "use strict";

    return new Class({
        Type: 'package/quiqqer/gdpr/bin/classes/CookieManager',

        Extends: QUIControl,

        /**
         * Sets the given cookie-categories as accepted.
         * Sends the categories to the server to process them and mark them as accepted.
         * Once the server-request is complete, the callback passed as the second parameter will be called.
         *
         * Fires the 'onCookieCategoriesAccepted' with the accepted categories.
         *
         * @param categories String[]
         * @param callback function
         *
         * @fires 'onCookieCategoriesAccepted'
         */
        acceptCookieCategories: function (categories, callback) {
            require(['qui/QUI'], function (QUI) {
                QUI.Storage.set('quiqqer-cookies-accepted-timestamp', Date.now());
                QUI.Storage.set(GDPR.storageKeys.hideBanner, '1');
                QUI.Storage.set(GDPR.storageKeys.acceptedCookiesHash, GDPR.getCookiesHash());
                QUI.Storage.set(GDPR.storageKeys.acceptedCookieCategories, JSON.stringify(categories));

                this.fireEvent('cookieCategoriesAccepted', categories);
                QUI.fireEvent('cookieCategoriesAccepted', categories);
            }.bind(this));
        },

        /**
         * Returns a promise which resolves with an array of the accepted cookie categories.
         *
         * @return {Promise}
         */
        getAcceptedCookieCategories: function () {
            return new Promise(function (resolve) {
                require(['qui/QUI'], function (QUI) {
                    var acceptedCookies = JSON.decode(QUI.Storage.get(GDPR.storageKeys.acceptedCookieCategories));

                    // No cookies accepted yet -> return empty array
                    if (!acceptedCookies) {
                        acceptedCookies = [];
                    }

                    resolve(acceptedCookies);
                });
            });
        },

        /**
         * Revokes all of the user's cookie settings.
         *
         * Set the first parameter to true to reload the page afterwards.
         *
         * @fires 'onCookiesRevoked'
         */
        revokeCookies: function (reloadPage) {
            if (reloadPage === undefined) {
                reloadPage = false;
            }

            require(['qui/QUI'], function (QUI) {
                QUI.Storage.remove('quiqqer-cookies-accepted-timestamp');
                QUI.Storage.remove(GDPR.storageKeys.acceptedCookieCategories);
                QUI.Storage.remove(GDPR.storageKeys.acceptedCookiesHash);
            });

            if (window.localStorage) {
                window.localStorage.removeItem(GDPR.storageKeys.hideBanner);
            }

            if (reloadPage) {
                location.reload();
            }
        },


        /**
         * Returns all of the given project's cookies formatted for a Grid-control.
         *
         * @param projectName
         * @param projectLanguage
         * @param page
         * @param perPage
         * @param sortOn
         * @param sortBy
         */
        getCookiesForGrid: function (projectName, projectLanguage, page, perPage, sortOn, sortBy) {
            return new Promise(function (resolve, reject) {
                QUIAjax.get('package_quiqqer_gdpr_ajax_getCookiesForGrid', resolve, {
                    'package'      : 'quiqqer/gdpr',
                    onError        : reject,
                    projectName    : projectName,
                    projectLanguage: projectLanguage,
                    page           : page,
                    perPage        : perPage,
                    sortOn         : sortOn,
                    sortBy         : sortBy
                });
            });
        },


        /**
         * Edit's the cookie with the given ID to take the given arguments.
         *
         * @param data
         * @param projectName
         * @param projectLanguage
         */
        editCookie: function (data, projectName, projectLanguage) {
            return new Promise(function (resolve, reject) {
                QUIAjax.post('package_quiqqer_gdpr_ajax_editCookie', resolve, {
                    'package'      : 'quiqqer/gdpr',
                    onError        : reject,
                    projectName    : projectName,
                    projectLanguage: projectLanguage,
                    data           : JSON.stringify(data)
                });
            });
        },


        /**
         * Deletes the manually added cookie with the given ID
         *
         * @param id
         * @param projectName
         * @param projectLanguage
         */
        deleteCookie: function (id, projectName, projectLanguage) {
            return new Promise(function (resolve, reject) {
                QUIAjax.post('package_quiqqer_gdpr_ajax_deleteCookie', resolve, {
                    'package'      : 'quiqqer/gdpr',
                    onError        : reject,
                    projectName    : projectName,
                    projectLanguage: projectLanguage,
                    id             : id
                });
            });
        }
    });
});
