/**
 * Manage payments restrictions for users and groups
 *
 * @module package/quiqqer/gdpr/bin/controls/user/GdprActions
 * @author www.pcsg.de (Patrick Müller)
 */
define('package/quiqqer/gdpr/bin/controls/user/GdprActions', [

    'qui/controls/Control',
    'qui/controls/loader/Loader',
    'qui/controls/windows/Confirm',
    'qui/controls/buttons/Button',

    'Permissions',
    'Ajax',
    'Locale',

    'css!package/quiqqer/gdpr/bin/controls/user/GdprActions.css'

], function (QUIControl, QUILoader, QUIConfirm, QUIButton, Permissions, QUIAjax, QUILocale) {
    "use strict";

    var lg = 'quiqqer/gdpr';

    return new Class({

        Extends: QUIControl,
        Type   : 'package/quiqqer/gdpr/bin/controls/user/GdprActions',

        Binds: [
            '$onInject',
            '$load',
            '$onClickDelete',
            '$deleteUser'
        ],

        options: {
            userId: false,       // QUIQQER user id
        },

        initialize: function (options) {
            this.parent(options);

            this.Loader      = new QUILoader();
            this.$Grid       = null;
            this.$GridParent = null;

            this.addEvents({
                onInject : this.$onInject,
                onRefresh: this.$onRefresh,
            });
        },

        /**
         * Event: onImport
         */
        $onInject: function () {
            this.$Elm.addClass(
                'quiqqer-gdpr-user-actions'
            );

            this.Loader.inject(this.$Elm);
            this.$load();
        },

        /**
         * Load memberships management
         */
        $load: async function () {
            this.Loader.show();

            const hasLicense = await this.$hasLicense();
            this.Loader.hide();

            if (!hasLicense) {
                new Element('div', {
                    'class': 'quiqqer-gdpr-user-actions-no-license',
                    html   : QUILocale.get(lg, 'controls.GdprActions.no_license')
                }).inject(this.$Elm);

                return;
            }

            Promise.all([
                Permissions.hasPermission('quiqqer.gdpr.request.create_gdpr_report'),
                Permissions.hasPermission('quiqqer.gdpr.request.delete_user')
            ]).then((result) => {
                const canCreateReport = result[0];
                const canDeleteUser   = result[1];

                if (!canCreateReport && !canDeleteUser) {
                    new Element('div', {
                        'class': 'quiqqer-gdpr-user-actions-no-permission',
                        html   : QUILocale.get(lg, 'controls.GdprActions.no_permission')
                    }).inject(this.$Elm);
                    return;
                }

                // Generate data request
                if (canCreateReport) {
                    new QUIButton({
                        'class'  : 'quiqqer-gdpr-user-actions-data',
                        textimage: 'fa fa-id-card-o',
                        text     : QUILocale.get(lg, 'controls.GdprActions.btn.get_data_report'),
                        title    : QUILocale.get(lg, 'controls.GdprActions.btn.get_data_report'),
                        events   : {
                            onClick: (Btn) => {
                                const id = 'accounting-export-download-' + '_' + Math.random().toString(36).substr(2, 9);

                                Btn.disable();
                                Btn.setAttribute('textimage', 'fa fa-spinner fa-spin');

                                new Element('iframe', {
                                    src   : URL_OPT_DIR + 'quiqqer/gdpr/bin/backend/download.php?uid=' + this.getAttribute('userId'),
                                    id    : id,
                                    styles: {
                                        position: 'absolute',
                                        top     : -200,
                                        left    : -200,
                                        width   : 50,
                                        height  : 50
                                    }
                                }).inject(document.body);

                                (function () {
                                    Btn.enable();
                                    Btn.setAttribute('textimage', 'fa fa-id-card-o');
                                }).delay(2000, this);

                                (function () {
                                    document.getElements('#' + id).destroy();
                                }).delay(20000, this);
                            }
                        }
                    }).inject(this.$Elm);
                }

                // Delete user
                if (canDeleteUser) {
                    new QUIButton({
                        'class'  : 'quiqqer-gdpr-user-actions-data',
                        textimage: 'fa fa-trash',
                        text     : QUILocale.get(lg, 'controls.GdprActions.btn.delete'),
                        title    : QUILocale.get(lg, 'controls.GdprActions.btn.delete'),
                        events   : {
                            onClick: this.$onClickDelete
                        },
                        styles   : {
                            'margin-left': 10
                        }
                    }).inject(this.$Elm);
                }
            });
        },

        /**
         * "Delete user" click
         */
        $onClickDelete: function () {
            let userUuidInput;

            new QUIConfirm({
                maxHeight: 400,
                maxWidth : 700,

                autoclose         : false,
                backgroundClosable: true,

                information: QUILocale.get(lg, 'controls.GdprActions.delete.information'),
                title      : QUILocale.get(lg, 'controls.GdprActions.delete.title'),
                texticon   : 'fa fa-trash',
                text       : QUILocale.get(lg, 'controls.GdprActions.delete.text', {
                    userId: this.getAttribute('userId')
                }),
                icon       : 'fa fa-trash',

                cancel_button: {
                    text     : false,
                    textimage: 'icon-remove fa fa-remove'
                },
                ok_button    : {
                    text     : QUILocale.get(lg, 'controls.GdprActions.delete.submit'),
                    textimage: 'icon-ok fa fa-trash'
                },
                events       : {
                    onOpen  : (Win) => {
                        const content = Win.getContent();

                        userUuidInput = new Element('input', {
                            'class': 'quiqqer-gdpr-user-actions-data-userUuid-input',
                            type   : 'text'
                        }).inject(content);

                        userUuidInput.focus();
                    },
                    onSubmit: (Win) => {
                        if (userUuidInput.value !== this.getAttribute('userId')) {
                            userUuidInput.focus();
                            return;
                        }

                        Win.Loader.show();

                        this.$deleteUser().then(() => {
                            Win.close();
                        }).catch(() => {
                            Win.Loader.hide();
                        });
                    }
                }
            }).open();
        },

        /**
         * Delete user data
         *
         * @return {Promise}
         */
        $deleteUser: function () {
            return new Promise((resolve, reject) => {
                QUIAjax.post('package_quiqqer_gdpr_ajax_backend_deleteUserData', resolve, {
                    'package': lg,
                    userId   : this.getAttribute('userId'),
                    onError  : reject
                });
            });
        },

        $hasLicense: async function () {
            return new Promise((resolve, reject) => {
                QUIAjax.post('package_quiqqer_gdpr_ajax_backend_hasLicense', resolve, {
                    'package': lg,
                    onError  : reject
                });
            });
        }
    });
});
