<?php

namespace QUI\GDPR\DataRequest\Controls\Profile;

use QUI;
use QUI\FrontendUsers\Controls\Profile\AbstractProfileControl;
use QUI\GDPR\DataRequest\Handler;
use QUI\Verification\Interface\VerificationRepositoryInterface;
use QUI\Verification\VerificationRepository;

/**
 * Class DataRequest
 *
 * Delete user account
 */
class DataRequest extends AbstractProfileControl
{
    /**
     * DataRequest constructor.
     * @param array $attributes
     * @param VerificationRepositoryInterface|null $verificationRepository
     */
    public function __construct(
        array $attributes = [],
        private ?VerificationRepositoryInterface $verificationRepository = null
    ) {
        parent::__construct($attributes);

        if (is_null($this->verificationRepository)) {
            $this->verificationRepository = new VerificationRepository();
        }

        $this->addCSSClass('quiqqer-gdpr-profile');
        $this->addCSSFile(dirname(__FILE__) . '/DataRequest.css');

        $this->setJavaScriptControl('package/quiqqer/gdpr/bin/controls/profile/DataRequest');
    }

    /**
     * @return string
     * @throws QUI\Exception
     */
    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();
        $actionDelete = false;
        $user = QUI::getUserBySession();

        // Check delete request status
        try {
            $deleteVerification = $this->verificationRepository->findByIdentifier(
                Handler::getDeleteUserConfirmationVerificationIdentifier($user)
            );

            if (!is_null($deleteVerification)) {
                if ($deleteVerification->isValid()) {
                    $actionDelete = 'deleteaccount_confirm_wait';
                    $this->setJavaScriptControlOption('deletestarted', 1);
                } else {
                    $this->verificationRepository->delete($deleteVerification);
                }
            }
        } catch (\Exception $Exception) {
            // nothing - no active user delete verification
        }

        if (empty($actionDelete) && !empty($_GET['action'])) {
            $actionDelete = $_GET['action'];
        }

        $csrfToken = \uniqid('', true);
        QUI::getSession()->set('gdpr_csrf_token', $csrfToken);

        $Engine->assign([
            'User' => QUI::getUserBySession(),
            'action' => $actionDelete,
            'csrfToken' => $csrfToken,
            'dataMailSent' => QUI::getSession()->get('gdpr_data_mail_sent'),
            'deleteMailSent' => QUI::getSession()->get('gdpr_delete_mail_sent'),
            'dataMailError' => QUI::getSession()->get('gdpr_data_mail_sent_error'),
            'deleteMailError' => QUI::getSession()->get('gdpr_delete_mail_sent_error'),
        ]);

        QUI::getSession()->set('gdpr_data_mail_sent', false);
        QUI::getSession()->set('gdpr_delete_mail_sent', false);
        QUI::getSession()->set('gdpr_data_mail_sent_error', false);
        QUI::getSession()->set('gdpr_delete_mail_sent_error', false);

        return $Engine->fetch(dirname(__FILE__) . '/DataRequest.tpl');
    }

    /**
     * event: on save
     *
     * @throws \Exception
     */
    public function onSave(): void
    {
        if (empty($_REQUEST['data'])) {
            return;
        }

        $requestData = \json_decode($_REQUEST['data'], true);

        if (empty($requestData['gdpr_csrf_token'])) {
            return;
        }

        $csrfTokenSession = QUI::getSession()->get('gdpr_csrf_token');

        if ($requestData['gdpr_csrf_token'] !== $csrfTokenSession) {
            return;
        }

        try {
            $User = QUI::getUserBySession();

            if (isset($requestData['gdpr_request']) && $requestData['gdpr_request'] === 'data') {
                Handler::sendDataRequestPdfMail($User);
                QUI::getSession()->set('gdpr_data_mail_sent', true);
            }

            if (isset($requestData['gdpr_delete']) && $requestData['gdpr_request'] === 'delete') {
                Handler::sendDeleteUserConfirmationMail($User);
                QUI::getSession()->set('gdpr_delete_mail_sent', true);
            }
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            if (isset($requestData['gdpr_data']) && $requestData['gdpr_request'] === 'data') {
                QUI::getSession()->set('gdpr_data_mail_sent_error', true);
            }

            if (isset($requestData['gdpr_delete']) && $requestData['gdpr_request'] === 'delete') {
                QUI::getSession()->set('gdpr_delete_mail_sent_error', true);
            }
        }
    }
}
