<?php

namespace QUI\GDPR\DataRequest;

use QUI;
use QUI\FrontendUsers\AbstractRegistrar;
use QUI\FrontendUsers\Handler;
use QUI\Users\Address;

use function class_exists;
use function json_validate;

class QuiqqerUserDataProvider extends AbstractDataProvider
{
    /**
     * @inheritDoc
     */
    public function getTitle(): string
    {
        return $this->locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.title');
    }

    /**
     * @inheritDoc
     */
    public function hasUserData(): bool
    {
        return true;
    }

    /**
     * @inheritDoc
     */
    public function getPurpose(): string
    {
        return $this->locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.purpose');
    }

    /**
     * @inheritDoc
     */
    public function getRecipients(): string
    {
        return $this->locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.recipients');
    }

    /**
     * @inheritDoc
     */
    public function getStorageDuration(): string
    {
        return $this->locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.storageDuration');
    }

    /**
     * @inheritDoc
     */
    public function getOrigin(): string
    {
        // Fetch registrar
        $registrarClass = $this->user->getAttribute('quiqqer.frontendUsers.registrar');

        if (\class_exists($registrarClass) && class_exists(AbstractRegistrar::class)) {
            /** @var AbstractRegistrar $Registrar */
            $Registrar = new $registrarClass();
            $registrarTitle = $Registrar->getTitle($this->locale);
        } else {
            $registrarTitle = $this->locale->get(
                'quiqqer/gdpr',
                'QuiqqerUserDataProvider.origin.defaultRegistrarTitle'
            );
        }

        return $this->locale->get(
            'quiqqer/gdpr',
            'QuiqqerUserDataProvider.origin',
            [
                'registrarTitle' => $registrarTitle,
                'registrationDate' => $this->locale->formatDate((int)$this->user->getAttribute('regdate'))
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function getCustomText(): ?string
    {
        return null;
    }

    /**
     * @inheritDoc
     */
    public function getUserDataFields(): array
    {
        $dataFields = [];

        // Basic user data
        $data = [
            'uuid' => $this->user->getUUID(),
            'firstname' => $this->user->getAttribute('firstname') ?: null,
            'lastname' => $this->user->getAttribute('lastname') ?: null,
            'email' => $this->user->getAttribute('email') ?: null,
            'registrationDate' => date('Y-m-d H:i:s', $this->user->getAttribute('regdate')),
            'lastVisitDate' => $this->user->getAttribute('lastvisit') ?
                date('Y-m-d H:i:s', $this->user->getAttribute('lastvisit')) :
                null,
            'userAgent' => $this->user->getAttribute('user_agent') ?: null,
            'birthday' => $this->user->getAttribute('birthday') ?: null,
            'lastLoginAttempt' => $this->user->getAttribute('last_login_attempt') ?: null,
            'comments' => $this->user->getAttribute('comments') ?: null
        ];

        foreach ($data as $field => $value) {
            if (empty($value)) {
                continue;
            }

            switch ($field) {
                case 'firstname':
                case 'lastname':
                case 'email':
                case 'birthday':
                    $title = $this->locale->get('quiqqer/core', $field);
                    break;

                default:
                    $title = $this->getDataFieldTitle($field);
            }

            $dataFields[] = new UserDataField($title, $value);
        }

        // Data from addresses
        /**
         * @var Address $address
         */
        foreach ($this->user->getAddressList() as $address) {
            $dataFields[] = Utils::parseAddressToUserDataField($address, $this->locale);
        }

        // Data from login logs
        $loginLogsUserDataField = $this->parseUserDataFieldFromLoginLogs();

        if ($loginLogsUserDataField) {
            $dataFields[] = $loginLogsUserDataField;
        }

        return $dataFields;
    }

    private function parseUserDataFieldFromLoginLogs(): ?UserDataField
    {
        if (!class_exists('\QUI\LoginLogger\LoginLogger')) {
            return null;
        }

        $loginLogs = QUI\LoginLogger\LoginLogger::getLogins([
            'where' => [
                'uid' => $this->user->getUUID()
            ]
        ]);

        $loginLogEntries = [];

        foreach ($loginLogs as $row) {
            $fields = [
                new UserDataField(
                    $this->locale->get(
                        'quiqqer/gdpr',
                        'QuiqqerUserDataProvider.user_data_field.login_log.username'
                    ),
                    $row['username']
                ),
                new UserDataField(
                    $this->locale->get(
                        'quiqqer/gdpr',
                        'QuiqqerUserDataProvider.user_data_field.login_log.ip'
                    ),
                    $row['ip']
                ),
                new UserDataField(
                    $this->locale->get(
                        'quiqqer/gdpr',
                        'QuiqqerUserDataProvider.user_data_field.login_log.date'
                    ),
                    $row['date']
                ),
                new UserDataField(
                    $this->locale->get(
                        'quiqqer/gdpr',
                        'QuiqqerUserDataProvider.user_data_field.login_log.successful'
                    ),
                    $row['successful'] == '1' ?
                        $this->locale->get('quiqqer/gdpr', 'word.yes') :
                        $this->locale->get('quiqqer/gdpr', 'word.no')
                )
            ];

            if (!empty($row['user_data']) && $row['user_data'] !== '[]') {
                $fields[] = new UserDataField(
                    $this->locale->get(
                        'quiqqer/gdpr',
                        'QuiqqerUserDataProvider.user_data_field.login_log.user_data'
                    ),
                    $row['user_data']
                );
            }

            $loginLogEntries[] = new UserDataField(
                $this->locale->get(
                    'quiqqer/gdpr',
                    'QuiqqerUserDataProvider.user_data_field.login_log'
                ),
                $fields
            );
        }

        return new UserDataField(
            $this->locale->get(
                'quiqqer/gdpr',
                'QuiqqerUserDataProvider.user_data_field.login_logs'
            ),
            $loginLogEntries
        );
    }

    private function getDataFieldTitle(string $field): string
    {
        return $this->locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.user_data_field.' . $field);
    }

    /**
     * @inheritDoc
     */
    public function deleteUserData(): array
    {
        // If quiqqer/frontend-users is not installed -> just delete user
        if (!class_exists('QUI\FrontendUsers\Handler')) {
            $this->user->logout();
            $this->user->delete(QUI::getUsers()->getSystemUser());
            $deleteMode = 'account_delete';
            return [$deleteMode];
        }

        try {
            $userProfileSettings = Handler::getInstance()->getUserProfileSettings();

            switch ($userProfileSettings['userDeleteMode']) {
                case 'wipe':
                    $this->user->disable(QUI::getUsers()->getSystemUser());
                    $deleteMode = 'account_disable';
                    break;

                case 'destroy':
                    $this->user->delete();
                    $deleteMode = 'account_delete';
                    break;

                default:
                case 'delete':
                    QUI::getDataBase()->update(
                        QUI::getDBTableName('users'),
                        ['active' => -1],
                        ['uuid' => $this->user->getUUID()]
                    );
                    $deleteMode = 'account_marked_as_inactive';
                    break;
            }

            $this->user->logout();
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            throw $Exception;
        }

        return [
            $deleteMode
        ];
    }
}
