<?php

namespace QUI\GDPR\DataRequest;

use QUI;
use QUI\Locale;
use QUI\Users\Address;

use function array_filter;

class Utils
{
    /**
     * @param Address $address
     * @param Locale $locale
     * @param bool $includeUUID
     * @return UserDataField
     */
    public static function parseAddressToUserDataField(
        Address $address,
        Locale $locale,
        bool $includeUUID = true
    ): UserDataField {
        $dataFieldValues = [];

        if ($includeUUID) {
            $dataFieldValues[] = new UserDataField(
                $locale->get('quiqqer/gdpr', 'QuiqqerUserDataProvider.user_data.addressUuid'),
                $address->getUUID()
            );
        }

        // Base address data
        $dataFieldValues[] = UserDataFieldValue::BLANK_LINE;
        $fields = [
            'title',
            'salutation',
            'firstname',
            'lastname',
            'company',
//            'delivery', // currently unknown what this field does
            'street_no',
            'zip',
            'city',
            'country'
        ];

        foreach ($fields as $field) {
            $value = $address->getAttribute($field);

            if (empty($value)) {
                continue;
            }

            $title = $locale->get('quiqqer/core', $field);
            $dataFieldValues[] = new UserDataField($title, $value);
        }

        // Email addresses
        if (!empty($address->getMailList())) {
            $dataFieldValues[] = UserDataFieldValue::BLANK_LINE;
        }
        $counter = 1;

        foreach ($address->getMailList() as $email) {
            if ($counter++ === 1) {
                $counterStr = '';
            } else {
                $counterStr = $counter;
            }

            $dataFieldValues[] = new UserDataField($locale->get('quiqqer/core', 'email') . ' ' . $counterStr, $email);
        }

        // Phone numbers
        $phoneList = array_filter($address->getPhoneList(), function ($phone) {
            return !empty($phone['no']);
        });

        if (!empty($phoneList)) {
            $dataFieldValues[] = UserDataFieldValue::BLANK_LINE;
        }
        $counterByType = [];

        foreach ($phoneList as $phone) {
            $type = $phone['type'];

            if (!isset($counterByType[$type])) {
                $counterByType[$type] = 1;
            }

            $counter = $counterByType[$type]++;

            if ($counter === 1) {
                $counter = '';
            }

            $dataFieldValues[] = new UserDataField(
                $locale->get('quiqqer/core', $phone['type']) . ' ' . $counter,
                $phone['no']
            );
        }

        // Create and edit date
        $dataFieldValues[] = UserDataFieldValue::BLANK_LINE;
        $dataFieldValues[] = new UserDataField(
            $locale->get('quiqqer/core', 'c_date'),
            $address->getAttribute('c_date')
        );
        $dataFieldValues[] = new UserDataField(
            $locale->get('quiqqer/core', 'e_date'),
            $address->getAttribute('e_date')
        );

        return new UserDataField(
            $locale->get('quiqqer/core', 'address'),
            $dataFieldValues
        );
    }

    /**
     * @param UserDataField[] $userDataFields
     * @return string
     */
    public static function parseUserDataHtmlFromUserDataEntries(array $userDataFields, int $level = 1): string
    {
        $engine = QUI::getTemplateManager()->getEngine();
        $engine->assign([
            'userDataFields' => $userDataFields,
            'level' => $level
        ]);
        return $engine->fetch(dirname(__FILE__) . '/DataRequest.UserDataField.tpl');
    }

    /**
     * @param string $str
     * @return string
     */
    public static function maskString(string $str): string
    {
        $len = mb_strlen($str);
        $showCharacterCount = 3;

        if ($len < 6) {
            $showCharacterCount = 1;
        }

        return mb_substr($str, 0, $showCharacterCount) . str_repeat('*', 5) . mb_substr($str, -$showCharacterCount);
    }
}
