<?php

namespace QUI\GDPR;

use QUI;

/**
 * Class EventHandler
 *
 * @package QUI\GDPR
 */
class EventHandler
{
    /**
     * @param QUI\Template $Template
     * @param QUI\Interfaces\Projects\Site $Site
     */
    public static function onTemplateSiteFetch(QUI\Template $Template, QUI\Interfaces\Projects\Site $Site): void
    {
        $Project = $Site->getProject();

        if (CookieManager::isConsentDisabledForProject($Project)) {
            return;
        }

        $cookiesHash = '';
        $showBannerOnCookieChange = $Project->getConfig('gdpr.cookieconsent.displayAgainOnCookieChange');
        $CookieManager = CookieManager::getInstance();

        if ($showBannerOnCookieChange) {
            // Only need to query/calculate the hash if the banner should be displayed when the cookies change
            $cookiesHash = $CookieManager->getHashOfAllCookies($Project);
        }

        $Engine = QUI::getTemplateManager()->getEngine();

        $Engine->assign([
            'cookiesHash' => $cookiesHash,
            'showBannerOnCookieChange' => $showBannerOnCookieChange,
            'essentialCookieCategoryName' => CookieInterface::COOKIE_CATEGORY_ESSENTIAL,
            'availableCookieCategories' => json_encode(
                array_keys($CookieManager->getAllRegisteredCookiesGroupedByCategory($Project))
            )
        ]);


        // Loads the cookie banner if LocalStorage isn't available or cookies weren't accepted yet.
        $Template->extendHeader($Engine->fetch(__DIR__ . '/HeaderExtension.tpl'), 0);
    }

    /**
     * Listens to project config save
     *
     * @param string $projectName
     * @param array $config
     * @param array $params
     *
     * @throws QUI\Exception
     */
    public static function onProjectConfigSave(string $projectName, array $config, array $params): void
    {
        if (
            !isset($params['gdpr.cookieconsent.text'])
            || !isset($params['gdpr.cookieconsent.buttontext'])
            || !isset($params['gdpr.cookieconsent.buttontext.accept.all'])
            || !isset($params['gdpr.cookieconsent.buttontext.accept.selected'])
        ) {
            return;
        }

        try {
            $Project = QUI::getProject($projectName);
        } catch (QUI\Exception) {
            return;
        }

        $group = 'quiqqer/gdpr';

        $localeVariables = [
            'setting.text.project.' . $Project->getName() => json_decode(
                $params['gdpr.cookieconsent.text'],
                true
            ),
            'setting.buttontext.project.' . $Project->getName() => json_decode(
                $params['gdpr.cookieconsent.buttontext'],
                true
            ),
            'setting.buttontext.accept.all.project.' . $Project->getName() => json_decode(
                $params['gdpr.cookieconsent.buttontext.accept.all'],
                true
            ),
            'setting.buttontext.accept.selected.project.' . $Project->getName() => json_decode(
                $params['gdpr.cookieconsent.buttontext.accept.selected'],
                true
            ),
        ];

        foreach ($localeVariables as $localeVariableName => $localeVariableContent) {
            try {
                QUI\Translator::add($group, $localeVariableName, $group);
            } catch (QUI\Exception) {
                // Throws error if lang var already exists
            }

            try {
                QUI\Translator::update(
                    $group,
                    $localeVariableName,
                    $group,
                    $localeVariableContent
                );
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);

                continue;
            }
        }

        QUI\Translator::publish($group);
    }
}
