<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\GDPR;

use QUI;
use QUI\Config;
use QUI\Exception;

use function in_array;

/**
 * Class Checklist
 *
 * @package gdpr\src\QUI\GDPR
 */
class Checklist
{
    /**
     * Returns an array containing all items of the checklist.
     *
     * @return array
     */
    public static function getItems(): array
    {
        return [
            'contact_ssl',
            'contact_smtp',
            'contact_privacypolicy',
            'contact_usage',
            'contact_dpa',
            'newsletter_unsubscribe',
            'newsletter_optin',
            'newsletter_dataProcessingDirectory',
            'newsletter_dpa',
            'newsletter_coupling',
            'google_dpa',
            'google_analytics_ips',
            'google_analytics_gdpr',
            'google_analytics_duration',
            'tools_socialmedia',
            'tools_calendars',
            'tools_youtube',
            'tools_newsletters',
            'tools_seals',
            'tools_google_maps',
            'tools_google_recaptcha',
            'tools_google_fonts',
            'cookies_inform',
            'cookies_consent',
            'cookies_revoke',
            'privacypolicy_added',
            'privacypolicy_verified',
            'privacypolicy_accessible',
            'imprint_name_provider',
            'imprint_name_company',
            'imprint_address',
            'imprint_contact',
            'imprint_register',
            'imprint_vat',
            'imprint_supervisor',
            'imprint_chamber',
        ];
    }

    /**
     * Returns the name of the given project's section in the package's config.
     *
     * @param $projectName
     *
     * @return string
     */
    protected static function getConfigSectionName($projectName): string
    {
        return "gdpr_checklist_$projectName";
    }

    /**
     * @param $projectName
     *
     * @return string
     * @throws Exception
     */
    public static function getHtml($projectName): string
    {
        $TemplateEngine = QUI::getTemplateManager()->getEngine();

        $packageName = 'quiqqer/gdpr';
        $values = self::getConfig()->getSection(self::getConfigSectionName($projectName));

        $Locale = QUI::getLocale();

        $result = [];
        foreach (self::getItems() as $item) {
            $titleVariable = "checklist.$item.title";
            $descriptionVariable = "checklist.$item.description";

            $title = $Locale->get($packageName, $titleVariable);

            $description = '';
            if ($Locale->exists($packageName, $descriptionVariable)) {
                $description = $Locale->get($packageName, $descriptionVariable);
            }

            $result[$item] = [
                'value' => $values[$item] ?? null,
                'title' => $title,
                'description' => $description
            ];
        }

        $TemplateEngine->assign(['checklist' => $result]);

        return $TemplateEngine->fetch(dirname(__FILE__) . '/Checklist.tpl');
    }

    /**
     * Saves the given items' values for the given project in the config.
     *
     * @param $projectName
     * @param $items
     *
     * @return boolean
     */
    public static function saveItemValuesForProject($projectName, $items): bool
    {
        try {
            QUI::getProject($projectName);
        } catch (Exception) {
            return false;
        }

        $validItems = self::getItems();
        $section = self::getConfigSectionName($projectName);

        try {
            $Config = self::getConfig();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            return false;
        }

        foreach ($items as $item => $value) {
            if (!in_array($item, $validItems)) {
                continue;
            }

            $Config->set($section, $item, $value);
        }

        try {
            $Config->save();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }

        return true;
    }

    /**
     * Returns this modules' config.
     *
     * @return bool|Config
     * @throws Exception
     */
    protected static function getConfig(): Config|bool
    {
        return QUI::getPackage('quiqqer/gdpr')->getConfig();
    }
}
