<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\GDPR;

use Exception;
use QUI;

/**
 * Class CookieManager
 *
 * @package QUI\GDPR
 */
class CookieManager extends QUI\Utils\Singleton
{
    const CACHE_KEY_PREFIX_REGISTERED_COOKIES = 'quiqqer/gdpr/cookieManager/registeredCookies';
    const CACHE_KEY_PREFIX_COOKIE_HASH = 'quiqqer/gdpr/cookieManager/cookieHash';
    const PROVIDER_KEY = 'cookie';

    /**
     * @var CookieCollection|null
     */
    protected ?CookieCollection $registeredCookies = null;

    /**
     * Returns the cache key used to store all registered cookies.
     * By passing a project as the first argument the project-specific key is returned.
     *
     * @param QUI\Projects\Project|null $Project
     *
     * @return string
     */
    protected static function getCacheKeyForAllRegisteredCookies(?QUI\Projects\Project $Project = null): string
    {
        if (empty($Project)) {
            return self::CACHE_KEY_PREFIX_REGISTERED_COOKIES;
        }

        return self::CACHE_KEY_PREFIX_REGISTERED_COOKIES . '/' . $Project->getName();
    }


    protected static function getCacheKeyForCookieHash(?QUI\Projects\Project $Project = null): string
    {
        if (empty($Project)) {
            return self::CACHE_KEY_PREFIX_COOKIE_HASH;
        }

        return self::CACHE_KEY_PREFIX_COOKIE_HASH . '/' . $Project->getName();
    }


    public function getHashOfAllCookies(?QUI\Projects\Project $Project = null): string
    {
        $cacheKeyForCookieHash = self::getCacheKeyForCookieHash($Project);

        try {
            return QUI\Cache\Manager::get($cacheKeyForCookieHash);
        } catch (QUI\Cache\Exception) {
            // Cookies' hash isn't cached (yet)
            // Continue function to build the registered cookies...
        }

        $Cookies = $this->getAllRegisteredCookies($Project);

        $string = '';
        foreach ($Cookies as $Cookie) {
            /** @var CookieInterface $Cookie */
            // Stringify/Serialize all the cookies contents
            $string .= $Cookie->getName()
                . $Cookie->getCategory()
                . $Cookie->getLifetime()
                . $Cookie->getOrigin()
                . $Cookie->getPurpose();
        }

        $hash = md5($string);

        try {
            // Cache the registered cookies
            QUI\Cache\Manager::set($cacheKeyForCookieHash, $hash);
        } catch (Exception $Exception) {
            QUI\System\Log::writeDebugException($Exception);
        }

        return $hash;
    }

    /**
     * Returns all cookies that the system's packages might set.
     *
     * @param QUI\Projects\Project|null $Project - The project to get all registered cookies for.
     *
     * @return CookieCollection
     */
    public function getAllRegisteredCookies(?QUI\Projects\Project $Project = null): CookieCollection
    {
        if (isset($this->registeredCookies)) {
            return $this->registeredCookies;
        }

        try {
            $this->registeredCookies = QUI\Cache\Manager::get(self::getCacheKeyForAllRegisteredCookies($Project));

            return $this->registeredCookies;
        } catch (QUI\Cache\Exception) {
            // Registered cookies aren't cached (yet)
            // Continue function to build the registered cookies...
        }

        $packages = QUI::getPackageManager()->getInstalled();

        $cookieProviders = [];

        /* @var QUI\Package\Package $Package */
        foreach ($packages as $packageData) {
            try {
                $Package = QUI::getPackage($packageData['name']);
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
                continue;
            }

            // Get all providers of this package
            $packagesCookieProviders = $Package->getProvider(self::PROVIDER_KEY);

            // Check if the specified classes really exist
            foreach ($packagesCookieProviders as $cookieProvider) {
                if (!class_exists($cookieProvider)) {
                    continue 2;
                }
            }

            // Add the packages dashboard providers to all providers
            $cookieProviders = array_merge($cookieProviders, $packagesCookieProviders);
        }

        $Cookies = new CookieCollection();

        // initialize the instances
        foreach ($cookieProviders as $cookieProvider) {
            try {
                $Provider = new $cookieProvider();

                // Check if the given provider is really a CookieProvider
                if (!($Provider instanceof CookieProviderInterface)) {
                    unset($Provider);
                    continue;
                }

                $Cookies->merge($Provider->getCookies());
            } catch (Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        if (!empty($Project)) {
            $Cookies->merge($this->getManualCookies($Project));
        }

        try {
            // Cache the registered cookies
            QUI\Cache\Manager::set(self::getCacheKeyForAllRegisteredCookies($Project), $Cookies);
        } catch (Exception $Exception) {
            QUI\System\Log::writeDebugException($Exception);
        }

        $this->registeredCookies = $Cookies;

        return $this->registeredCookies;
    }

    /**
     * Returns all registered cookies grouped by categories
     *
     * @param QUI\Projects\Project|null $Project - The project to get the cookies for.
     *
     * @return CookieCollection[]
     */
    public function getAllRegisteredCookiesGroupedByCategory(?QUI\Projects\Project $Project = null): array
    {
        $Cookies = $this->getAllRegisteredCookies($Project);

        /** @var CookieCollection[] $result */
        $result = [];

        foreach ($Cookies as $Cookie) {
            /** @var CookieInterface $Cookie */

            $category = $Cookie->getCategory();
            if (isset($result[$category])) {
                $result[$category]->append($Cookie);
            } else {
                $result[$category] = new CookieCollection([$Cookie]);
            }
        }

        return $result;
    }

    /**
     * Returns the cookies of a given category, that are registered in the system.
     * The passed category should be one of the "COOKIE_CATEGORY_"-prefixed constants of CookieInterface.
     *
     * @param $category
     * @param QUI\Projects\Project|null $Project - The project to get the registered cookies for
     *
     * @return CookieCollection
     */
    public function getRegisteredCookiesForCategory($category, ?QUI\Projects\Project $Project = null): CookieCollection
    {
        $cookies = $this->getAllRegisteredCookies($Project);

        return $cookies->filter(function ($Cookie) use ($category) {
            /** @var CookieInterface $Cookie */
            return $Cookie->getCategory() == $category;
        });
    }


    /**
     * Returns the name of the table used to store the given project's manually added cookies.
     *
     * @param QUI\Projects\Project $Project
     *
     * @return string
     */
    public static function getManualCookiesTableName(QUI\Projects\Project $Project): string
    {
        return QUI::getDBProjectTableName('cookies', $Project);
    }


    /**
     * Edits a manually added cookie with the given data.
     * If no or an unused cookie-id is given, a new cookie with the given data is added.
     *
     * @param                      $data
     * @param QUI\Projects\Project $Project
     *
     * @return boolean
     */
    public static function editManualCookie($data, QUI\Projects\Project $Project): bool
    {
        try {
            QUI::getDataBase()->replace(
                self::getManualCookiesTableName($Project),
                $data
            );
        } catch (QUI\Database\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            return false;
        }

        // Cache has to be cleared in order to update the list of available cookies
        self::clearRegisteredCookiesCache($Project);
        self::clearCookieHashCache($Project);

        return true;
    }


    /**
     * Clears the registered cookies cache for the given project.
     *
     * @param QUI\Projects\Project $Project
     */
    public static function clearRegisteredCookiesCache(QUI\Projects\Project $Project): void
    {
        QUI\Cache\Manager::clear(CookieManager::getCacheKeyForAllRegisteredCookies($Project));
    }


    public static function clearCookieHashCache(QUI\Projects\Project $Project): void
    {
        QUI\Cache\Manager::clear(CookieManager::getCacheKeyForCookieHash($Project));
    }

    /**
     * Deletes the manually added cookie with the given ID from the given project.
     *
     * @param                      $id
     * @param QUI\Projects\Project $Project
     *
     * @return boolean
     */
    public static function deleteManualCookie($id, QUI\Projects\Project $Project): bool
    {
        try {
            QUI::getDataBase()->delete(
                self::getManualCookiesTableName($Project),
                ['id' => $id]
            );
        } catch (QUI\Database\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return false;
        }

        // Cache has to be cleared in order to update the list of available cookies
        self::clearRegisteredCookiesCache($Project);
        self::clearCookieHashCache($Project);

        return true;
    }


    /**
     * Returns a collection of all manual cookies in the given project.
     *
     * @param QUI\Projects\Project $Project
     *
     * @return CookieCollection
     */
    protected static function getManualCookies(QUI\Projects\Project $Project): CookieCollection
    {
        $cookieTable = self::getManualCookiesTableName($Project);

        $CookieCollection = new CookieCollection();

        try {
            $result = QUI::getDataBase()->fetch([
                'from' => $cookieTable
            ]);
        } catch (QUI\Database\Exception $Exception) {
            // Couldn't query database, this shouldn't happen.
            QUI\System\Log::writeException($Exception);

            return $CookieCollection;
        }

        foreach ($result as $cookieData) {
            $CookieCollection->append(
                new QUI\GDPR\Cookies\ManualCookie(
                    $cookieData['id'],
                    $cookieData['name'],
                    $cookieData['origin'],
                    $cookieData['purpose'],
                    $cookieData['lifetime'],
                    $cookieData['category']
                )
            );
        }

        return $CookieCollection;
    }

    public static function getPreselectedCookieCategories(QUI\Projects\Project $Project): array
    {
        $preselectedCategories = [];

        if ($Project->getConfig('gdpr.cookieconsent.preselects.preferences')) {
            $preselectedCategories[] = CookieInterface::COOKIE_CATEGORY_PREFERENCES;
        }

        if ($Project->getConfig('gdpr.cookieconsent.preselects.statistics')) {
            $preselectedCategories[] = CookieInterface::COOKIE_CATEGORY_STATISTICS;
        }

        if ($Project->getConfig('gdpr.cookieconsent.preselects.marketing')) {
            $preselectedCategories[] = CookieInterface::COOKIE_CATEGORY_MARKETING;
        }

        return $preselectedCategories;
    }

    /**
     * Returns if the consent is disabled for the given project.
     *
     * @param QUI\Projects\Project $Project
     *
     * @return bool
     */
    public static function isConsentDisabledForProject(QUI\Projects\Project $Project): bool
    {
        return $Project->getConfig('gdpr.cookieconsent.disabled');
    }
}
