<?php

namespace QUI\GDPR\DataRequest;

use Dom\Document;
use QUI;
use QUI\GDPR\DataRequest\Verification\UserDataDeleteConfirm;
use QUI\Interfaces\Users\User as UserInterface;
use QUI\Verification\VerificationFactory;

use function class_exists;
use function md5;
use function serialize;
use function usort;

/**
 * Class Handler
 *
 * GDPR data request handler.
 */
class Handler
{
    /**
     * Get complete data request for a $User with basic data and
     *
     * @param UserInterface $User
     * @return array
     *
     * @throws QUI\Exception
     */
    public static function getDataRequestData(UserInterface $User): array
    {
        // Short address
        $shortAddress = '';

        if (QUI::getPackageManager()->isInstalled('quiqqer/erp') && class_exists(QUI\ERP\Defaults::class)) {
            $shortAddress = QUI\ERP\Defaults::getShortAddress();
        }

        $data = [
            'date' => $User->getLocale()->formatDate(\time()) . '; ' . \date('H:i'),
            'User' => $User,
            'dataProtectionOfficer' => self::getDataProtectionOfficer(),
            'regulatoryAuthority' => self::getRegulatoryAuthority(),
            'providers' => [],
            'shortAddress' => $shortAddress
        ];

        $dataProviders = self::getGdprDataProviders($User);

        // Sort
        usort($dataProviders, function (DataProviderInterface $dataProvider) {
            if ($dataProvider instanceof QuiqqerUserDataProvider) {
                return -1;
            }

            return 1;
        });

        foreach ($dataProviders as $Provider) {
            if (!$Provider->hasUserData()) {
                continue;
            }

            $userDataFields = $Provider->getUserDataFields();

            if (empty($userDataFields)) {
                continue;
            }

            // Remove duplicates
            $hashes = [];

            foreach ($userDataFields as $k => $userDataField) {
                $hash = md5($userDataField->title . serialize($userDataField->data));

                if (isset($hashes[$hash])) {
                    unset($userDataFields[$k]);
                    continue;
                }

                $hashes[$hash] = true;
            }

            $providerData = [
                'title' => $Provider->getTitle(),
                'purpose' => $Provider->getPurpose(),
                'recipients' => $Provider->getRecipients(),
                'origin' => $Provider->getOrigin(),
                'storageDuration' => $Provider->getStorageDuration(),
                'customText' => $Provider->getCustomText(),
                'userDataFields' => $userDataFields
            ];

            $data['providers'][] = $providerData;
        }

        return $data;
    }

    /**
     * @param UserInterface $User
     * @return string
     *
     * @throws QUI\Exception
     */
    public static function getDataRequestPdf(UserInterface $User): string
    {
        if (!QUI::getPackageManager()->hasLicense('quiqqer/gdpr')) {
            throw new QUI\Exception('Package "quiqqer/gdpr" is not licensed.');
        }

        if (!class_exists(QUI\HtmlToPdf\Document::class)) {
            throw new QUI\Exception('Package "quiqqer/htmltopdf" is not installed.');
        }

        $Document = new QUI\HtmlToPdf\Document();
        $Locale = $User->getLocale();

//        $Document->setAttribute('foldingMarks', true);
        $Document->setAttribute('disableSmartShrinking', true);
        $Document->setAttribute('headerSpacing', 10);
        $Document->setAttribute('marginTop', 50);
        $Document->setAttribute('marginBottom', 15);
        $Document->setAttribute('marginLeft', 0);
        $Document->setAttribute('marginRight', 0);
        $Document->setAttribute('showPageNumbers', true);

        $Engine = QUI::getTemplateManager()->getEngine();

        $requestData = self::getDataRequestData($User);

        // Regulatory authority
        $regAuthority = $requestData['regulatoryAuthority'];
        $regAuthorityAddressLines = [];

        $regAuthorityAttributes = [
            'name',
            'street',
            'city',
            'po_box',
            'email',
            'phone'
        ];

        foreach ($regAuthorityAttributes as $attribute) {
            if (empty($regAuthority[$attribute])) {
                continue;
            }

            $val = $regAuthority[$attribute];

            switch ($attribute) {
                case 'email':
                    $prefix = $Locale->get('quiqqer/gdpr', 'DataRequest.regAuthority.prefix.email');
                    $regAuthorityAddressLines[] = ''; // empty line
                    $regAuthorityAddressLines[] = $prefix . ' ' . $val;
                    break;

                case 'phone':
                    $prefix = $Locale->get('quiqqer/gdpr', 'DataRequest.regAuthority.prefix.phone');
                    $regAuthorityAddressLines[] = $prefix . ' ' . $val;
                    break;

                default:
                    $regAuthorityAddressLines[] = $val;
            }
        }

        $requestData['regulatoryAuthorityAddressLines'] = \nl2br(
            \implode(
                "\n",
                $regAuthorityAddressLines
            )
        );

        QUI::getLocale()->setTemporaryCurrent($Locale->getCurrent());

        $Engine->assign($requestData);

        $Document->setHeaderHTML(
            $Engine->fetch(\dirname(__FILE__) . '/DataRequest.Header.tpl')
        );

        $Document->setContentHTML(
            $Engine->fetch(\dirname(__FILE__) . '/DataRequest.tpl')
        );

        $Document->setFooterCSS(
            $Engine->fetch(\dirname(__FILE__) . '/DataRequest.Footer.css')
        );

        QUI::getLocale()->resetCurrent();

        return $Document->createPDF();
    }

    /**
     * Get data protection officer contact data.
     *
     * @return array
     * @throws QUI\Exception
     */
    protected static function getDataProtectionOfficer(): array
    {
        $Conf = QUI::getPackage('quiqqer/gdpr')->getConfig();
        return $Conf->getSection('data_protection_officer');
    }

    /**
     * Get regulatory authority contact data.
     *
     * @return array
     * @throws QUI\Exception
     */
    protected static function getRegulatoryAuthority(): array
    {
        $Conf = QUI::getPackage('quiqqer/gdpr')->getConfig();
        return $Conf->getSection('regulatory_authority');
    }

    /**
     * Get all available GDPR Data provider classes
     *
     * @param UserInterface $User
     * @return DataProviderInterface[] - Provider classes
     */
    protected static function getGdprDataProviders(UserInterface $User): array
    {
        $packages = QUI::getPackageManager()->getInstalled();
        $providers = [];

        foreach ($packages as $installedPackage) {
            try {
                $Package = QUI::getPackage($installedPackage['name']);

                if (!$Package->isQuiqqerPackage()) {
                    continue;
                }

                $packageProvider = $Package->getProvider();

                if (empty($packageProvider['gdprDataProvider'])) {
                    continue;
                }

                foreach ($packageProvider['gdprDataProvider'] as $class) {
                    if (!\class_exists($class)) {
                        continue;
                    }

                    if (!\is_a($class, DataProviderInterface::class, true)) {
                        continue;
                    }

                    /** @var DataProviderInterface $instance */
                    $instance = new $class();
                    $instance->setUser($User);
                    $instance->setLocale($User->getLocale());
                    $providers[] = $instance;
                }
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $providers;
    }

    /**
     * Delete all GDPR-relevant user data.
     *
     * @param UserInterface $User
     * @return void
     */
    public static function deleteUserData(UserInterface $User): void
    {
        $deletedData = [];

        foreach (self::getGdprDataProviders($User) as $Provider) {
            $deletedData[] = [
                'area' => $Provider->getTitle(),
                'deletedFields' => $Provider->deleteUserData()
            ];
        }

        // Send confirmation to user
        $email = $User->getAttribute('email');
        $Locale = $User->getLocale();

        if (!empty($email)) {
            try {
                $Mailer = QUI::getMailManager()->getMailer();

                $Mailer->setSubject(
                    $Locale->get(
                        'quiqqer/gdpr',
                        'DataRequest.delete_confirm_mail.subject',
                        [
                            'userName' => $User->getName()
                        ]
                    )
                );

                $Mailer->setBody(
                    $Locale->get(
                        'quiqqer/gdpr',
                        'DataRequest.delete_confirm_mail.body',
                        [
                            'companyName' => self::getCompanyName()
                        ]
                    )
                );

                $Mailer->send();
            } catch (\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        // Log deleted data
        QUI\System\Log::write(
            'User (#' . $User->getId() . ') GDPR data deleted.',
            QUI\System\Log::LEVEL_INFO,
            $deletedData,
            'gdpr_deletion_requests',
            true
        );
//        $User->delete(); // @todo muss noch geklärt werden
    }

    /**
     * Send verification mail for user account deletion
     *
     * @param UserInterface $User
     * @return void
     *
     * @throws QUI\Verification\Exception
     * @throws QUI\Exception
     * @throws \Exception
     */
    public static function sendDeleteUserConfirmationMail(UserInterface $User)
    {
        $verificationFactory = new VerificationFactory();
        $deleteUserVerification = $verificationFactory->createLinkVerification(
            self::getDeleteUserConfirmationVerificationIdentifier($User),
            new UserDataDeleteConfirm(),
            [
                'userUuid' => $User->getUUID()
            ],
            true
        );

        $confirmLink = $deleteUserVerification->getVerificationUrl();
        $Locale = $User->getLocale();

        $Mailer = QUI::getMailManager()->getMailer();

        $Mailer->addRecipient($User->getAttribute('email'));

        $Mailer->setSubject(
            $Locale->get(
                'quiqqer/gdpr',
                'DataRequest.delete_verification_mail.subject',
                [
                    'userName' => $User->getName()
                ]
            )
        );

        $Mailer->setBody(
            $Locale->get(
                'quiqqer/gdpr',
                'DataRequest.delete_verification_mail.body',
                [
                    'companyName' => self::getCompanyName(),
                    'date' => $Locale->formatDate(time()),
                    'confirmLink' => $confirmLink,
                    'userName' => $User->getName()
                ]
            )
        );

        $Mailer->send();
    }

    /**
     * Get identifier used for user account deletion verification.
     *
     * @param UserInterface $user
     * @return string
     */
    public static function getDeleteUserConfirmationVerificationIdentifier(UserInterface $user): string
    {
        return 'quiqqer-gdpr-user-delete-' . $user->getUUID();
    }

    /**
     * Send mail containing the data request PDF file
     *
     * @param UserInterface $User
     * @return void
     *
     * @throws QUI\Exception
     * @throws \Exception
     */
    public static function sendDataRequestPdfMail(UserInterface $User)
    {
        $pdfFile = self::getDataRequestPdf($User);
        $Locale = $User->getLocale();

        // Rename file
        $newPdfFileName = $Locale->get(
            'quiqqer/gdpr',
            'DataRequest.filename',
            [
                'userId' => $User->getId(),
                'date' => $Locale->formatDate(\time())
            ]
        );

        $newPdfFile = \str_replace(
            \basename($pdfFile),
            $newPdfFileName,
            $pdfFile
        );

        \rename($pdfFile, $newPdfFile);

        $Mailer = QUI::getMailManager()->getMailer();

        $Mailer->addRecipient($User->getAttribute('email'));
        $Mailer->setSubject($Locale->get('quiqqer/gdpr', 'DataRequest.information_mail.subject'));

        $Mailer->setBody(
            $Locale->get(
                'quiqqer/gdpr',
                'DataRequest.information_mail.body',
                [
                    'userName' => $User->getName(),
                    'companyName' => self::getCompanyName(),
                    'date' => $Locale->formatDate(time())
                ]
            )
        );

        $Mailer->addAttachment($newPdfFile);
        $Mailer->send();

        \unlink($newPdfFile);
    }

    /**
     * Return the company name of the quiqqer system
     *
     * @return string
     */
    protected static function getCompanyName(): string
    {
        if (!QUI::getPackageManager()->isInstalled('quiqqer/erp')) {
            return '';
        }

        try {
            $Conf = QUI::getPackage('quiqqer/erp')->getConfig();
            $company = $Conf->get('company', 'name');
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return '';
        }

        if (empty($company)) {
            return '';
        }

        return $company;
    }
}
