<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\LoginLogger;

use QUI;
use QUI\Exception;
use QUI\Users\User;

/**
 * Class EventHandler
 *
 * @package QUI\LoginLogger
 */
class EventListeners
{
    /**
     * Called when a user successfully logged in
     *
     * @param User $User
     */
    public static function onUserLogin(User $User): void
    {
        $isBackendLogin = defined('QUIQQER_BACKEND') && QUIQQER_BACKEND === true;

        LoginLogger::logLoginForCurrentRequest(
            $User->getId(),
            $User->getUsername(),
            $isBackendLogin ? LoginLogger::LOGIN_TYPE_BACKEND : LoginLogger::LOGIN_TYPE_FRONTEND,
        );
    }

    /**
     * Called when a user successfully logged in via CLI
     *
     * @param User $User
     */
    public static function onUserCliLogin(User $User): void
    {
        LoginLogger::logLogin(
            $User->getId(),
            $User->getUsername(),
            true,
            LoginLogger::LOGIN_TYPE_CLI
        );
    }

    /**
     * Called when a login from CLI failed.
     *
     * @param string $username
     */
    public static function onUserCliLoginError(string $username): void
    {
        $QuiUsers = QUI::getUsers();

        try {
            $userId = $QuiUsers->getUserByName($username)->getId();
        } catch (Exception $Exception) {
            $userId = $QuiUsers->getNobody()->getId();
        }

        LoginLogger::logLogin(
            $userId,
            $username,
            false,
            LoginLogger::LOGIN_TYPE_CLI
        );
    }


    /**
     * Called when a user failed to login
     *
     * @param int $userId
     * @param Exception $Exception
     */
    public static function onUserLoginError(int $userId, Exception $Exception): void
    {
        $QuiUsers = QUI::getUsers();

        try {
            $User = $QuiUsers->get($userId);
        } catch (Exception $Exception) {
            $User = $QuiUsers->getNobody();
        }

        // Invalid user specified on login
        if ($QuiUsers->isNobodyUser($User)) {
            // Get username from the login request
            $username = Utils::getUsernameFromLoginRequest();

            // Couldn't determine the username
            if (empty($username)) {
                return;
            }
        } else {
            $username = $User->getUsername();
        }

        $isBackendLogin = defined('QUIQQER_BACKEND') && QUIQQER_BACKEND === true;
        LoginLogger::logLoginForCurrentRequest(
            $userId,
            $username,
            $isBackendLogin ? LoginLogger::LOGIN_TYPE_BACKEND : LoginLogger::LOGIN_TYPE_FRONTEND,
        );
    }
}
