<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\LoginLogger;

use QUI;
use QUI\Exception;
use QUI\System\Console\Tools\MigrationV2;
use QUI\Users\User;

use function is_numeric;

/**
 * Class EventHandler
 *
 * @package QUI\LoginLogger
 */
class EventListeners
{
    /**
     * Called when a user successfully logged in
     *
     * @param User $User
     */
    public static function onUserLogin(User $User): void
    {
        $isBackendLogin = defined('QUIQQER_BACKEND') && QUIQQER_BACKEND === true;

        LoginLogger::logLoginForCurrentRequest(
            $User->getUUID(),
            $User->getUsername(),
            $isBackendLogin ? LoginLogger::LOGIN_TYPE_BACKEND : LoginLogger::LOGIN_TYPE_FRONTEND,
        );
    }

    /**
     * Called when a user successfully logged in via CLI
     *
     * @param User $User
     */
    public static function onUserCliLogin(User $User): void
    {
        LoginLogger::logLogin(
            $User->getUUID(),
            $User->getUsername(),
            true,
            LoginLogger::LOGIN_TYPE_CLI
        );
    }

    /**
     * Called when a login from CLI failed.
     *
     * @param string $username
     */
    public static function onUserCliLoginError(string $username): void
    {
        $QuiUsers = QUI::getUsers();

        try {
            $userId = $QuiUsers->getUserByName($username)->getUUID();
        } catch (Exception) {
            $userId = $QuiUsers->getNobody()->getUUID();
        }

        LoginLogger::logLogin(
            $userId,
            $username,
            false,
            LoginLogger::LOGIN_TYPE_CLI
        );
    }


    /**
     * Called when a user failed to log in
     *
     * @param int|string $userId
     * @param Exception $Exception
     */
    public static function onUserLoginError(int | string $userId, Exception $Exception): void
    {
        $QuiUsers = QUI::getUsers();

        try {
            $User = $QuiUsers->get($userId);
        } catch (Exception) {
            $User = $QuiUsers->getNobody();
        }

        // Invalid user specified on login
        if ($QuiUsers->isNobodyUser($User)) {
            // Get username from the login request
            $username = Utils::getUsernameFromLoginRequest();

            // Couldn't determine the username
            if (empty($username)) {
                return;
            }
        } else {
            $username = $User->getUsername();
        }

        $isBackendLogin = defined('QUIQQER_BACKEND') && QUIQQER_BACKEND === true;
        LoginLogger::logLoginForCurrentRequest(
            $userId,
            $username,
            $isBackendLogin ? LoginLogger::LOGIN_TYPE_BACKEND : LoginLogger::LOGIN_TYPE_FRONTEND,
        );
    }

    public static function onQuiqqerMigrationV2(MigrationV2 $Console): void
    {
        $Console->writeLn('- Migrate login logs');
        $table = QUI::getDBTableName('login_log');

        $result = QUI::getDataBase()->fetch([
            'from' => $table
        ]);

        foreach ($result as $entry) {
            $uid = $entry['uid'];

            if (empty($uid)) {
                continue;
            }

            if (!is_numeric($uid)) {
                continue;
            }

            try {
                QUI::getDataBase()->update(
                    $table,
                    ['uid' => QUI::getUsers()->get($uid)->getUUID()],
                    ['uid' => $uid]
                );
            } catch (QUI\Exception) {
            }
        }
    }
}
