<?php

/**
 * @author PCSG (Jan Wennrich)
 */

namespace QUI\LoginLogger;

use DateTime;
use QUI;
use QUI\Database\Exception;
use QUI\System\Log;

/**
 * Class LoginLogger
 *
 * @package QUI\LoginLogger
 */
class LoginLogger
{
    /**
     * Login from backend
     *
     * @var string
     */
    const LOGIN_TYPE_BACKEND = 'backend';

    /**
     * Login from CLI
     *
     * @var string
     */
    const LOGIN_TYPE_CLI = 'cli';

    /**
     * Login from frontend
     *
     * @var string
     */
    const LOGIN_TYPE_FRONTEND = 'frontend';

    /**
     * Returns the logins from the database.
     * The first parameter is used as the SQL-query parameters (see Database-class on how to use them).
     * The table for the query is set by default and will overwrite any given table name.
     *
     * @param array $sqlQueryParams
     *
     * @return array
     *
     * @throws Exception
     */
    public static function getLogins(array $sqlQueryParams = []): array
    {
        $sqlQueryParams['from'] = static::getTableName();

        return QUI::getDataBase()->fetch($sqlQueryParams);
    }

    /**
     * Returns the name of the table where the logins are logged.
     *
     * @return string
     */
    public static function getTableName(): string
    {
        return QUI::getDBTableName('login_log');
    }

    /**
     * Removes all logged logins that are older than the given amount of days.
     *
     * @param int $days
     *
     * @throws Exception
     */
    public static function deleteLoginsOlderThanDays(int $days): void
    {
        $minimumDate = new DateTime("now - $days days");
        $minimumDate = $minimumDate->format('Y-m-d H:i:s');

        static::deleteLogins([
            'date' => [
                'type' => '<=',
                'value' => $minimumDate
            ]
        ]);
    }

    /**
     * Deletes logins from the database.
     * The first parameter is used as the SQL-WHERE-query parameters (see Database-class delete-method on how to use it).
     * Attention: If no parameter is passed all entries will be deleted.
     *
     * @param array $sqlQueryWhereParam
     *
     * @throws Exception
     */
    public static function deleteLogins(array $sqlQueryWhereParam = []): void
    {
        QUI::getDataBase()->delete(static::getTableName(), $sqlQueryWhereParam);
    }

    /**
     * Logs a login for the current session.
     *
     * @param int|string $userId - ID of the user logging in
     * @param string $usedUsername - The username used to log in
     * @param string $type - How did the login occur (see LOGIN_TYPE constants)
     */
    public static function logLoginForCurrentRequest(int | string $userId, string $usedUsername, string $type): void
    {
        $isSuccessful = false;

        if (QUI::getUsers()->isAuth(QUI::getUserBySession())) {
            $isSuccessful = true;
        }

        static::logLogin(
            $userId,
            $usedUsername,
            $isSuccessful,
            $type,
            QUI::getRequest()->getClientIp()
        );
    }

    /**
     * Logs the login attempt to the database.
     *
     * @param int|string $userId
     * @param string $username
     * @param bool $isSuccessful
     * @param string $type
     * @param string|null $ip
     * @param array $userData (optional) Empty array by default.
     * @param DateTime|null $Date (optional) Current date by default.
     */
    public static function logLogin(
        int | string $userId,
        string $username,
        bool $isSuccessful,
        string $type,
        ?string $ip = '',
        array $userData = [],
        null | DateTime $Date = null
    ): void {
        if (strlen($userId) > 50) {
            $userId = substr($userId, 0, 47) . '...';
        }

        if (strlen($username) > 50) {
            $username = substr($username, 0, 47) . '...';
        }

        if ($ip !== null && strlen($ip) > 45) {
            $ip = substr($ip, 0, 47) . '...';
        }

        if (is_null($Date)) {
            try {
                $Date = new DateTime();
            } catch (\Exception $Exception) {
                // We should never end up here, but just in case...
                Log::writeException($Exception);

                return;
            }
        }

        try {
            QUI::getDataBase()->insert(
                static::getTableName(),
                [
                    'uid' => $userId,
                    'username' => $username,
                    'ip' => $ip,
                    'date' => $Date->format('Y-m-d H:i:s'),
                    'successful' => $isSuccessful ? 1 : 0,
                    'user_data' => json_encode($userData),
                    'type' => $type
                ]
            );
        } catch (Exception $Exception) {
            Log::writeException($Exception);
            return;
        }
    }
}
