<?php

namespace QUI\ERP\Order\SimpleCheckout\Steps;

use QUI;
use QUI\ERP\Address;
use QUI\ERP\Order\SimpleCheckout\Checkout;
use QUI\ERP\Order\SimpleCheckout\CheckoutStepInterface;
use QUI\Exception;

use function dirname;

/**
 * Class CheckoutDelivery
 *
 * @package Your\Package\Namespace
 */
class CheckoutBillingAddress extends QUI\Control implements CheckoutStepInterface
{
    protected Checkout $Checkout;

    /**
     * Constructor method for the SimpleCheckoutDelivery class.
     *
     * @param Checkout $Checkout
     * @param mixed[] $attributes
     * @return void
     */
    public function __construct(Checkout $Checkout, array $attributes = [])
    {
        $this->Checkout = $Checkout;

        parent::__construct($attributes);

        $this->addCSSFile(dirname(__FILE__) . '/CheckoutBillingAddress.css');
        $this->addCSSClass('quiqqer-simple-checkout-billing quiqqer-simple-checkout-step');

        $this->setJavaScriptControl(
            'package/quiqqer/order-simple-checkout/bin/frontend/controls/SimpleCheckoutBillingAddress'
        );
    }

    /**
     * Returns the HTML body content for the checkout delivery step.
     *
     * @return string The HTML body content for the checkout delivery step.
     * @throws Exception
     * @throws QUI\ERP\Order\Exception
     */
    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();
        $User = QUI::getUserBySession();

        $Engine->assign([
            'User' => $User,
            'Address' => $this->getDeliveryAddress()
        ]);

        return $Engine->fetch(dirname(__FILE__) . '/CheckoutBillingAddress.html');
    }

    /**
     * Retrieves the invoice address for the current user.
     *
     * @return null|Address
     * @throws Exception
     * @throws QUI\ERP\Order\Exception
     */
    protected function getDeliveryAddress(): ?QUI\ERP\Address
    {
        return $this->Checkout->getOrder()?->getDeliveryAddress();
    }

    /**
     * Validates the invoice address of the current order.
     *
     * @throws QUI\ERP\Order\Exception|Exception
     */
    public function validate(): void
    {
        $Address = $this->Checkout->getOrder()?->getInvoiceAddress();

        if ($Address instanceof QUI\Users\Address) {
            QUI\ERP\Order\Controls\OrderProcess\CustomerData::validateAddress($Address);
        } else {
            throw new QUI\ERP\Order\Exception([
                'quiqqer/order',
                'exception.missing.address.field',
                ['field' => QUI::getLocale()->get('quiqqer/order', 'firstname')]
            ]);
        }
    }
}
