<?php

/**
 * Get data for new payment transaction
 *
 * @param string|int $entityId - Payment receiver entity ID
 * @param string $entityType - Payment receiver entity type
 * @return array
 *
 * @throws Exception
 */

use QUI\ERP\Accounting\Payments\Transactions\Exception;
use QUI\ERP\Accounting\Payments\Transactions\IncomingPayments\Handler;
use QUI\ERP\Constants as ERPConstants;
use QUI\Utils\Security\Orthos;

QUI::$Ajax->registerFunction(
    'package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_getData',
    function ($entityId, $entityType) {
        $entityType = Orthos::clear($entityType);
        $entityId = Orthos::clear($entityId);

        $Provider = Handler::getPaymentReceiver($entityType, $entityId);

        if (empty($Provider)) {
            QUI\System\Log::addError(
                'No paymentReceiver provider found for entity type: ' . $entityType
            );

            throw new Exception([
                'quiqqer/payment-transactions',
                'exception.ajax.backend.IncomingPayments.getData.no_provider'
            ]);
        }

        $Locale = QUI::getLocale();
        $Currency = $Provider->getCurrency();
        $canBookPayment = true;
        $noBookReason = '';

        if ($Provider->getPaymentStatus() == ERPConstants::PAYMENT_STATUS_DEBIT) {
            $canBookPayment = false;
            $noBookReason = QUI::getLocale()->get(
                'quiqqer/payment-transactions',
                'no_book.reason.debit'
            );
        }

        $data = [
            'debtorNo' => $Provider->getDebtorNo(),
            'addressSalutation' => '',
            'addressName' => '',
            'addressStreet' => '',
            'addressCity' => '',
            'addressCountry' => '',
            'documentType' => $Provider::getTypeTitle($Locale),
            'documentNo' => $Provider->getDocumentNo(),
            'date' => $Locale->formatDate($Provider->getDate()->getTimestamp()),
            'dueDate' => $Provider->getDueDate() ?
                $Locale->formatDate($Provider->getDueDate()->getTimestamp()) : '',
            'amountTotal' => $Currency->format($Provider->getAmountTotal(), $Locale),
            'amountPaid' => $Currency->format($Provider->getAmountPaid(), $Locale),
            'amountOpen' => $Currency->format($Provider->getAmountOpen(), $Locale),
            'amountOpenRaw' => $Provider->getAmountOpen(),
            'paymentId' => $Provider->getPaymentMethod() ? $Provider->getPaymentMethod()->getId() : false,
            'canBookPayment' => $canBookPayment,
            'noBookReason' => $noBookReason
        ];

        // Address
        $Address = $Provider->getDebtorAddress();

        if ($Address) {
            $data['addressSalutation'] = $Address->getAttribute('salutation') ?: '';
            $data['addressName'] = $Address->getName() ?: '';
            $data['addressStreet'] = $Address->getAttribute('street_no') ?: '';
            $data['addressCity'] = $Address->getAttribute('city') ?: '';
            $data['addressCountry'] = $Address->getAttribute('country') ?
                $Address->getCountry()->getName($Locale) : '';
        }

        return $data;
    },
    ['entityId', 'entityType'],
    'Permission::checkAdminUser'
);
