/**
 * Add a payment to an ERP entity (e.g. invoice, offer...)
 *
 * @module package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment
 *
 * @event onLoad [self, canBookPayment]
 * @event onSubmit [TransactionData, self]
 * @event onSubmitModeChange [mode, this] - Fires if the user switches between new transaction / assigning an existing transaction
 */
define('package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment', [

    'qui/controls/Control',
    'qui/controls/loader/Loader',
    'qui/controls/buttons/Button',
    'qui/controls/windows/Confirm',

    'qui/utils/Form',

    'Permissions',
    'Mustache',
    'Locale',
    'Ajax',

    'package/quiqqer/payment-transactions/bin/backend/Transactions',
    'package/quiqqer/payment-transactions/bin/backend/controls/TransactionsGrid',
    'package/quiqqer/payments/bin/backend/Payments',

    'text!package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment.html',
    'text!package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment.TransactionsGrid.html',
    'text!package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment.ExistingTransaction.html',
    'text!package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment.LinkedElements.html',
    'css!package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment.css'

], function (QUIControl, QUILoader, QUIButton, QUIConfirm, QUIFormUtils, Permissions, Mustache, QUILocale,
             QUIAjax, Transactions, TransactionsGrid, Payments,
             template, templateTransactionsGrid, templateExistingTransaction, templateLinkedElements
) {
    'use strict';

    const lg = 'quiqqer/payment-transactions';
    const cssClassHidden = 'quiqqer-payment-transactions-AddPayment--hidden';

    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment',

        Binds: [
            '$onInject',
            '$getData',
            '$onClickAddExistingTransaction'
        ],

        options: {
            hash: false,
            paymentId: false,   // ID of the payment that is preselected when adding new transactions
            globalProcessId: false, // ID of the process
            ignoreEntities: []
        },

        initialize: function (options) {
            this.parent(options);

            this.$Form = null;
            this.Loader = new QUILoader();
            this.$ExistingTransaction = null;

            this.addEvents({
                onInject: this.$onInject
            });
        },

        /**
         * Create the DomNode Element
         *
         * @return {Element|null}
         */
        create: function () {
            this.$Elm = new Element('div', {
                'class': 'quiqqer-payment-transactions-add'
            });

            this.Loader.inject(this.$Elm);

            return this.$Elm;
        },

        /**
         * event: on inject
         */
        $onInject: function () {
            Promise.all([
                Payments.getPayments(),
                this.$getData(),
                Permissions.hasPermission('quiqqer.paymentTransactions.create')
            ]).then((result) => {
                let title, payment;

                const payments = result[0],
                    Data = result[1],
                    permission = result[2],
                    current = QUILocale.getCurrent();

                if (!permission) {
                    this.$Elm.set(
                        'html',
                        '<p>' + QUILocale.get(lg, 'controls.AddPayment.no.permission') + '</p>'
                    );

                    this.fireEvent('load', [this, true]);
                    return;
                }

                // Build content
                this.$Elm.set('html', Mustache.render(template, Object.merge({}, Data, {
                    labelDebtorNo: QUILocale.get(lg, 'controls.AddPayment.tpl.labelDebtorNo'),
                    labelAddressName: QUILocale.get('quiqqer/core', 'name'),
                    labelAddressStreet: QUILocale.get(lg, 'street'),
                    labelAddressCity: QUILocale.get(lg, 'city'),
                    labelAddressCountry: QUILocale.get(lg, 'country'),
                    labelAddressSalutation: QUILocale.get(lg, 'salutation'),
                    labelDocumentType: QUILocale.get(lg, 'controls.AddPayment.tpl.labelDocumentType'),
                    labelDocumentNo: QUILocale.get(lg, 'controls.AddPayment.tpl.labelDocumentNo'),
                    labelDate: QUILocale.get(lg, 'controls.AddPayment.tpl.labelDate'),
                    labelDueDate: QUILocale.get(lg, 'controls.AddPayment.tpl.labelDueDate'),
                    labelAmountTotal: QUILocale.get(lg, 'controls.AddPayment.tpl.labelAmountTotal'),
                    labelAmountPaid: QUILocale.get(lg, 'controls.AddPayment.tpl.labelAmountPaid'),
                    labelAmountOpen: QUILocale.get(lg, 'controls.AddPayment.tpl.labelAmountOpen'),
                    labelPayment: QUILocale.get(lg, 'controls.AddPayment.tpl.labelPayment'),
                    labelPaymentDate: QUILocale.get(lg, 'controls.AddPayment.tpl.labelPaymentDate'),
                    labelTransactionAmount: QUILocale.get(lg, 'controls.AddPayment.tpl.labelTransactionAmount'),
                    headerDebtor: QUILocale.get(lg, 'controls.AddPayment.tpl.headerDebtor'),
                    headerDocument: QUILocale.get(lg, 'controls.AddPayment.tpl.headerDocument'),
                    headerPayment: QUILocale.get(lg, 'controls.AddPayment.tpl.headerPayment'),
                    headerLinkedElements: QUILocale.get(lg, 'controls.AddPayment.tpl.linkedEntities')
                })));

                if (!Data.canBookPayment) {
                    const PaymentContainer = this.$Elm.getElement('.quiqqer-payment-transactions-add-payment');

                    PaymentContainer.set('html', QUILocale.get(lg, 'controls.AddPayment.tpl.noBook_info', {
                        reason: Data.noBookReason
                    }));

                    PaymentContainer.addClass('box message-attention');
                    PaymentContainer.setStyles({
                        padding: 10,
                        'font-size': 14
                    });

                    this.fireEvent('load', [this, false]);

                    return;
                }

                this.$Form = this.$Elm.getElement('form');

                this.$Form.addEvent('submit', function (event) {
                    event.stop();
                    this.fireEvent('submit', [this.getValue(), this]);
                }.bind(this));

                const Payments = this.getElm().getElement('[name="payment_method"]');
                const Amount = this.getElm().getElement('[name="amount"]');

                const preselectedPaymentId = parseInt(this.getAttribute('paymentId'));

                this.getElm().getElement('[name="date"]').valueAsDate = new Date();

                for (let i = 0, len = payments.length; i < len; i++) {
                    payment = payments[i];
                    title = payment.title;

                    if (typeOf(payment.title) === 'object' && current in payment.title) {
                        title = payment.title[current];
                    }

                    if (typeOf(payment.workingTitle) === 'object' &&
                        current in payment.workingTitle &&
                        payment.workingTitle[current] !== ''
                    ) {
                        title = payment.workingTitle[current];
                    }

                    new Element('option', {
                        html: title,
                        value: parseInt(payment.id),
                        selected: parseInt(payment.id) === preselectedPaymentId
                    }).inject(Payments);
                }

                if (Data.paymentId) {
                    Payments.value = Data.paymentId;
                }

                if (Data.amountOpenRaw) {
                    Amount.value = Data.amountOpenRaw;
                }

                new QUIButton({
                    'class': 'quiqqer-payment-transactions-AddPayment-btn-addExisting',
                    textimage: 'fa fa-money',
                    text: QUILocale.get(lg, 'controls.AddPayment.btn.addExisting'),
                    title: QUILocale.get(lg, 'controls.AddPayment.btn.addExisting'),
                    events: {
                        onClick: this.$onClickAddExistingTransaction
                    }
                }).inject(
                    this.$Elm.getElement('.quiqqer-payment-transactions-add-payment-btn-existing')
                );

                const LoaderRow = this.getElm().getElement('.quiqqer-payment-transactions-entities--loader');

                if (!this.getAttribute('globalProcessId')) {
                    this.fireEvent('load', [this, true]);
                    LoaderRow.getParent('table').destroy();
                    return;
                }

                this.$getLinkedEntities().then((entities) => {
                    const groups = [];
                    const Tbody = LoaderRow.getParent('tbody');

                    for (let key in entities.grouped) {
                        if (!entities.grouped.hasOwnProperty(key)) {
                            continue;
                        }

                        groups.push({
                            groupName: key,
                            items: entities.grouped[key]
                        });
                    }

                    if (groups.length) {
                        Tbody.set('html', Mustache.render(templateLinkedElements, {
                            groups: groups
                        }));

                        if (Tbody.querySelectorAll('[name="linked-element"]').length === 1) {
                            Tbody.querySelector('[name="linked-element"]').checked = true;
                        }

                        this.fireEvent('load', [this, true]);
                        return;
                    }

                    const message = '<tr><td>' +
                        QUILocale.get(lg, 'controls.AddPayment.empty.message', {
                            entityId: Data.documentNo
                        }) +
                        '</td></tr>';

                    Tbody.set('html', message);
                    this.fireEvent('load', [this, true]);
                }).catch(() => {
                    this.fireEvent('load', [this, true]);
                });
            });
        },

        $onClickAddExistingTransaction: function () {
            let TransactionsGridControl;
            let SubmitBtn;
            let txId;
            let SelectWindow;

            const NewPaymentContainer = this.getElm().getElement('.quiqqer-payment-transactions-add-payment');

            const displayExistingTransaction = (Transaction) => {
                this.$ExistingTransaction = Transaction;
                this.fireEvent('onSubmitModeChange', ['existing', this]);

                NewPaymentContainer.addClass(cssClassHidden);

                const ExistingPaymentContainer = new Element('div', {
                    'class': 'quiqqer-payment-transactions-add-payment-existing',
                    html: Mustache.render(templateExistingTransaction, Object.merge({}, {
                        headerExistingTransaction: QUILocale.get(
                            lg,
                            'controls.AddPayment.tpl.headerExistingTransaction'
                        ),
                        labelPayment: QUILocale.get(lg, 'controls.AddPayment.tpl.labelPayment'),
                        labelPaymentDate: QUILocale.get(lg, 'controls.AddPayment.tpl.labelPaymentDate'),
                        labelTransactionAmount: QUILocale.get(lg, 'controls.AddPayment.tpl.labelTransactionAmount'),
                        labelTxId: QUILocale.get(lg, 'txid')
                    }, Transaction))
                }).inject(NewPaymentContainer, 'after');

                new QUIButton({
                    'class': 'quiqqer-payment-transactions-AddPayment-btn-addNew',
                    textimage: 'fa fa-money',
                    text: QUILocale.get(lg, 'controls.AddPayment.btn.addNew'),
                    title: QUILocale.get(lg, 'controls.AddPayment.btn.addNew'),
                    events: {
                        onClick: () => {
                            ExistingPaymentContainer.destroy();
                            NewPaymentContainer.removeClass(cssClassHidden);

                            this.$ExistingTransaction = null;
                            this.fireEvent('onSubmitModeChange', ['new', this]);
                        }
                    }
                }).inject(
                    ExistingPaymentContainer.getElement(
                        '.quiqqer-payment-transactions-AddPayment-ExistingTransaction-new'
                    )
                );
            };

            SelectWindow = new QUIConfirm({
                maxHeight: 800,
                maxWidth: 1200,

                autoclose: false,
                backgroundClosable: true,

                title: QUILocale.get(lg, 'controls.AddPayment.TransactionsGrid.title'),
                icon: 'fa fa-money',

                cancel_button: {
                    text: QUILocale.get('quiqqer/core', 'cancel'),
                    textimage: 'fa fa-remove'
                },
                ok_button: {
                    text: QUILocale.get(lg, 'controls.AddPayment.TransactionsGrid.btn.submit'),
                    textimage: 'fa fa-check'
                },
                events: {
                    onOpen: (Win) => {
                        SubmitBtn = Win.getButton('submit');
                        SubmitBtn.disable();

                        const Content = Win.getContent();

                        Content.set('html', Mustache.render(templateTransactionsGrid, {
                            info: QUILocale.get(lg, 'controls.AddPayment.TransactionsGrid.info')
                        }));

                        TransactionsGridControl = new TransactionsGrid({
                            dblClickAction: 'submit',
                            events: {
                                onSelect: (selectedTxId) => {
                                    txId = selectedTxId;
                                    SubmitBtn.enable();
                                },
                                onSubmit: () => {
                                    Win.submit();
                                }
                            }
                        }).inject(Content.getElement('.quiqqer-payment-transactions-AddPayment-TransactionsGrid-grid'));

                        TransactionsGridControl.resize();
                    },
                    onResize: () => {
                        if (TransactionsGridControl) {
                            TransactionsGridControl.resize();
                        }
                    },
                    onSubmit: (Win) => {
                        Win.Loader.show();

                        Transactions.getTransaction(txId).then((Transaction) => {
                            displayExistingTransaction(Transaction);
                            Win.close();
                        }).catch(() => {
                            Win.Loader.hide();
                        });
                    }
                }
            });

            SelectWindow.open();
        },

        /**
         * Return the form data
         *
         * @return {Object}
         */
        getValue: function () {
            if (this.$ExistingTransaction) {
                return {
                    txId: this.$ExistingTransaction.txid
                };
            }

            const data = QUIFormUtils.getFormData(this.$Form);

            if (typeof data['linked-element'] === 'undefined') {
                data['linked-element'] = this.getAttribute('entityId');
            }

            return data;
        },

        /**
         * Focus the amount field
         */
        focus: function () {
            if (this.getElm().getElement('[name="amount"]')) {
                this.getElm().getElement('[name="amount"]').focus();
            }
        },

        /**
         * Get details for transactions
         *
         * @return {Promise}
         */
        $getData: function () {
            return new Promise((resolve, reject) => {
                QUIAjax.get('package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_getData', resolve, {
                    'package': 'quiqqer/payment-transactions',
                    entityId: this.getAttribute('entityId'),
                    entityType: this.getAttribute('entityType'),
                    onError: reject
                });
            });
        },

        /**
         * Returns the entities to which postings can be made
         *
         * @return {Promise}
         * */
        $getLinkedEntities: function () {
            return new Promise((resolve, reject) => {
                QUIAjax.get(
                    'package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_getLinkedEntities',
                    resolve,
                    {
                        'package': 'quiqqer/payment-transactions',
                        globalProcessId: this.getAttribute('globalProcessId'),
                        onError: reject
                    }
                );
            });
        }
    });
});