/**
 * Add a payment to an ERP entity (e.g. invoice, offer...)
 *
 * @module package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPaymentWindow
 *
 * @event onSubmit [this, Transaction] - fires if the user submits a new transaction
 * @event onSubmitExisting [txId, this] - fires if the user submits an existing transaction
 */
define('package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPaymentWindow', [

    'qui/controls/windows/Confirm',
    'package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment',
    'Permissions',
    'Locale',
    'Ajax'

], function (QUIConfirm, AddPayment, Permissions, QUILocale, QUIAjax) {
    'use strict';

    const lg = 'quiqqer/payment-transactions';

    return new Class({

        Extends: QUIConfirm,
        Type: 'package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPayment',

        Binds: [
            'submit'
        ],

        options: {
            hash: false,
            entityId: false,
            entityType: false,
            globalProcessId: false,
            paymentId: false   // ID of the payment that is preselected when adding new transactions
        },

        initialize: function (options) {
            this.setAttributes({
                icon: 'fa fa-money',
                title: QUILocale.get(lg, 'controls.AddPaymentWindow.title'),
                maxHeight: 800,
                maxWidth: 800,
                ok_button: {
                    text: QUILocale.get(lg, 'controls.AddPaymentWindow.btn.confirm'),
                    textimage: 'fa fa-check'
                }
            });

            this.parent(options);

            this.$AddPayment = null;

            this.addEvents({
                onOpen: this.$onOpen
            });
        },

        /**
         * event: on open
         */
        $onOpen: function () {
            this.Loader.show();
            this.getContent().set('html', '');

            const SubmitBtn = this.getButton('submit');

            let submitMode = 'new';

            this.$AddPayment = new AddPayment({
                entityId: this.getAttribute('entityId'),
                entityType: this.getAttribute('entityType'),
                paymentId: this.getAttribute('paymentId'),
                hash: this.getAttribute('hash'),
                globalProcessId: this.getAttribute('globalProcessId'),
                events: {
                    onLoad: (Control, canBookPayment) => {
                        Permissions.hasPermission(
                            'quiqqer.paymentTransactions.create'
                        ).then((permission) => {
                            if (!permission) {
                                this.$Buttons.querySelectorAll('button').forEach((btn) => {
                                    btn.setAttribute('disabled', 'disabled');
                                });
                            }

                            this.Loader.hide();
                        });

                        if (canBookPayment) {
                            this.$AddPayment.focus();
                            return;
                        }

                        SubmitBtn.disable();
                    },

                    onSubmitModeChange: (mode) => {
                        switch (mode) {
                            case 'new':
                                SubmitBtn.setAttribute(
                                    'text', QUILocale.get(lg, 'controls.AddPaymentWindow.btn.confirm')
                                );
                                break;

                            case 'existing':
                                SubmitBtn.setAttribute(
                                    'text', QUILocale.get(lg, 'controls.AddPaymentWindow.btn.confirm_existing')
                                );
                                break;
                        }

                        submitMode = mode;
                    },

                    onSubmit: this.submit
                }
            }).inject(this.getContent());
        },

        /**
         * Submit the window
         */
        submit: function () {
            const transactionData = this.$AddPayment.getValue();

            let hash = this.getAttribute('hash');

            if (!hash && this.getAttribute('uuid')) {
                hash = this.getAttribute('uuid');
            }

            if (!hash && this.getAttribute('entityId')) {
                hash = this.getAttribute('entityId');
            }

            if (
                transactionData
                && typeof transactionData.txId !== 'undefined'
                && hash
            ) {
                this.Loader.show();

                QUIAjax.post('package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_addTransaction', () => {
                    this.fireEvent('submit', [this, transactionData]);
                    this.close();
                }, {
                    'package': 'quiqqer/payment-transactions',
                    'uuid': hash,
                    'txId': transactionData.txId
                });

                return;
            }

            // global process id stuff
            if (this.getAttribute('globalProcessId')) {
                if (!transactionData['linked-element']) {
                    QUI.getMessageHandler().then((MH) => {
                        MH.addError(
                            'Bitte wähle ein verknüpftes Element aus, auf welches die Zahlung gebucht werden soll'
                        );
                    });

                    return;
                }

                this.Loader.show();

                QUIAjax.post(
                    'package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_addTransaction',
                    () => {
                        this.fireEvent('submit', [this, transactionData]);
                        this.close();
                    },
                    {
                        'package': 'quiqqer/payment-transactions',
                        'uuid': transactionData['linked-element'],
                        'amount': transactionData.amount,
                        'paymentMethod': transactionData.payment_method,
                        'date': transactionData.date
                    }
                );

                return;
            }

            if (!hash) {
                QUI.getMessageHandler().then((MH) => {
                    MH.addError('Missing hash');
                });
                return;
            }

            this.Loader.show();

            QUIAjax.post('package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_addTransaction', () => {
                this.fireEvent('submit', [this, transactionData]);
                this.close();
            }, {
                'package': 'quiqqer/payment-transactions',
                'uuid': hash,
                'amount': transactionData.amount,
                'paymentMethod': transactionData.payment_method,
                'date': transactionData.date
            });
        }
    });
});
