/**
 * List all transactions in a grid.
 *
 * @module package/quiqqer/payment-transactions/bin/backend/controls/TransactionsGrid
 *
 * @event onSelect [txId, this] - fires if a transaction entry is clicked (selected) in the grid
 * @event onSubmit [txId, this] - fires if a transaction entry is "submitted" via double click
 *                                (requires "dblClickAction" to be set to "submit")
 */
define('package/quiqqer/payment-transactions/bin/backend/controls/TransactionsGrid', [

    'qui/controls/Control',
    'controls/grid/Grid',
    'Ajax',
    'Locale'

], function (QUIControl, Grid, QUIAjax, QUILocale) {
    'use strict';

    const lg = 'quiqqer/payment-transactions';

    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/payment-transactions/bin/backend/controls/TransactionsGrid',

        Binds: [
            '$onInject',
            'openTransaction',
            'refresh',
            'getGrid'
        ],

        options: {
            title: QUILocale.get(lg, 'menu.erp.payment-transactions.title'),
            icon: 'fa fa-money',

            dblClickAction: 'open' // "open" = opens transactions details; "select" = fires select event; "submit" = fires submit event
        },

        initialize: function (options) {
            this.parent(options);

            this.$Grid = null;
            this.$GridContainer = null;

            this.addEvents({
                onInject: this.$onInject
            });
        },

        /**
         * @return {void}
         */
        $onInject: function () {
            const Elm = this.getElm();

            Elm.addClass('quiqqer-payment-transactions-TransactionsGrid');
            Elm.setStyle('height', '100%');

            this.$GridContainer = new Element('div', {
                'class': 'quiqqer-payment-transactions-TransactionsGrid-container'
            }).inject(Elm);

            this.$Grid = new Grid(this.$GridContainer, {
                pagination: true,
                columnModel: [
                    {
                        header: QUILocale.get(lg, 'grid.status'),
                        dataIndex: 'status_node',
                        dataType: 'node',
                        width: 60,
                        className: 'centering'
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'date'),
                        dataIndex: 'date',
                        dataType: 'string',
                        width: 140
                    },
                    {
                        header: QUILocale.get(lg, 'txid'),
                        dataIndex: 'txid',
                        dataType: 'string',
                        width: 260,
                        className: 'monospace'
                    },
                    {
                        header: QUILocale.get(lg, 'grid.amount'),
                        dataIndex: 'amount',
                        dataType: 'numeric',
                        width: 100,
                        className: 'amountCell'
                    },
                    {
                        header: QUILocale.get(lg, 'grid.currency'),
                        dataIndex: 'currency_code',
                        dataType: 'string',
                        width: 60
                    },
                    {
                        header: QUILocale.get('quiqqer/erp', 'global_process_id'),
                        dataIndex: 'global_process_id',
                        dataType: 'string',
                        width: 260,
                        className: 'monospace'
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'user_id'),
                        dataIndex: 'uid',
                        dataType: 'numeric',
                        width: 100
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'username'),
                        dataIndex: 'username',
                        dataType: 'string',
                        width: 120
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'name'),
                        dataIndex: 'user_name',
                        dataType: 'string',
                        width: 140
                    },
                    {
                        header: QUILocale.get(lg, 'grid.payment'),
                        dataIndex: 'payment',
                        dataType: 'string',
                        width: 300,
                        className: 'monospace'
                    },
                    {
                        header: QUILocale.get(lg, 'grid.erpEntity'),
                        dataIndex: 'hash',
                        dataType: 'string',
                        width: 300,
                        className: 'monospace'
                    }
                ],
                onrefresh: this.refresh
            });

            this.$Grid.addEvents({
                onDblClick: () => {
                    const Row = this.$Grid.getSelectedData()[0];

                    switch (this.getAttribute('dblClickAction')) {
                        case 'open':
                            this.openTransaction(Row.txid);
                            break;

                        case 'submit':
                            this.fireEvent('submit', [Row.txid, this]);
                            break;
                    }
                },
                onClick: () => {
                    const Row = this.$Grid.getSelectedData()[0];
                    this.fireEvent('select', [Row.txid, this]);
                }
            });

            this.resize();
            this.refresh();
        },

        /**
         * refresh the data
         *
         * @return {Promise|*}
         */
        refresh: function () {
            return new Promise((resolve, reject) => {
                QUIAjax.get('package_quiqqer_payment-transactions_ajax_backend_list', (result) => {
                    let i, len, icon;

                    for (i = 0, len = result.grid.data.length; i < len; i++) {
                        icon = 'fa fa-minus';

                        switch (parseInt(result.grid.data[i].status)) {
                            case 1:
                                icon = 'fa fa-check';
                                break;

                            case 2:
                                icon = 'fa fa-clock-o';
                                break;

                            case 3:
                                icon = 'fa fa-bolt';
                                break;
                        }

                        result.grid.data[i].status_node = new Element('span', {
                            'class': icon
                        });
                    }

                    this.$Grid.setData(result.grid);
                    resolve();
                }, {
                    'package': 'quiqqer/payment-transactions',
                    params: JSON.encode({
                        perPage: this.$Grid.options.perPage,
                        page: this.$Grid.options.page
                    }),
                    onError: reject
                });
            });
        },

        /**
         * Open the add dialog
         *
         * @param {String} txid
         */
        openTransaction: function (txid) {
            require([
                'package/quiqqer/payment-transactions/bin/backend/controls/windows/Transaction'
            ], function (Transaction) {
                new Transaction({
                    txid: txid
                }).open();
            });
        },

        /**
         * @return {void}
         */
        resize: function () {
            if (!this.$GridContainer) {
                return;
            }

            const size = this.getElm().getSize();

            this.$Grid.setHeight(size.y);
            this.$Grid.setWidth(size.x);
        },

        /**
         * @return {Object}
         */
        getGrid: function () {
            return this.$Grid;
        }
    });
});
