<?php

/**
 * This file contains QUI\ERP\Accounting\Payments\Transactions\Factory
 */

namespace QUI\ERP\Accounting\Payments\Transactions;

use QUI;
use QUI\ERP\Currency\Currency;
use QUI\Interfaces\Users\User;

use function date;
use function is_numeric;
use function json_encode;
use function strtotime;
use function time;

/**
 * Class Factory
 *
 * @package QUI\ERP\Accounting\Payments\Transactions
 */
class Factory
{
    /**
     * @return string
     */
    public static function table(): string
    {
        return QUI::getDBTableName('payment_transactions');
    }

    /**
     * Create a new payment transaction
     *
     * @param float|int $amount
     * @param Currency $Currency - currency
     * @param bool|string $hash - erp entity hash
     * @param string $payment - name of the Payment
     * @param array $data - variable, optional data
     * @param User|null $User - user which execute the transaction, or from whom the transaction comes from
     * @param bool|int|string $date - transaction date, 0000-00-00 || 0000-00-00 00:00:00 || Unix Timestamp
     * @param bool|string $globalProcessId - for the global process hash, if empty, the hash will be used
     * @param int $status (optional) - Initial transaction status; see \QUI\ERP\Accounting\Payments\Transactions\Handler::STATUS_*
     *
     * @return Transaction
     *
     * @throws Exception
     */
    public static function createPaymentTransaction(
        float | int $amount,
        Currency $Currency,
        bool | string $hash = false,
        string $payment = '',
        array $data = [],
        QUI\Interfaces\Users\User | null $User = null,
        bool | int | string $date = false,
        bool | string $globalProcessId = false,
        int $status = Handler::STATUS_COMPLETE
    ): Transaction {
        $txId = QUI\Utils\Uuid::get();

        if (empty($hash)) {
            $hash = '';
        }

        if (empty($globalProcessId)) {
            $globalProcessId = $hash;
        }

        if (!QUI::getUsers()->isUser($User)) {
            $User = QUI::getUserBySession();
        }

        // date
        if (
            QUI\Utils\Security\Orthos::checkMySqlDateSyntax($date) ||
            QUI\Utils\Security\Orthos::checkMySqlDatetimeSyntax($date)
        ) {
            if (!str_contains($date, ' ')) {
                $date .= ' ' . date('H:i:s');
            }

            $date = strtotime($date);
        }

        if (!is_numeric($date)) {
            $date = time();
        }

        if (!is_numeric($date)) {
            $date = strtotime($date);
        }

        $date = date('Y-m-d H:i:s', $date);
        $uuid = $User->getUUID();

        if (empty($uuid)) {
            $uuid = QUI::getUsers()->getSystemUser()->getUUID();
        }

        try {
            $data = QUI\Security\Encryption::encrypt(json_encode($data));
        } catch (\Exception) {
            $data = json_encode($data);
        }

        try {
            QUI::getDataBase()->insert(self::table(), [
                'txid' => $txId,
                'hash' => $hash,
                'date' => $date,
                'uid' => $uuid,
                'amount' => $amount,
                'currency' => json_encode($Currency->toArray()),
                'data' => $data,
                'payment' => $payment,
                'global_process_id' => $globalProcessId,
                'status' => $status
            ]);
        } catch (QUI\Database\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            throw new QUI\ERP\Accounting\Payments\Transactions\Exception(
                ['quiqqer/payment-transactions', 'exception.something.went.wrong']
            );
        }

        $Transaction = Handler::getInstance()->get($txId);

        try {
            QUI::getEvents()->fireEvent('transactionCreate', [$Transaction]);
        } catch (\Exception $Exception) {
            QUI\System\Log::writeRecursive(
                $Exception->getMessage(),
                QUI\System\Log::LEVEL_ERROR
            );

            QUI\System\Log::writeRecursive(
                $Exception->getTraceAsString(),
                QUI\System\Log::LEVEL_ERROR
            );
        }

        return $Transaction;
    }

    /**
     * Create a refund transaction
     * A refund transaction use the negative value to the transaction list.
     *
     * @param float|int $amount
     * @param Currency $Currency - currency
     * @param bool|string $hash - invoice / order hash
     * @param string $payment - name of the Payment
     * @param array $data - variable, optional data
     * @param User|null $User - user which execute the transaction, or from whom the transaction comes from
     * @param bool|int|string $date - transaction date, 0000-00-00 || 0000-00-00 00:00:00 || Unix Timestamp
     * @param bool|string $globalProcessId - for the global process hash
     * @param int $status (optional) - Initial transaction status; see \QUI\ERP\Accounting\Payments\Transactions\Handler::STATUS_*
     *
     * @return Transaction
     *
     * @throws Exception
     */
    public static function createPaymentRefundTransaction(
        float | int $amount,
        Currency $Currency,
        bool | string $hash = false,
        string $payment = '',
        array $data = [],
        QUI\Interfaces\Users\User | null $User = null,
        bool | int | string $date = false,
        bool | string $globalProcessId = false,
        int $status = Handler::STATUS_COMPLETE
    ): Transaction {
        $amount = $amount * -1; // for the internal system the amount must be the opposite

        return self::createPaymentTransaction(
            $amount,
            $Currency,
            $hash,
            $payment,
            $data,
            $User,
            $date,
            $globalProcessId,
            $status
        );
    }
}
