<?php

namespace QUI\ERP\Accounting\Payments\Transactions\IncomingPayments;

use QUI;

use function class_exists;

/**
 * Class Handler
 *
 * Handles incoming payment transaction provider
 */
class Handler
{
    /**
     * Get ERP payment receiver by entity type
     *
     * @param string $type
     * @param int|string $id - Payment receiver entity ID
     * @return PaymentReceiverInterface|false - PaymentReceiverInterface class or false if not found
     */
    public static function getPaymentReceiver(string $type, int|string $id): PaymentReceiverInterface|bool
    {
        $ProviderInstance = false;

        /** @var PaymentReceiverInterface $Provider */
        foreach (self::getAllPaymentReceiverProviders() as $Provider) {
            if ($Provider::getType() === $type) {
                $ProviderInstance = new $Provider($id);
                break;
            }
        }

        return $ProviderInstance;
    }

    /**
     * Get all available ERP payment receiver providers
     *
     * @return string[] - Provider classes (static)
     */
    public static function getAllPaymentReceiverProviders(): array
    {
        $packages = QUI::getPackageManager()->getInstalled();
        $providerClasses = [];

        foreach ($packages as $installedPackage) {
            try {
                $Package = QUI::getPackage($installedPackage['name']);

                if (!$Package->isQuiqqerPackage()) {
                    continue;
                }

                $packageProvider = $Package->getProvider();

                if (empty($packageProvider['paymentReceiver'])) {
                    continue;
                }

                foreach ($packageProvider['paymentReceiver'] as $class) {
                    if (!class_exists($class)) {
                        continue;
                    }

                    $providerClasses[] = $class;
                }
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $providerClasses;
    }
}
