<?php

/**
 * This file contains QUI\ERP\Accounting\Payments\Transactions\Factory
 */

namespace QUI\ERP\Accounting\Payments\Transactions;

use PDO;
use QUI;
use QUI\ERP\Accounting\Payments\Payments;

use function array_flip;
use function floatval;
use function implode;
use function is_array;
use function json_decode;

/**
 * Class Search
 *
 * @package QUI\ERP\Accounting\Payments\Transactions
 */
class Search extends QUI\Utils\Singleton
{
    /**
     * @var array
     */
    protected array $filter = [];

    /**
     * @var array
     */
    protected array $limit = [0, 20];

    /**
     * @var string
     */
    protected string $order = 'date DESC';

    /**
     * @var array
     */
    protected array $allowedFilters = [
        'from',
        'to'
    ];

    /**
     * @var array
     */
    protected array $cache = [];

    /**
     * Set a filter
     *
     * @param string $filter
     * @param array|string $value
     * @todo implement
     */
    public function setFilter(string $filter, array|string $value): void
    {
        $keys = array_flip($this->allowedFilters);

        if (!isset($keys[$filter])) {
            return;
        }

        if (!is_array($value)) {
            $value = [$value];
        }
    }

    /**
     * Clear all filters
     */
    public function clearFilter(): void
    {
        $this->filter = [];
    }

    /**
     * Set the limit
     *
     * @param int $from
     * @param int $to
     */
    public function limit(int $from, int $to): void
    {
        $this->limit = [$from, $to];
    }

    /**
     * Set the order
     *
     * @param $order
     */
    public function order($order): void
    {
        switch ($order) {
            case 'txid':
            case 'txid ASC':
            case 'txid DESC':
            case 'amount':
            case 'amount ASC':
            case 'amount DESC':
            case 'date':
            case 'date ASC':
            case 'date DESC':
                $this->order = $order;
                break;
        }
    }

    /**
     * Execute the search and return the order list
     *
     * @return array
     *
     * @throws QUI\Exception
     */
    public function search(): array
    {
        return $this->executeQueryParams($this->getQuery());
    }

    /**
     * Execute the search and return the order list for a grid control
     *
     * @return array
     * @throws QUI\Exception
     */
    public function searchForGrid(): array
    {
        $this->cache = [];

        // select display orders
        $orders = $this->executeQueryParams($this->getQuery());

        // count
        $count = $this->executeQueryParams($this->getQueryCount());
        $count = (int)$count[0]['count'];

        // result
        $result = $this->parseListForGrid($orders);
        $Grid = new QUI\Utils\Grid();
        $Grid->setAttribute('page', ($this->limit[0] / $this->limit[1]) + 1);

        return [
            'grid' => $Grid->parseResult($result, $count),
            'total' => $count
        ];
    }

    /**
     * @param bool $count - Use count select, or not
     * @return array
     */
    protected function getQuery(bool $count = false): array
    {
        $table = Factory::table();
        $order = $this->order;

        // limit
        $limit = '';

        if ($this->limit && isset($this->limit[0]) && isset($this->limit[1])) {
            $start = $this->limit[0];
            $end = $this->limit[1];
            $limit = " LIMIT $start,$end";
        }

        if (empty($this->filter)) {
            if ($count) {
                return [
                    'query' => " SELECT COUNT(txid) AS count FROM $table",
                    'binds' => []
                ];
            }

            return [
                'query' => "
                    SELECT *
                    FROM {$table}
                    ORDER BY {$order}
                    {$limit}
                ",
                'binds' => []
            ];
        }

        $where = [];
        $binds = [];
        $fc = 0;

        foreach ($this->filter as $filter) {
            $bind = ':filter' . $fc;

            switch ($filter['filter']) {
                case 'from':
                    $where[] = 'date >= ' . $bind;
                    continue 2;

                case 'to':
                    $where[] = 'date <= ' . $bind;
                    continue 2;
            }

            $binds[$bind] = [
                'value' => $filter['value'],
                'type' => PDO::PARAM_STR
            ];

            $fc++;
        }

        $whereQuery = 'WHERE ' . implode(' AND ', $where);


        if ($count) {
            return [
                "query" => "
                    SELECT COUNT(txid) AS count
                    FROM {$table}
                    {$whereQuery}
                ",
                'binds' => $binds
            ];
        }

        return [
            "query" => "
                SELECT id
                FROM {$table}
                {$whereQuery}
                ORDER BY {$order}
                {$limit}
            ",
            'binds' => $binds
        ];
    }

    /**
     * @return array
     */
    protected function getQueryCount(): array
    {
        return $this->getQuery(true);
    }

    /**
     * @param array $queryData
     * @return array
     * @throws QUI\Exception
     */
    protected function executeQueryParams(array $queryData = []): array
    {
        $PDO = QUI::getDataBase()->getPDO();
        $binds = $queryData['binds'];
        $query = $queryData['query'];

        $Statement = $PDO->prepare($query);

        foreach ($binds as $var => $bind) {
            $Statement->bindValue($var, $bind['value'], $bind['type']);
        }

        try {
            $Statement->execute();

            return $Statement->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Exception $Exception) {
            QUI\System\Log::writeRecursive($Exception);
            QUI\System\Log::writeRecursive($query);
            QUI\System\Log::writeRecursive($binds);
            throw new QUI\Exception('Something went wrong');
        }
    }

    /**
     * @param array $data
     * @return array
     */
    protected function parseListForGrid(array $data): array
    {
        $Users = QUI::getUsers();

        foreach ($data as $key => $entry) {
            $data[$key]['currency'] = json_decode($entry['currency'], true);
            $data[$key]['data'] = json_decode($entry['data'], true);
            $data[$key]['uid'] = $entry['uid'];

            // user
            try {
                $User = $Users->get($data[$key]['uid']);

                $data[$key]['username'] = $User->getUsername();
                $data[$key]['user_name'] = $User->getName();
            } catch (QUI\Exception) {
                $data[$key]['username'] = '---';
                $data[$key]['user_name'] = '---';
            }

            // currency
            $Currency = null;
            $amount = floatval($entry['amount']);

            try {
                $Currency = QUI\ERP\Currency\Handler::getCurrency(
                    $data[$key]['currency']['code']
                );

                $data[$key]['currency_code'] = $Currency->getCode();
            } catch (QUI\Exception) {
                $data[$key]['currency_code'] = '---';
            }

            $data[$key]['amount'] = $amount;

            // payment
            try {
                $Payment = Payments::getInstance()->getPaymentType(
                    $data[$key]['payment']
                );

                $data[$key]['payment'] = $Payment->getTitle();
            } catch (\Exception $Exception) {
                QUI\System\Log::addWarning(
                    $Exception->getMessage(),
                    [
                        'payment' => $data[$key]['payment']
                    ]
                );

                $data[$key]['payment'] = '---';
            }
        }

        return $data;
    }
}
