<?php

/**
 * This file contains QUI\ERP\Accounting\Payments\Transactions\EventHandler
 */

namespace QUI\ERP\Accounting\Payments\Transactions;

use QUI;
use QUI\Database\Exception;
use QUI\ERP\Accounting\Invoice\Invoice;
use QUI\ERP\Accounting\Invoice\InvoiceTemporary;
use QUI\ERP\Accounting\Payments\Api\AbstractPayment;

use function abs;
use function floatval;
use function is_array;
use function is_numeric;

/**
 * Class EventHandler
 *
 * @package QUI\ERP\Accounting\Payments\Transactions
 */
class EventHandler
{
    /**
     * @param QUI\Package\Package $Package
     * @throws Exception
     */
    public static function onPackageSetup(QUI\Package\Package $Package): void
    {
        if ($Package->getName() !== 'quiqqer/payment-transactions') {
            return;
        }

        $result = QUI::getDataBase()->fetch([
            'from' => Factory::table(),
            'where' => [
                'status' => null
            ]
        ]);

        foreach ($result as $entry) {
            QUI::getDataBase()->update(
                Factory::table(),
                ['status' => Handler::STATUS_COMPLETE],
                ['txid' => $entry['txid']]
            );
        }
    }

    /**
     * @param QUI\ERP\Accounting\Invoice\InvoiceTemporary $Invoice - CreditNote
     */
    public static function onQuiqqerInvoiceTemporaryInvoicePostBegin(InvoiceTemporary $Invoice): void
    {
        // Do not refund if invoice is not a credit note
        if ($Invoice->getInvoiceType() !== QUI\ERP\Constants::TYPE_INVOICE_CREDIT_NOTE) {
            return;
        }

        $refund = $Invoice->getData('refund');

        if (!$refund) {
            return;
        }

        if (!is_array($refund)) {
            return;
        }

        try {
            $Transaction = Handler::getInstance()->get($refund['txid']);
            $amount = abs($refund['refund']);
            $sum = abs($Invoice->getAttribute('sum')); // amount is positive for the dev api

            if ($sum && $sum <= $amount) {
                $amount = $sum;
            }

            $Transaction->refund(
                $amount,
                $refund['message'],
                $Invoice->getUUID()
            );

            $Invoice->addHistory(
                QUI::getLocale()->get('quiqqer/payment-transactions', 'invoice.history.refund', [
                    'amount' => $refund['refund'],
                    'currency' => $Invoice->getCurrency()->getSign(),
                    'txid' => ''
                ])
            );
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeDebugException($Exception);

            try {
                $Invoice->addHistory(
                    QUI::getLocale()->get('quiqqer/payment-transactions', 'invoice.history.refund.exception', [
                        'message' => $Exception->getMessage()
                    ])
                );

                QUI::getMessagesHandler()->sendError(
                    QUI::getUserBySession(),
                    QUI::getLocale()->get('quiqqer/payment-transactions', 'invoice.history.refund.exception', [
                        'message' => $Exception->getMessage()
                    ])
                );
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeDebugException($Exception);
            }
        }
    }

    /**
     * @param QUI\ERP\Accounting\Payments\Transactions\Transaction $Transaction
     * @param AbstractPayment $Payment
     * @throws Exception
     */
    public static function onTransactionSuccessfullyRefunded(
        QUI\ERP\Accounting\Payments\Transactions\Transaction $Transaction,
        AbstractPayment $Payment
    ): void {
        $alreadyFunded = $Transaction->getData('refundAmount');

        if (!$alreadyFunded) {
            $alreadyFunded = 0;
        }

        $alreadyFunded = floatval($alreadyFunded) + $Transaction->getAmount();

        $Transaction->setData('refund', 1);
        $Transaction->setData('refundAmount', $alreadyFunded);
        $Transaction->updateData();
    }

    /**
     * When an invoice is canceled, the payments (if any were posted on the invoice)
     * must be posted in the opposite direction on the cancellation invoice
     *
     * @param Invoice $Invoice
     * @param Invoice $CreditNote
     * @return void
     * @throws QUI\Exception
     * @throws Exception
     */
    public static function onQuiqqerInvoiceReversalEnd(
        QUI\ERP\Accounting\Invoice\Invoice $Invoice,
        QUI\ERP\Accounting\Invoice\Invoice $CreditNote
    ): void {
        $Handler = QUI\ERP\Accounting\Payments\Transactions\Handler::getInstance();
        $list = $Handler->getTransactionsByHash($Invoice->getGlobalProcessId());

        foreach ($list as $TX) {
            // create the transaction
            QUI\ERP\Accounting\Payments\Transactions\Factory::createPaymentTransaction(
                $TX->getAmount() * -1,
                $CreditNote->getCurrency(),
                $CreditNote->getUUID(),
                '',
                [],
                QUI::getUserBySession(),
                false,
                $CreditNote->getGlobalProcessId()
            );
        }
    }

    /**
     * @throws Exception
     */
    public static function onQuiqqerMigrationV2(QUI\System\Console\Tools\MigrationV2 $Console): void
    {
        $Console->writeLn('- Migrate payment transactions');

        $result = QUI::getDataBase()->fetch([
            'from' => Factory::table()
        ]);

        foreach ($result as $entry) {
            $uid = $entry['uid'];

            if (!is_numeric($uid)) {
                continue;
            }

            try {
                QUI::getDataBase()->update(
                    Factory::table(),
                    ['uid' => QUI::getUsers()->get($uid)->getUUID()],
                    ['txid' => $entry['txid']]
                );
            } catch (QUI\Exception) {
            }
        }
    }
}
