<?php

/**
 * This file contains QUI\ERP\Accounting\Payments\Transactions\Handler
 */

namespace QUI\ERP\Accounting\Payments\Transactions;

use QUI;

/**
 * Class Handler
 *
 * @package QUI\ERP\Accounting\Payments\Transactions
 */
class Handler extends QUI\Utils\Singleton
{
    /**
     * This is the default status of a transaction
     */
    const STATUS_DEFAULT = 0;

    /**
     * Transaction is successfully transmitted and completed
     */
    const STATUS_COMPLETE = 1;

    /**
     * Pending status
     * can happen if the transaction has not been completed yet.
     * payment provider still needs to confirm
     */
    const STATUS_PENDING = 2;

    /**
     * Some error occurred
     */
    const STATUS_ERROR = 3;

    /**
     * @var array
     */
    protected array $tx = [];

    /**
     * Return a specific Transaction
     *
     * @param string $txId - transaction ID
     * @return Transaction
     *
     * @throws Exception
     */
    public function get(string $txId): Transaction
    {
        if (!isset($this->tx[$txId])) {
            $this->tx[$txId] = new Transaction($txId);
        }

        return $this->tx[$txId];
    }

    /**
     * Return the data from a specific Transaction
     *
     * @param string $txId - transaction ID
     * @return array
     *
     * @throws Exception
     */
    public function getTxData(string $txId): array
    {
        try {
            $result = QUI::getDataBase()->fetch([
                'from' => Factory::table(),
                'where' => [
                    'txid' => $txId
                ],
                'limit' => 1
            ]);
        } catch (QUI\Database\Exception) {
            throw new Exception('Transaction not found');
        }

        if (!isset($result[0])) {
            throw new Exception('Transaction not found');
        }

        return $result[0];
    }

    /**
     * Return all transactions from a specific hash
     *
     * @param string $hash
     * @return Transaction[]
     */
    public function getTransactionsByHash(string $hash): array
    {
        try {
            $result = QUI::getDataBase()->fetch([
                'select' => 'txid',
                'from' => Factory::table(),
                'where_or' => [
                    'hash' => $hash,
                    'linkedHashes' => [
                        'type' => '%LIKE%',
                        'value' => '"' . $hash . '"'
                    ]
                ]
            ]);
        } catch (QUI\Database\Exception) {
            return [];
        }

        $transactions = [];

        foreach ($result as $entry) {
            try {
                $transactions[] = $this->get($entry['txid']);
            } catch (Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $transactions;
    }

    /**
     * Return all transactions from a specific process
     *
     * @param string $processId
     * @return Transaction[]
     */
    public function getTransactionsByProcessId(string $processId): array
    {
        try {
            $result = QUI::getDataBase()->fetch([
                'select' => 'txid',
                'from' => Factory::table(),
                'where' => [
                    'global_process_id' => $processId
                ]
            ]);
        } catch (QUI\Database\Exception) {
            return [];
        }

        $transactions = [];

        foreach ($result as $entry) {
            try {
                $transactions[] = $this->get($entry['txid']);
            } catch (Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $transactions;
    }
}
