/**
 * Lists all payment transactions for a specific global process hash
 *
 * @modue package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/TransactionList
 * @author www.pcsg.de (Henning Leutz)
 * @author www.pcsg.de (Patrick)
 *
 * @event onLoad [self]
 * @event onAddTransaction [TransactionData, self]
 * @event onLinkTransaction [txId, self]
 */
define('package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/TransactionList', [

    'qui/controls/Control',
    'controls/grid/Grid',
    'Locale',
    'Ajax'

], function(QUIControl, Grid, QUILocale, QUIAjax) {
    'use strict';

    const lg = 'quiqqer/payment-transactions';

    return new Class({

        Type: 'package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/TransactionList',
        Extends: QUIControl,

        Binds: [
            '$onInject',
            'openAddPaymentDialog',
            'openRefundDialog'
        ],

        options: {
            globalProcessId: false,
            hash: false,
            entityType: false,
            Panel: false,
            disabled: false,
            paymentId: false   // ID of the payment that is preselected when adding new transactions
        },

        initialize: function(options) {
            this.parent(options);

            this.$Grid = null;
            this.$GridContainer = null;

            this.addEvents({
                onInject: this.$onInject,
                onResize: this.$onResize
            });
        },

        /**
         * Refresh the data and the display
         *
         * @return {Promise}
         */
        refresh: function() {
            const self = this;

            return this.$getList().then(function(result) {
                let i, len, icon;

                for (i = 0, len = result.length; i < len; i++) {
                    icon = 'fa fa-minus';

                    switch (parseInt(result[i].status)) {
                        case 1:
                            icon = 'fa fa-check';
                            break;

                        case 2:
                            icon = 'fa fa-clock-o';
                            break;

                        case 3:
                            icon = 'fa fa-bolt';
                            break;
                    }

                    result[i].status_node = new Element('span', {
                        'class': icon
                    });
                }

                self.$Grid.setData({
                    data: result
                });

                self.fireEvent('load', [self]);
            }).then(function() {
                const AddButton = self.$Grid.getButtons().filter(function(Button) {
                    return Button.getAttribute('name') === 'add';
                })[0];

                if (!self.getAttribute('disabled')) {
                    AddButton.enable();
                } else {
                    AddButton.disable();
                }

                const RefundButton = self.$Grid.getButtons().filter(function(Button) {
                    return Button.getAttribute('name') === 'refund';
                })[0];

                RefundButton.disable();
            });
        },

        /**
         * Creates the DomNode Element
         *
         * @return {Element}
         */
        create: function() {
            this.$Elm = this.parent();

            this.$Elm.setStyles({
                height: '100%'
            });

            return this.$Elm;
        },

        /**
         * event: on inject
         */
        $onInject: function() {
            this.$GridContainer = new Element('div', {
                styles: {
                    height: '100%'
                }
            }).inject(this.$Elm);

            this.$Grid = new Grid(this.$GridContainer, {
                'button-reload': true,
                buttons: [
                    {
                        name: 'add',
                        text: QUILocale.get(lg, 'controls.TransactionList.btn.postPayment'),
                        textimage: 'fa fa-money',
                        disabled: true,
                        events: {
                            onClick: this.openAddPaymentDialog
                        }
                    },
                    {
                        name: 'refund',
                        text: QUILocale.get(lg, 'controls.TransactionList.btn.refund'),
                        textimage: 'fa fa-handshake-o',
                        disabled: true,
                        events: {
                            onClick: this.openRefundDialog
                        }
                    }
                ],
                columnModel: [
                    {
                        header: QUILocale.get(lg, 'grid.status'),
                        dataIndex: 'status_node',
                        dataType: 'node',
                        width: 60,
                        className: 'centering'
                    },
                    {
                        header: QUILocale.get(lg, 'controls.TransactionList.date'),
                        dataIndex: 'date',
                        dataType: 'date',
                        width: 160
                    },
                    {
                        header: QUILocale.get(lg, 'controls.TransactionList.amount'),
                        dataIndex: 'amount',
                        dataType: 'string',
                        className: 'journal-grid-amount',
                        width: 160
                    },
                    {
                        header: QUILocale.get(lg, 'controls.TransactionList.paymentMethod'),
                        dataIndex: 'payment',
                        dataType: 'string',
                        width: 200
                    },
                    {
                        header: QUILocale.get(lg, 'controls.TransactionList.txid'),
                        dataIndex: 'txid',
                        dataType: 'string',
                        width: 225
                    }
                ]
            });

            this.$Grid.addEvents({
                onRefresh: () => {
                    this.refresh();
                },
                onDblClick: () => {
                    this.$openTransactionId(
                        this.$Grid.getSelectedData()[0].txid
                    );
                },
                onClick: () => {
                    const data = this.$Grid.getSelectedData();
                    const RefundButton = this.$Grid.getButtons().filter(function(Button) {
                        return Button.getAttribute('name') === 'refund';
                    })[0];

                    if (data.length === 1) {
                        RefundButton.enable();
                    } else {
                        RefundButton.disable();
                    }
                }
            });

            this.resize();
            this.refresh();
        },

        /**
         * Opens the add payment dialog
         */
        openAddPaymentDialog: function() {
            const self = this;

            const Button = this.$Grid.getButtons().filter(function(Button) {
                return Button.getAttribute('name') === 'add';
            })[0];

            Button.setAttribute('textimage', 'fa fa-spinner fa-spin');

            require([
                'package/quiqqer/payment-transactions/bin/backend/controls/IncomingPayments/AddPaymentWindow'
            ], function(AddPaymentWindow) {
                new AddPaymentWindow({
                    entityId: self.getAttribute('hash'),
                    entityType: self.getAttribute('entityType'),
                    paymentId: self.getAttribute('paymentId'),
                    globalProcessId: self.getAttribute('globalProcessId'),
                    events: {
                        onSubmit: function(Win, data) {
                            self.fireEvent('addTransaction', [data, self]);
                        },
                        onSubmitExisting: (txId) => {
                            self.fireEvent('linkTransaction', [txId, self]);
                        },
                        onClose: function() {
                            Button.setAttribute('textimage', 'fa fa-money');
                        }
                    }
                }).open();
            });
        },

        openRefundDialog: function() {
            const data = this.$Grid.getSelectedData();

            if (data.length !== 1) {
                return;
            }

            const txid = data[0].txid;
            const Button = this.$Grid.getButtons().filter(function(Button) {
                return Button.getAttribute('name') === 'refund';
            })[0];

            Button.setAttribute('textimage', 'fa fa-spinner fa-spin');

            require([
                'package/quiqqer/payment-transactions/bin/backend/controls/refund/Window'
            ], (RefundWindow) => {
                new RefundWindow({
                    txid: txid,
                    events: {
                        onSubmit: () => {
                            this.$Grid.refresh();
                        },
                        onClose: () => {
                            Button.setAttribute('textimage', 'fa fa-handshake-o');
                        }
                    }
                }).open();
            });
        },

        /**
         * opens a transaction window
         *
         * @param {String} txid - Transaction ID
         */
        $openTransactionId: function(txid) {
            const self = this;

            if (this.getAttribute('Panel')) {
                this.getAttribute('Panel').Loader.show();
            }

            require([
                'package/quiqqer/payment-transactions/bin/backend/controls/windows/Transaction'
            ], function(Window) {
                if (self.getAttribute('Panel')) {
                    self.getAttribute('Panel').Loader.hide();
                }

                new Window({
                    txid: txid
                }).open();
            });
        },

        /**
         * Fetch transactions
         *
         * @return {Promise}
         */
        $getList: function() {
            const self = this;

            return new Promise(function(resolve, reject) {
                QUIAjax.get(
                    'package_quiqqer_payment-transactions_ajax_backend_IncomingPayments_getTransactionList',
                    resolve,
                    {
                        'package': 'quiqqer/payment-transactions',
                        hash: self.getAttribute('hash'),
                        onError: reject
                    }
                );
            });
        },

        $onResize: function() {
            if (!this.$Grid) {
                return;
            }

            const size = this.$GridContainer.getSize();

            this.$Grid.setHeight(size.y - 10);
            this.$Grid.setWidth(size.x - 10);
        }
    });
});