<?php

/**
 * This file contains QUI\ERP\Products\Field\Model
 */

namespace QUI\ERP\Products\Search;

use PDO;
use QUI;
use QUI\ERP\Products\Handler\Fields;
use QUI\ERP\Products\Handler\Search as SearchHandler;
use QUI\ERP\Products\Utils\Tables;
use QUI\Exception;

use function array_unique;
use function array_values;
use function count;
use function get_class;
use function implode;
use function in_array;
use function is_array;
use function is_null;
use function is_numeric;
use function is_string;
use function mb_strpos;
use function mb_substr;
use function md5;
use function sort;

/**
 * Class Search
 *
 * @package QUI\ERP\Products\Search
 */
abstract class Search extends QUI\QDOM
{
    /**
     * All fields that are used in the search
     *
     * @var array|null
     */
    protected ?array $searchFields = null;

    /**
     * All fields that are eligible for search
     *
     * @var ?array
     */
    protected ?array $eligibleFields = null;

    /**
     * Search language
     *
     * @var ?string
     */
    protected ?string $lang = null;

    /**
     * All search types that need cached values
     *
     * @var array
     */
    protected array $searchTypesWithValues = [
        SearchHandler::SEARCHTYPE_INPUTSELECTRANGE,
        SearchHandler::SEARCHTYPE_INPUTSELECTSINGLE,
        SearchHandler::SEARCHTYPE_CHECKBOX_LIST,
        SearchHandler::SEARCHTYPE_SELECTMULTI,
        SearchHandler::SEARCHTYPE_SELECTRANGE,
        SearchHandler::SEARCHTYPE_SELECTSINGLE
    ];

    /**
     * Return all fields with values and labels (used for building search control)
     *
     * @return array
     */
    abstract public function getSearchFieldData(): array;

    /**
     * Return all fields that can be used in this search with search status (active/inactive)
     *
     * @return array
     */
    abstract public function getSearchFields(): array;

    /**
     * Set fields that are searchable
     *
     * @param array $searchFields
     * @return array - search fields
     */
    abstract public function setSearchFields(array $searchFields): array;

    /**
     * Return all fields that are searchable
     *
     * Searchable Field = Is of a field type that is generally searchable +
     *                      field is public
     *
     * @return array
     */
    abstract public function getEligibleSearchFields(): array;

    /**
     * Execute product search
     *
     * @param array $searchParams - search parameters
     * @param bool $countOnly (optional) - return count of search results only [default: false]
     * @return array|int - product ids or count result
     * @throws QUI\Exception
     */
    abstract public function search(array $searchParams, bool $countOnly = false): array | int;

    /**
     * Gets all unique field values for a specific Field
     *
     * @param QUI\ERP\Products\Field\Field $Field
     * @param bool $activeProductsOnly (optional) - only get values from active products
     * @param int[]|null $categoryIds (optional) - limit values to these product categories
     *
     * @return array - unique field values
     * @throws Exception
     */
    protected function getValuesFromField(
        QUI\ERP\Products\Field\Field $Field,
        bool $activeProductsOnly = true,
        ?array $categoryIds = null
    ): array {
        $cname = 'products/search/backend/fieldvalues/';
        $cname .= $Field->getId() . '/';
        $cname .= $this->lang . '/';
        $cname .= $activeProductsOnly ? 'active' : 'inactive';

        if (is_array($categoryIds) && count($categoryIds)) {
            $cname .= '/' . implode('_', $categoryIds);
        }

        try {
            return Cache::get($cname);
        } catch (QUI\Exception) {
            // nothing, retrieve values
        }

        $values = [];
        $column = SearchHandler::getSearchFieldColumnName($Field);

        $select = [];
        $where = [];
        $binds = [];
        $groupBy = [];

        if (!empty($categoryIds)) {
            $whereOr = [];

            foreach ($categoryIds as $categoryId) {
                $whereOr[] = '`category` LIKE :cat' . $categoryId;

                $binds['cat' . $categoryId] = [
                    'value' => '%,' . $categoryId . ',%',
                    'type' => PDO::PARAM_STR
                ];
            }

            $where[] = '(' . implode(' OR ', $whereOr) . ')';
        }

        if ($activeProductsOnly) {
            $where[] = '`active` = 1';
        }

        // special queries depending on search type
        switch ($Field->getSearchDataType()) {
            case SearchHandler::SEARCHDATATYPE_NUMERIC:
                switch ($Field->getSearchType()) {
                    case SearchHandler::SEARCHTYPE_SELECTSINGLE:
                    case SearchHandler::SEARCHTYPE_INPUTSELECTSINGLE:
                    case SearchHandler::SEARCHTYPE_SELECTRANGE:
                    case SearchHandler::SEARCHTYPE_INPUTSELECTRANGE:
                        switch ($Field->getId()) {
                            case Fields::FIELD_PRICE:
                                $select[] = 'MIN(`minPrice`)';
                                $select[] = 'MAX(`maxPrice`)';
                                break;

                            default:
                                $select[] = 'MIN(`' . $column . '`)';
                                $select[] = 'MAX(`' . $column . '`)';
                        }

                        break;
                }

                break;

            case SearchHandler::SEARCHDATATYPE_TEXT:
                switch ($Field->getSearchType()) {
                    case SearchHandler::SEARCHTYPE_SELECTSINGLE:
                    case SearchHandler::SEARCHTYPE_INPUTSELECTSINGLE:
                        $select[] = $column;
                        $groupBy[] = $column;
                        break;

                    case SearchHandler::SEARCHTYPE_CHECKBOX_LIST:
                    case SearchHandler::SEARCHTYPE_SELECTMULTI:
                        $select[] = $column;
                        break;
                }

                break;
        }

        // Build query
        $query = "SELECT " . implode(',', $select);
        $query .= " FROM " . Tables::getProductCacheTableName();

        if (!empty($where)) {
            $query .= " WHERE " . implode(' AND ', $where);
        }

        if (!empty($groupBy)) {
            $query .= " GROUP BY " . implode(',', $groupBy);
        }

        $PDO = QUI::getDataBase()->getPDO();
        $Stmt = $PDO->prepare($query);

        // bind search values
        foreach ($binds as $var => $bind) {
            if (!str_contains($var, ':')) {
                $var = ':' . $var;
            }

            $Stmt->bindValue($var, $bind['value'], $bind['type']);
        }

        try {
            $Stmt->execute();
            $result = $Stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            return [];
        }

        if (empty($result)) {
            return $values;
        }

        switch ($Field->getSearchDataType()) {
            case SearchHandler::SEARCHDATATYPE_NUMERIC:
                $values = match ($Field->getId()) {
                    Fields::FIELD_PRICE => $Field->calculateValueRange(
                        $result[0]['MIN(`minPrice`)'],
                        $result[0]['MAX(`maxPrice`)']
                    ),
                    default => $Field->calculateValueRange(
                        $result[0]['MIN(`' . $column . '`)'],
                        $result[0]['MAX(`' . $column . '`)']
                    ),
                };

                break;

            case SearchHandler::SEARCHDATATYPE_TEXT:
                foreach ($result as $row) {
                    if (empty($row[$column])) {
                        continue;
                    }

                    $values[] = $row[$column];
                }
                break;
        }

        $values = array_values(array_unique($values));
        sort($values);

        Cache::set($cname, $values);

        return $values;
    }

    /**
     * Get where strings and binds with values and PDO datatypes
     *
     * @param array $fieldSearchData
     * @return array - where strings and binds with values and PDO datatypes
     * @throws QUI\Exception
     */
    protected function getFieldQueryData(array $fieldSearchData): array
    {
        $where = [];
        $binds = [];

        foreach ($fieldSearchData as $fieldId => $value) {
            if (empty($value) && !is_numeric($value)) {
                continue;
            }

            try {
                $Field = Fields::getField($fieldId);
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::addWarning(
                    'Product Search :: could not build query data for field #'
                    . $fieldId . ' -> ' . $Exception->getMessage()
                );

                continue;
            }

            // if field is not searchable -> ignore in search
            if (!$this->canSearchField($Field)) {
                continue;
            }

            $isPriceField = false;

            // wenn feld -> price feld
            // scheiss vorgehensweise, wollen aber kein doppelten code
            if (get_class($this) == FrontendSearch::class) {
                $User = QUI::getUserBySession();

                if ($Field->getType() == Fields::TYPE_PRICE) {
                    $isPriceField = true;

                    if (!QUI\ERP\Utils\User::isNettoUser($User) && QUI::isFrontend()) {
                        $Tax = QUI\ERP\Tax\Utils::getTaxByUser(QUI::getUserBySession());
                        $calc = ($Tax->getValue() + 100) / 100;

                        // calc netto sum
                        if (
                            is_array($value)
                            && isset($value['from'])
                            && isset($value['to'])
                            && $calc
                        ) {
                            $Calc = QUI\ERP\Accounting\Calc::getInstance();

                            $value['from'] = $Calc->round($value['from'] / $calc);
                            $value['to'] = $Calc->round($value['to'] / $calc);
                        }
                    }
                }
            }


            $columnName = SearchHandler::getSearchFieldColumnName($Field);
            $column = '`' . $columnName . '`';

            switch ($Field->getSearchType()) {
                case SearchHandler::SEARCHTYPE_HASVALUE:
                    if ($value) {
                        $where[] = $column . ' IS NOT NULL';
                    } else {
                        $where[] = $column . ' IS NULL';
                    }
                    break;

                case SearchHandler::SEARCHTYPE_BOOL:
                    if ($value) {
                        $where[] = $column . ' = 1';
                    } else {
                        $where[] = $column . ' = 0';
                    }
                    break;

                case SearchHandler::SEARCHTYPE_SELECTSINGLE:
                case SearchHandler::SEARCHTYPE_INPUTSELECTSINGLE:
                    if (empty($value)) {
                        break;
                    }

                    if (!is_string($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $where[] = $column . ' = :' . $columnName;
                    $binds[$columnName] = [
                        'value' => $this->sanitizeString($value),
                        'type' => PDO::PARAM_STR
                    ];
                    break;

                case SearchHandler::SEARCHTYPE_SELECTRANGE:
                case SearchHandler::SEARCHTYPE_INPUTSELECTRANGE:
                    if (empty($value)) {
                        break;
                    }

                    if (!is_array($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $from = false;
                    $to = false;

                    if (!empty($value['from'])) {
                        $from = $value['from'];

                        if (!is_string($from) && !is_numeric($from)) {
                            throw new Exception([
                                'quiqqer/products',
                                'exception.search.value.invalid',
                                [
                                    'fieldId' => $Field->getId(),
                                    'fieldTitle' => $Field->getTitle()
                                ]
                            ]);
                        }
                    }

                    if (!empty($value['to'])) {
                        $to = $value['to'];

                        if (!is_string($to) && !is_numeric($to)) {
                            throw new Exception([
                                'quiqqer/products',
                                'exception.search.value.invalid',
                                [
                                    'fieldId' => $Field->getId(),
                                    'fieldTitle' => $Field->getTitle()
                                ]
                            ]);
                        }
                    }

                    if ($from !== false && $to !== false) {
                        if ($from > $to) {
                            $_from = $from;
                            $from = $to;
                            $to = $_from;
                        }
                    }

                    $where = [];

                    if ($from !== false) {
                        if ($isPriceField) {
                            $column = 'minPrice';
                        }

                        $where[] = $column . ' >= :' . $columnName . 'From';

                        $binds[$columnName . 'From'] = [
                            'value' => $this->sanitizeString($from),
                            'type' => PDO::PARAM_STR  // have to use STR here because there is no DECIMAL type
                        ];
                    }

                    if ($to !== false) {
                        if ($isPriceField) {
                            $column = 'maxPrice';
                        }

                        $where[] = $column . ' <= :' . $columnName . 'To';

                        $binds[$columnName . 'To'] = [
                            'value' => $this->sanitizeString($to),
                            'type' => PDO::PARAM_STR  // have to use STR here because there is no DECIMAL type
                        ];
                    }
                    break;

                case SearchHandler::SEARCHTYPE_DATERANGE:
                    if (empty($value)) {
                        break;
                    }

                    if (!is_array($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $from = false;
                    $to = false;

                    if (!empty($value['from'])) {
                        $from = $value['from'];

                        if (!is_numeric($from)) {
                            throw new Exception([
                                'quiqqer/products',
                                'exception.search.value.invalid',
                                [
                                    'fieldId' => $Field->getId(),
                                    'fieldTitle' => $Field->getTitle()
                                ]
                            ]);
                        }
                    }

                    if (!empty($value['to'])) {
                        $to = $value['to'];

                        if (!is_numeric($from)) {
                            throw new Exception([
                                'quiqqer/products',
                                'exception.search.value.invalid',
                                [
                                    'fieldId' => $Field->getId(),
                                    'fieldTitle' => $Field->getTitle()
                                ]
                            ]);
                        }
                    }

                    if ($from !== false && $to !== false) {
                        if ($from > $to) {
                            $_from = $from;
                            $from = $to;
                            $to = $_from;
                        }
                    }

                    $where = [];

                    if ($from !== false) {
                        $where[] = $column . ' >= :' . $columnName . 'From';
                        $binds[$columnName . 'From'] = [
                            'value' => (int)$value,
                            'type' => PDO::PARAM_INT
                        ];
                    }

                    if ($to !== false) {
                        $where[] = $column . ' <= :' . $columnName . 'To';
                        $binds[$columnName . 'To'] = [
                            'value' => (int)$value,
                            'type' => PDO::PARAM_INT
                        ];
                    }
                    break;

                case SearchHandler::SEARCHTYPE_DATE:
                    if (empty($value)) {
                        break;
                    }

                    if (!is_string($value) && !is_numeric($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $where = $column . ' = :' . $columnName;
                    $binds[$columnName] = [
                        'value' => (int)$value,
                        'type' => PDO::PARAM_INT
                    ];
                    break;

                case SearchHandler::SEARCHTYPE_CHECKBOX_LIST:
                case SearchHandler::SEARCHTYPE_SELECTMULTI:
                    if (empty($value)) {
                        break;
                    }

                    if (is_string($value)) {
                        $value = [$value];
                    }

                    if (!is_array($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $whereOr = [];

                    for ($i = 0; $i < count($value); $i++) {
                        $whereOr[] = $column . ' = :' . $columnName . $i;
                        $binds[$columnName . $i] = [
                            'value' => $this->sanitizeString($value[$i]),
                            'type' => PDO::PARAM_STR
                        ];
                    }

                    $where[] = '(' . implode(' OR ', $whereOr) . ')';
                    break;

                case SearchHandler::SEARCHTYPE_TEXT:
                    if (empty($value)) {
                        break;
                    }

                    if (!is_string($value) && !is_numeric($value)) {
                        throw new Exception([
                            'quiqqer/products',
                            'exception.search.value.invalid',
                            [
                                'fieldId' => $Field->getId(),
                                'fieldTitle' => $Field->getTitle()
                            ]
                        ]);
                    }

                    $where[] = $column . ' LIKE :' . $columnName;
                    $binds[$columnName] = [
                        'value' => '%' . $this->sanitizeString($value) . '%',
                        'type' => PDO::PARAM_STR
                    ];
                    break;

                default:
                    throw new Exception([
                        'quiqqer/products',
                        'exception.search.field.unknown.searchtype',
                        [
                            'fieldId' => $Field->getId(),
                            'fieldTitle' => $Field->getTitle()
                        ]
                    ]);
            }
        }

        return [
            'where' => $where,
            'binds' => $binds
        ];
    }

    /**
     * Checks if the currently logged-in user is allowed to search a category field
     *
     * @param QUI\ERP\Products\Field\Field $Field
     * @param QUI\Users\User|null $User (optional)
     * @return bool
     */
    protected function canSearchField(QUI\ERP\Products\Field\Field $Field, null | QUI\Users\User $User = null): bool
    {
        if (is_null($User)) {
            $User = QUI::getUserBySession();
        }

        // calculate group hash
        $userGroups = $User->getGroups(false);
        sort($userGroups);

        $groupHash = md5(implode('', $userGroups));
        $cName = 'products/search/userfieldids/' . $Field->getId() . '/' . $groupHash;

        try {
            return Cache::get($cName);
        } catch (\Exception) {
            // build cache entry
        }

        $canSearch = false;

        if ($Field->isPublic()) {
            $canSearch = true;
        } elseif ($Field->hasViewPermission($User)) {
            $canSearch = true;
        } else {
            $eligibleFields = $this->getEligibleSearchFields();

            foreach ($eligibleFields as $EligibleField) {
                if ($Field->getId() === $EligibleField->getId()) {
                    $canSearch = true;
                    break;
                }
            }
        }

        try {
            Cache::set($cName, $canSearch);
        } catch (QUI\Exception $exception) {
            QUI\System\Log::addError($exception->getMessage());
        }

        return $canSearch;
    }

    /**
     * Sanitizes a string so it can be used for search
     *
     * @param string $str
     * @return float|bool|int|string
     */
    protected function sanitizeString(mixed $str): float | bool | int | string
    {
        if (!is_string($str) && !is_numeric($str)) {
            return false;
        }

        $str = trim($str);

//        $str = Orthos::removeHTML($str);
//        $str = Orthos::clearPath($str);
//        $str = htmlspecialchars_decode($str);
//        $str = str_replace(
//            array(
//                '<',
//                '%3C',
//                '>',
//                '%3E',
//                '"',
//                '%22',
////                '\\',
////                '%5C',
////                '/',
////                '%2F',
//                '\'',
//                '%27',
//            ),
//            '',
//            $str
//        );
//        $str = htmlspecialchars($str);

        return $str;
    }

    /**
     * Filters all fields that are not eligible for use in search
     *
     * @param array $fields - array with Field objects
     * @return array
     */
    protected function filterEligibleSearchFields(array $fields): array
    {
        $eligibleFields = [];

        /** @var QUI\ERP\Products\Field\Field $Field */
        foreach ($fields as $Field) {
            if ($Field->isSearchable() && $Field->getSearchType()) {
                $eligibleFields[] = $Field;
            }
        }

        return $eligibleFields;
    }

    /**
     * Validates an order statement
     *
     * @param array $searchParams - search params
     * @return string - valid order statement
     */
    protected function validateOrderStatement(array $searchParams): string
    {
        $order = 'ORDER BY';

        if (empty($searchParams['sortOn'])) {
            $order .= ' `F' . Fields::FIELD_PRIORITY . '` ASC, `id` ASC';

            return $order;
        }

        $idSort = false;

        // check special fields
        switch ($searchParams['sortOn']) {
            case 'id':
                $idSort = true;
            // no break
            case 'title':
            case 'productNo':
            case 'category':
            case 'active':
            case 'lang':
            case 'tags':
            case 'c_date':
            case 'e_date':
            case 'description':
                $order .= ' ' . $searchParams['sortOn'];
                break;

            case 'priority':
                $order .= ' F' . Fields::FIELD_PRIORITY;
                break;

            default:
                if (mb_strpos($searchParams['sortOn'], 'F') === 0) {
                    $searchParams['sortOn'] = mb_substr($searchParams['sortOn'], 1);
                }

                $orderFieldId = (int)$searchParams['sortOn'];

                try {
                    $OrderField = Fields::getField($orderFieldId);

                    if (!$this->canSearchField($OrderField)) {
                        throw new QUI\Exception();
                    }

                    /**
                     * Special handling when sorting by price (frontend search only)!
                     *
                     * If sorted by price use the currentPrice field to consider special
                     * prices like offer prices.
                     */
                    if ($OrderField->getId() === Fields::FIELD_PRICE && $this instanceof FrontendSearch) {
                        $order .= ' currentPrice';
                    } else {
                        $order .= ' ' . SearchHandler::getSearchFieldColumnName($OrderField);
                    }
                } catch (\Exception) {
                    // if field does not exist or throws some other kind of error - it is not searchable
                    $order .= ' F' . Fields::FIELD_PRIORITY;

                    return $order;
                }
        }

        if (empty($searchParams['sortBy'])) {
            $order .= ' ASC';

            return $order;
        }

        $order .= match ($searchParams['sortBy']) {
            'ASC', 'DESC' => " " . $searchParams['sortBy'],
            default => " ASC",
        };

        if (!$idSort) {
            $order .= ', `id` ASC';
        }

        return $order;
    }

    /**
     * Build the query for the tag groups
     *
     * @param array $tags
     * @return array
     * @throws Exception
     */
    protected function getTagQuery(array $tags): array
    {
        $Tags = new QUI\Tags\Manager(QUI::getRewrite()->getProject());
        $list = [];

        $where = '';
        $binds = [];
        $whereGroups = [];

        $i = 0;

        $isInList = function ($tag, $list) {
            foreach ($list as $tags) {
                if (in_array($tag, $tags)) {
                    return true;
                }
            }

            return false;
        };

        foreach ($tags as $tag) {
            $groups = $Tags->getGroupsFromTag($tag);

            if (empty($groups)) {
                $whereGroups[] = '`tags` LIKE :tag' . $i;

                $binds['tag' . $i] = [
                    'value' => '%,' . $tag . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $i++;

                continue;
            }

            foreach ($groups as $group) {
                if (!$isInList($tag, $list)) {
                    $list[$group['id']][] = $tag;
                }
            }
        }

        foreach ($list as $tags) {
            $tagList = [];

            foreach ($tags as $tag) {
                $tagList[] = '`tags` LIKE :tag' . $i;

                $binds['tag' . $i] = [
                    'value' => '%,' . $tag . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $i++;
            }

            if (count($tagList)) {
                $whereGroups[] = '(' . implode(' OR ', $tagList) . ')';
            }
        }

        if (!empty($whereGroups)) {
            $where = '(' . implode(' AND ', $whereGroups) . ')';
        }

        return [
            'where' => $where,
            'binds' => $binds
        ];
    }
}
