/**
 * @module package/quiqqer/productsearch/bin/controls/products/search/Form
 * @author www.pcsg.de (Henning Leutz)
 *
 * Backend suche für produkte, nur das suchformular
 *
 * @event onSearchBegin [this]
 */
define('package/quiqqer/productsearch/bin/controls/products/search/Form', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/buttons/Button',
    'qui/controls/toolbar/Bar',
    'qui/controls/toolbar/Tab',
    'controls/grid/Grid',
    'package/quiqqer/products/bin/Fields',
    'package/quiqqer/products/bin/controls/categories/Sitemap',
    'Ajax',
    'Locale',
    'Mustache',

    'text!package/quiqqer/productsearch/bin/controls/products/search/Form.html',
    'css!package/quiqqer/productsearch/bin/controls/products/search/Form.css'

], function (QUI, QUIControl, QUIButton, QUIBar, QUITab, Grid, Fields, CategorySitemap, Ajax, QUILocale, Mustache, template) {
    "use strict";

    const lg = 'quiqqer/products';

    return new Class({
        Extends: QUIControl,
        Type   : 'package/quiqqer/productsearch/bin/controls/products/search/Search',

        Binds: [
            'search',
            'resetSearch',
            '$onSubmit',
            '$onInject'
        ],

        options: {
            searchfields   : {},
            searchbutton   : true,
            sortOn         : false,
            sortBy         : false,
            limit          : false,
            sheet          : 1,
            freeTextSearch : true,
            variantChildren: true,
            productTypes   : []
        },

        initialize: function (options) {
            this.parent(options);

            this.$Container = null;
            this.$CatContainer = null;
            this.$TabContainer = null;
            this.$Sitemap = null;
            this.$SearchButton = null;

            this.$loaded = false;

            this.addEvents({
                onCreate: this.$onCreate,
                onInject: this.$onInject
            });
        },

        /**
         * Resize
         *
         * @return {Promise}
         */
        resize: function () {
            if (this.$loaded) {
                return Promise.resolve();
            }

            return new Promise(function (resolve) {
                let interval;

                interval = (function () {
                    if (this.$loaded) {
                        clearInterval(interval);
                        resolve();
                    }
                }.bind(this)).periodical(100);
            }.bind(this));
        },

        /**
         * Create the domnode element
         *
         * @return {HTMLFormElement}
         */
        create: function () {
            this.$Elm = new Element('form', {
                'class': 'quiqqer-products-search-form',
                events : {
                    // submit event, because we have no real submit button
                    keyup: function (event) {
                        if (event.key === 'enter') {
                            this.$onSubmit();
                        }
                    }.bind(this)
                }
            });

            return this.$Elm;
        },

        /**
         * event : on inject
         */
        $onInject: function () {
            const self = this;

            Ajax.get('package_quiqqer_products_ajax_search_backend_getSearchFieldData', function (result) {
                const fieldsIds = result.map(function (Entry) {
                    return Entry.id;
                });

                Fields.getChildren(fieldsIds).then(function (fields) {
                    let i, id, len, fieldId, Field;

                    const fieldFilter = function (Field) {
                        return Field.id == this;
                    };

                    for (i = 0, len = result.length; i < len; i++) {
                        fieldId = result[i].id;
                        Field = fields.filter(fieldFilter.bind(result[i].id))[0];

                        result[i].fieldTitle = Field.title;
                    }

                    this.$Elm.set({
                        html: Mustache.render(template, {
                            fields                  : result,
                            freeTextSearch          : this.getAttribute('freeTextSearch'),
                            variantChildren         : this.getAttribute('variantChildren'),
                            fieldTitleFreeTextSearch: QUILocale.get(lg, 'searchtype.freeTextSearch.title'),
                            text_no_fields          : QUILocale.get(lg, 'searchtypes.empty'),
                            descConsiderChild       : QUILocale.get(lg, 'searchtypes.considerVariantChild')
                        })
                    });

                    this.$Container = this.$Elm.getElement('.quiqqer-products-search-fieldContainer');
                    this.$TabContainer = this.$Elm.getElement('.quiqqer-products-search-tabsContainer');
                    this.$CatContainer = this.$Elm.getElement('.quiqqer-products-search-categoryContainer');

                    const onTabChange = (Tab) => {
                        if (Tab.getAttribute('name') === 'categories') {
                            if (this.$SearchButton) {
                                this.$SearchButton.hide();
                            }

                            if (this.$SearchReset) {
                                this.$SearchReset.hide();
                            }

                            this.$showCategories();
                            return;
                        }

                        this.$showFilter();

                        if (this.$SearchButton) {
                            this.$SearchButton.show();
                        }

                        if (this.$SearchReset) {
                            this.$SearchReset.show();
                        }
                    };

                    this.$TabBar = new QUIBar({
                        slide        : false,
                        'menu-button': false,
                        mousewheel   : false,
                        type         : 'tabbar',
                        width        : '100%',
                        vertical     : false
                    }).inject(this.$TabContainer);

                    this.$TabBar.appendChild(
                        new QUITab({
                            name  : 'categories',
                            text  : 'Kategorien',
                            icon  : 'fa fa-sitemap',
                            events: {
                                onEnter: onTabChange
                            }
                        })
                    );

                    this.$TabBar.appendChild(
                        new QUITab({
                            name  : 'filter',
                            text  : 'Filter',
                            icon  : 'fa fa-filter',
                            events: {
                                onEnter: onTabChange
                            }
                        })
                    );

                    this.$TabBar.firstChild().setActive();

                    this.$Sitemap = new CategorySitemap({
                        multiple: true,
                        events  : {
                            onClick: function () {
                                self.search();
                            }
                        }
                    }).inject(this.$CatContainer);

                    QUI.parse(this.$Elm).then(function () {
                        let Field;

                        const controls = QUI.Controls.getControlsInElement(this.$Container);
                        const Search = this.$Elm.getElement('[name="search"]');

                        const getControlByFieldById = function (fieldId) {
                            for (let c = 0, len = controls.length; c < len; c++) {
                                if (controls[c].getAttribute('fieldid') === fieldId) {
                                    return controls[c];
                                }
                            }
                            return false;
                        };

                        for (i = 0, len = result.length; i < len; i++) {
                            id = result[i].id;

                            if (!Search && i === 0) {
                                getControlByFieldById(result[i].id).focus();
                            }

                            if (!("searchData" in result[i])) {
                                continue;
                            }

                            Field = getControlByFieldById(result[i].id);

                            if (Field) {
                                Field.setSearchData(result[i].searchData);
                            }
                        }

                        if (Search) {
                            Search.focus();
                        }
                    }.bind(this));

                    if (this.getAttribute('searchbutton')) {
                        this.$SearchButton = new QUIButton({
                            textimage: 'fa fa-search',
                            text     : QUILocale.get('quiqqer/productsearch', 'quiqqer.productsearch.button.filter'),
                            events   : {
                                onClick: this.$onSubmit
                            },
                            styles   : {
                                display : 'none',
                                'float' : 'none',
                                margin  : '10px 0 0 2px',
                                maxWidth: '100%',
                                width   : 'calc(100% - 72px)'
                            }
                        }).inject(this.$Elm);

                        this.$SearchReset = new QUIButton({
                            image : 'fa fa-paint-brush',
                            title : QUILocale.get('quiqqer/productsearch', 'quiqqer.productsearch.button.reset'),
                            events: {
                                onClick: this.resetSearch
                            },
                            styles: {
                                display : 'none',
                                'float' : 'none',
                                margin  : '10px 0 0 2px',
                                maxWidth: '100%',
                                width   : '50px'
                            }
                        }).inject(this.$Elm);

                        this.$Elm.addClass('quiqqer-products-search-form-submitButton');
                    }

                    this.$loaded = true;
                }.bind(this));
            }.bind(this), {
                'package': 'quiqqer/products'
            });
        },

        /**
         * event : on submit
         */
        $onSubmit: function () {
            this.setAttribute('sheet', 1);
            this.search();
        },

        /**
         * Reset all search fields
         */
        resetSearch: function () {
            this.getElm().getElements('[name="show-variant-children"]').set('checked', false);

            const quiIds = this.getElm().getElements('[data-quiid]');

            let i, len, Control;

            for (i = 0, len = quiIds.length; i < len; i++) {
                Control = QUI.Controls.getById(quiIds[i].get('data-quiid'));

                if (typeof Control.reset === 'function') {
                    Control.reset();
                }
            }
        },

        /**
         * Execute the search
         */
        search: function () {
            this.fireEvent('searchBegin', [this]);

            return new Promise((resolve) => {
                let i, len, Field, fieldid, sortOn;

                const searchvalues    = {},
                      FreeText        = this.$Elm.getElement('[name="search"]'),
                      VariantChildren = this.$Elm.getElement('[name="show-variant-children"]');

                switch (this.getAttribute('sortOn')) {
                    case 'price_netto':
                        sortOn = '1';
                        break;

                    case 'status':
                        sortOn = 'active';
                        break;

                    case 'id':
                    case 'productNo':
                    case 'title':
                    case 'priority':
                    case 'description':
                    case 'c_date':
                    case 'e_date':
                        sortOn = this.getAttribute('sortOn');
                        break;

                    default:
                        let so = this.getAttribute('sortOn');

                        // if field is a product custom search field
                        if (typeof so === 'string' && so.indexOf('F') !== -1) {
                            so = so.substring(1);
                            if (/^\d+$/.test(so)) { // only numbers?
                                sortOn = so;
                            }
                        } else {
                            sortOn = '';
                        }
                }

                const params = {
                    sheet : this.getAttribute('sheet'),
                    limit : this.getAttribute('limit'),
                    sortOn: sortOn,
                    sortBy: this.getAttribute('sortBy')
                };

                if (FreeText) {
                    params.freetext = FreeText.value;
                }

                if (VariantChildren) {
                    params.considerVariantChildren = VariantChildren.checked ? 1 : 0;
                }
                
                const controls = QUI.Controls.getControlsInElement(this.$Elm);

                const searchFields = controls.filter(function (Control) {
                    return Control.getType() === 'package/quiqqer/productsearch/bin/controls/search/SearchField';
                });

                for (i = 0, len = searchFields.length; i < len; i++) {
                    Field = searchFields[i];
                    fieldid = Field.getFieldId();

                    searchvalues[fieldid] = Field.getSearchValue();
                }

                params.fields = searchvalues;
                params.productTypes = this.getAttribute('productTypes');

                // categories
                if (this.$Sitemap) {
                    params.categories = this.$Sitemap.getSelected().map(function (Item) {
                        return Item.getAttribute('value');
                    });
                }

                // cleanup fields
                if (typeof params.fields !== 'undefined') {
                    let fields = params.fields;

                    for (let fieldId in fields) {
                        if (!fields.hasOwnProperty(fieldId)) {
                            continue;
                        }

                        if (fields[fieldId] && typeof fields[fieldId].from !== 'undefined' && typeof fields[fieldId].to !== 'undefined') {
                            if (fields[fieldId].from === '0.00' && fields[fieldId].to === '0.00') {
                                delete params.fields[fieldId];
                            }
                        }
                    }
                }

                Ajax.get('package_quiqqer_products_ajax_search_backend_executeForGrid', (result) => {
                    resolve(result);

                    this.fireEvent('search', [
                        this,
                        result
                    ]);
                }, {
                    'package'   : 'quiqqer/products',
                    searchParams: JSON.encode(params)
                });
            });
        },


        $showCategories: function () {
            if (!this.$Container || !this.$CatContainer) {
                return Promise.resolve();
            }

            const self = this;

            return new Promise(function (resolve) {
                moofx(self.$Container).animate({
                    opacity: 0,
                    left   : 20
                }, {
                    duration: 200,
                    callback: function () {
                        self.$Container.setStyle('display', 'none');

                        self.$CatContainer.setStyle('opacity', 0);
                        self.$CatContainer.setStyle('left', -20);
                        self.$CatContainer.setStyle('display', 'inline-block');

                        moofx(self.$CatContainer).animate({
                            opacity: 1,
                            left   : 0
                        }, {
                            duration: 200,
                            callback: resolve
                        });
                    }
                });
            });
        },

        $showFilter: function () {
            if (!this.$Container || !this.$CatContainer) {
                return Promise.resolve();
            }

            const self = this;

            return new Promise(function (resolve) {
                moofx(self.$CatContainer).animate({
                    opacity: 0,
                    left   : -20
                }, {
                    duration: 200,
                    callback: function () {
                        self.$CatContainer.setStyle('display', 'none');

                        self.$Container.setStyle('opacity', 0);
                        self.$Container.setStyle('left', 20);
                        self.$Container.setStyle('display', 'inline-block');

                        moofx(self.$Container).animate({
                            opacity: 1,
                            left   : 0
                        }, {
                            duration: 200,
                            callback: resolve
                        });
                    }
                });
            });
        }
    });
});
