/**
 * @module package/quiqqer/productsearch/bin/controls/products/search/Result
 * @author www.pcsg.de (Henning Leutz)
 *
 * Display the results from a product search
 *
 * @event onRefresh [this, {Object} GridOptions]
 * @event onSubmit [this, {Array} productIds]
 * @event onDblClick [this, {Array} productIds]
 */
define('package/quiqqer/productsearch/bin/controls/products/search/Result', [

    'qui/QUI',
    'qui/controls/Control',
    'controls/grid/Grid',
    'Locale',
    'Ajax',

    'css!package/quiqqer/productsearch/bin/controls/products/search/Result.css'

], function(QUI, QUIControl, Grid, QUILocale, QUIAjax) {
    'use strict';

    const lg = 'quiqqer/products';

    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/productsearch/bin/controls/products/search/Result',

        Binds: [
            '$onInject',
            'submit'
        ],

        options: {
            styles: false,
            sortOn: false,
            sortBy: false,
            perPage: 150,
            page: false
        },

        initialize: function(options) {
            this.$data = null;
            this.$Grid = null;

            this.$GridContainer = null;

            this.parent(options);

            this.addEvents({
                onInject: this.$onInject
            });
        },

        /**
         * Create the domnode element
         *
         * @return {HTMLDirectoryElement}
         */
        create: function() {
            this.$Elm = new Element('div', {
                'class': 'quiqqer-products-search-results',
                html: '<div class="quiqqer-products-search-grid"></div>',
                styles: {
                    'float': 'left',
                    height: '100%',
                    width: '100%'
                }
            });

            this.$GridContainer = this.$Elm.getElement(
                '.quiqqer-products-search-grid'
            );

            if (this.getAttribute('styles')) {
                this.$Elm.setStyles(this.getAttribute('styles'));
            }

            return this.$Elm;
        },

        /**
         * Resize
         *
         * @return {Promise}
         */
        resize: function() {
            const size = this.$Elm.getSize();

            if (!this.$Grid) {
                return Promise.resolve();
            }

            return Promise.all([
                this.$Grid.setHeight(size.y),
                this.$Grid.setWidth(size.x)
            ]);
        },

        /**
         * Set result data to the grid
         *
         * @param {Object} data - grid data
         */
        setData: function(data) {
            let i, len, type, entry, Type;
            const localeActive = QUILocale.get(lg, 'product.search.result.status.active');
            const localeDeActive = QUILocale.get(lg, 'product.search.result.status.deactive');

            for (i = 0, len = data.data.length; i < len; i++) {
                entry = data.data[i];
                type = data.data[i].type;

                Type = new Element('span', {
                    'class': 'fa fa-shopping-bag',
                    title: QUILocale.get(lg, 'product.type.product.title')
                });

                if (type === 'QUI\\ERP\\Products\\Product\\Types\\VariantParent') {
                    Type.setStyle('color', '#8DA290');
                    Type.set('title', QUILocale.get(lg, 'product.type.variant.parent.title'));
                } else {
                    if (type === 'QUI\\ERP\\Products\\Product\\Types\\VariantChild') {
                        Type.setStyle('color', '#BEC7B4');
                        Type.set('title', QUILocale.get(lg, 'product.type.variant.child.title'));
                    }
                }

                data.data[i].type = Type;

                // active status
                data.data[i].status = new Element('span', {
                    'class': parseInt(entry.active) ? 'fa fa-check' : 'fa fa-remove',
                    'title': parseInt(entry.active) ? localeActive : localeDeActive
                });

                // product no
                data.data[i].productNo = entry.productNo;

                if (data.data[i].price_netto || data.data[i].price_netto === 0) {
                    data.data[i].price_netto = new Element('span', {
                        html: data.data[i].price_netto.toFixed(2),
                        'class': 'quiqqer-products-search-results--price-display'
                    });
                } else {
                    data.data[i].price_netto = new Element('span', {
                        html: '---',
                        'class': 'quiqqer-products-search-results--price-display'
                    });
                }

                if (data.data[i].price_offer || data.data[i].price_offer === 0) {
                    data.data[i].price_offer = new Element('span', {
                        html: data.data[i].price_offer.toFixed(2),
                        'class': 'quiqqer-products-search-results--price-display'
                    });
                } else {
                    data.data[i].price_offer = new Element('span', {
                        html: '---',
                        'class': 'quiqqer-products-search-results--price-display'
                    });
                }
            }

            if (!this.$Grid) {
                this.$data = data;
                return;
            }

            this.$Grid.setData(data);
        },

        /**
         * event : on inject
         */
        $onInject: function() {
            this.getProductSearchFields().then((fields) => {
                let columnModel = [
                    {
                        header: QUILocale.get('quiqqer/system', 'id'),
                        dataIndex: 'id',
                        dataType: 'number',
                        width: 50
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'type'),
                        dataIndex: 'type',
                        dataType: 'node',
                        width: 40,
                        className: 'grid-align-center'
                    },
                    {
                        header: '&nbsp;',
                        dataIndex: 'status',
                        dataType: 'node',
                        width: 40,
                        className: 'grid-align-center'
                    },
                    {
                        header: QUILocale.get(lg, 'productNo'),
                        dataIndex: 'productNo',
                        dataType: 'text',
                        width: 100
                    },
                    {
                        header: QUILocale.get('quiqqer/products', 'products.field.4.title'),
                        dataIndex: 'title',
                        dataType: 'text',
                        width: 200
                    },
                    {
                        header: QUILocale.get('quiqqer/products', 'products.field.5.title'),
                        dataIndex: 'description',
                        dataType: 'text',
                        width: 200
                    },
                    {
                        header: QUILocale.get(lg, 'products.product.panel.grid.nettoprice'),
                        dataIndex: 'price_netto',
                        dataType: 'node',
                        width: 100
                    },
                    {
                        header: QUILocale.get(lg, 'products.product.panel.grid.offerprice'),
                        dataIndex: 'price_offer',
                        dataType: 'node',
                        width: 100
                    },
                    {
                        header: QUILocale.get(lg, 'products.product.panel.grid.currency'),
                        dataIndex: 'price_currency',
                        dataType: 'text',
                        width: 60
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'editdate'),
                        dataIndex: 'e_date',
                        dataType: 'text',
                        width: 120
                    },
                    {
                        header: QUILocale.get('quiqqer/system', 'createdate'),
                        dataIndex: 'c_date',
                        dataType: 'text',
                        width: 120
                    },
                    {
                        header: QUILocale.get(lg, 'priority'),
                        dataIndex: 'priority',
                        dataType: 'number',
                        width: 50
                    }
                ];

                fields.forEach((entry) => {
                    // already at the column model
                    switch (entry.id) {
                        case 1: // price
                        case 3: // product no / article no
                        case 4: // title
                        case 5: // short
                        case 18: // sort
                            return;
                    }

                    columnModel.push({
                        header: entry.title,
                        dataIndex: 'F' + entry.id,
                        dataType: 'string',
                        width: 100
                    });
                });

                this.$Grid = new Grid(this.$GridContainer, {
                    pagination: true,
                    multipleSelection: true,
                    perPage: this.getAttribute('perPage'),
                    page: this.getAttribute('page'),
                    sortOn: this.getAttribute('sortOn'),
                    serverSort: true,
                    storageKey: 'quiqqer-productsearch-search-result',
                    configurable: true,
                    columnModel: columnModel,
                    exportData: true
                });

                this.$Grid.addEvents({
                    onRefresh: () => {
                        this.fireEvent('refresh', [
                            this,
                            this.$Grid.options
                        ]);
                    },

                    onDblClick: () => {
                        this.fireEvent('dblClick', [
                            this,
                            this.getSelected()
                        ]);

                        this.submit();
                    },

                    onClick: () => {
                        this.fireEvent('click', [
                            this,
                            this.getSelected()
                        ]);
                    }
                });

                if (this.$data) {
                    this.$Grid.setData(this.$data);
                }

                return this.resize();
            });
        },

        getProductSearchFields: function() {
            return new Promise((resolve) => {
                QUIAjax.get('package_quiqqer_products_ajax_search_backend_getProductSearchFieldsData', resolve, {
                    'package': 'quiqqer/products'
                });
            });
        },

        /**
         * Return the selected product ids
         *
         * @returns {Array}
         */
        getSelected: function() {
            const selected = this.$Grid.getSelectedData();

            if (!selected.length) {
                return [];
            }

            return selected.map(function(entry) {
                return entry.id;
            });
        },

        /**
         * submit the selected data
         *
         * @fires onSelect
         */
        submit: function() {
            this.fireEvent('submit', [
                this,
                this.getSelected()
            ]);
        }
    });
});
