<?php

namespace QUI\ERP\Products\Search;

use QUI;
use QUI\ERP\Products\Handler\Fields;
use QUI\ERP\Products\Utils\Tables as ProductsTables;
use QUI\Package\Package;
use Smarty;

use function explode;
use function implode;
use function json_encode;

/**
 * Class Events
 *
 * Main event handler for quiqqer/productsearch
 */
class Events
{
    /**
     * quiqqer/core: onPackageSetup
     *
     * @param Package $Package
     * @return void
     */
    public static function onPackageSetup(Package $Package): void
    {
        if ($Package->getName() !== 'quiqqer/productsearch') {
            return;
        }

        self::patchProductSearchSettings();
        self::patchProductSearchSiteType();
        self::createDefaultFrontendSearchFieldWeights();

        try {
            self::addFulltextIndices();
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }
    }

    /**
     * Set default search settings if none are set
     *
     * @return void
     * @throws QUI\Exception
     */
    protected static function setDefaultSearchSettings(): void
    {
        try {
            $Conf = QUI::getPackage('quiqqer/productsearch')->getConfig();
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return;
        }

        $search = $Conf->getSection('search');

        // Backend search fields
        if (empty($search['backend'])) {
            $search['backend'] = implode(',', [
                Fields::FIELD_PRODUCT_NO,
                Fields::FIELD_TITLE
            ]);
        }

        if (empty($search['freetext'])) {
            $search['freetext'] = implode(',', [
                Fields::FIELD_PRODUCT_NO,
                Fields::FIELD_TITLE,
                Fields::FIELD_SHORT_DESC,
                Fields::FIELD_KEYWORDS
            ]);
        }

        $Conf->setSection('search', $search);
        $Conf->save();
    }

    /**
     * Takes the "search" config section from quiqqer/products
     * and writes them to the same config section of quiqqer/productsearch
     *
     */
    protected static function patchProductSearchSettings(): void
    {
        try {
            $ProductsConf = QUI::getPackage('quiqqer/products')->getConfig();
            $ProductSearchConf = QUI::getPackage('quiqqer/productsearch')->getConfig();

            $searchConfig = $ProductsConf->getSection('search');

            if (!empty($searchConfig)) {
                if (!empty($searchConfig['backend'])) {
                    $backend = explode(',', $searchConfig['backend']);
                    $searchConfig['backend'] = implode(',', array_unique($backend));
                }

                if (!empty($searchConfig['frontend'])) {
                    $backend = explode(',', $searchConfig['frontend']);
                    $searchConfig['frontend'] = implode(',', array_unique($backend));
                }

                if (!empty($searchConfig['freetext'])) {
                    $backend = explode(',', $searchConfig['freetext']);
                    $searchConfig['freetext'] = implode(',', array_unique($backend));
                }

                $ProductSearchConf->setSection('search', $searchConfig);
                $ProductSearchConf->save();

                $ProductsConf->del('search');
                $ProductsConf->save();
            } else {
                self::setDefaultSearchSettings();
            }
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }
    }

    /**
     * Changes all sites of type "quiqqer/products:type/search"
     * to "quiqqer/productsearch:types/search"
     *
     * @return void
     */
    protected static function patchProductSearchSiteType(): void
    {
        try {
            $Projects = QUI::getProjectManager();
            $projects = $Projects->getProjects(true);

            /** @var QUI\Projects\Project $Project */
            foreach ($projects as $Project) {
                $langs = $Project->getLanguages();

                foreach ($langs as $lang) {
                    try {
                        $EditProject = $Projects->getProject($Project->getName(), $lang);

                        QUI::getDataBase()->update(
                            $EditProject->table(),
                            [
                                'type' => FrontendSearch::SITETYPE_SEARCH
                            ],
                            [
                                'type' => 'quiqqer/products:types/search'
                            ]
                        );
                    } catch (\Exception $Exception) {
                        QUI\System\Log::writeException($Exception);
                    }
                }
            }
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }
    }

    /**
     * Add FULLTEXT indices to specific products cache table columns.
     *
     * @return void
     * @throws QUI\Database\Exception
     */
    private static function addFulltextIndices(): void
    {
        // Check if indices are already created
        $productsCacheTbl = ProductsTables::getProductCacheTableName();
        $sql = 'SHOW INDEX FROM ' . $productsCacheTbl . ' WHERE `Index_Type` = "FULLTEXT"';
        $result = QUI::getDataBase()->fetchSQL($sql);

        $requiredFulltextIndexColumns = [
            'productNo',
            'F' . Fields::FIELD_TITLE,
            'F' . Fields::FIELD_SHORT_DESC
        ];

        $createFulltextQueries = [];

        foreach ($requiredFulltextIndexColumns as $column) {
            if (in_array($column, array_column($result, 'Column_name'))) {
                continue;
            }

            $createFulltextQueries[] = 'ALTER TABLE ' . $productsCacheTbl . ' ADD FULLTEXT(' . $column . ')';
        }

        foreach ($createFulltextQueries as $query) {
            try {
                QUI::getDataBase()->execSQL($query);
            } catch (\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }
    }

    /**
     * Event: on smarty init
     *
     * @param Smarty $Smarty
     * @return void
     */
    public static function onSmartyInit(Smarty $Smarty): void
    {
        $Smarty->registerClass('QUI\ERP\Products\Handler\Fields', '\QUI\ERP\Products\Handler\Fields');
    }

    /**
     * @return void
     * @throws QUI\Exception
     */
    private static function createDefaultFrontendSearchFieldWeights(): void
    {
        $conf = QUI::getPackage('quiqqer/productsearch')->getConfig();
        $fieldWeights = $conf->getSection('frontend_field_weights');

        if (!empty($fieldWeights)) {
            return;
        }

        $conf->setSection('frontend_field_weights', [
            'productNo' => json_encode([
                'priority' => 1000,
                'lengthModifier' => 100
            ]),
            'F' . Fields::FIELD_TITLE => json_encode([
                'priority' => 100,
                'lengthModifier' => 1
            ]),
            'F' . Fields::FIELD_SHORT_DESC => json_encode([
                'priority' => 100,
                'lengthModifier' => 1
            ])
        ]);
        $conf->save();
    }
}
