<?php

/**
 * This file contains QUI\ERP\Products\Search\GlobalFrontendSearch
 */

namespace QUI\ERP\Products\Search;

use PDO;
use QUI;
use QUI\ERP\Products\Handler\Fields;
use QUI\ERP\Products\Handler\Products;
use QUI\ERP\Products\Handler\Search as SearchHandler;
use QUI\ERP\Products\Utils\Tables as TablesUtils;
use QUI\Exception;

use function array_merge;
use function array_unique;
use function count;
use function current;
use function explode;
use function implode;
use function in_array;
use function is_array;
use function is_null;

/**
 * Global, site-independent frontend search - searches ALL products
 *
 * @package QUI\ERP\Products\Search
 */
class GlobalFrontendSearch extends Search
{
    /**
     * Current search language
     *
     * @var string
     */
    protected ?string $lang = null;

    /**
     * FrontendSearch constructor.
     *
     * @param string|null $lang (optional) - if omitted, use QUILocale::getCurrent()
     */
    public function __construct(null | string $lang = null)
    {
        if (is_null($lang)) {
            $lang = QUI::getLocale()->getCurrent();
        }

        $this->lang = $lang;
    }

    /**
     * Execute product search
     *
     * @param array $searchParams - search parameters
     * @param bool $countOnly (optional) - return count of search results only [default: false]
     * @return array|int - product ids
     * @throws QUI\Exception
     */
    public function search(array $searchParams, bool $countOnly = false): array | int
    {
        QUI\Permissions\Permission::checkPermission(
            SearchHandler::PERMISSION_FRONTEND_EXECUTE
        );

        $PDO = QUI::getDataBase()->getPDO();

        $binds = [];
        $where = [];

        if ($countOnly) {
            $sql = "SELECT COUNT(*)";
        } else {
            $sql = "SELECT id";
        }

        $sql .= " FROM " . TablesUtils::getProductCacheTableName();

        $where[] = 'lang = :lang';
        $binds['lang'] = [
            'value' => $this->lang,
            'type' => PDO::PARAM_STR
        ];

        if (!empty($searchParams['category'])) {
            $where[] = '`category` LIKE :category';
            $binds['category'] = [
                'value' => '%,' . (int)$searchParams['category'] . ',%',
                'type' => PDO::PARAM_STR
            ];
        }

        if (!empty($searchParams['categories']) && is_array($searchParams['categories'])) {
            $c = 0;
            $whereCategories = [];

            foreach ($searchParams['categories'] as $categoryId) {
                $whereCategories[] = '`category` LIKE :category' . $c;

                $binds['category' . $c] = [
                    'value' => '%,' . (int)$categoryId . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $c++;
            }

            // @todo das OR als setting (AND oder OR) (ist gedacht für die Navigation)
            $where[] = '(' . implode(' OR ', $whereCategories) . ')';
        }

        if (!isset($searchParams['fields']) && !isset($searchParams['freetext'])) {
            throw new Exception(
                'Wrong search parameters.',
                400
            );
        }

        // freetext search
        if (!empty($searchParams['freetext'])) {
            $whereFreeText = [];
            $value = $this->sanitizeString($searchParams['freetext']);

            // split search value by space
            $freetextValues = explode(' ', $value);

            foreach ($freetextValues as $value) {
                // always search tags
                $whereFreeText[] = '`tags` LIKE :freetextTags';
                $binds['freetextTags'] = [
                    'value' => '%,' . $value . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $searchFields = $this->getSearchFields();

                foreach ($searchFields as $fieldId => $search) {
                    if (!$search) {
                        continue;
                    }

                    $Field = Fields::getField($fieldId);

                    // can only search fields with permission
                    if (!$this->canSearchField($Field)) {
                        continue;
                    }

                    $columnName = SearchHandler::getSearchFieldColumnName($Field);

                    $whereFreeText[] = '`' . $columnName . '` LIKE :freetext' . $fieldId;
                    $binds['freetext' . $fieldId] = [
                        'value' => '%' . $value . '%',
                        'type' => PDO::PARAM_STR
                    ];
                }
            }

            if (count($whereFreeText)) {
                $where[] = '(' . implode(' OR ', $whereFreeText) . ')';
            }
        }

        // tags search
        if (!empty($searchParams['tags']) && is_array($searchParams['tags'])) {
            $data = $this->getTagQuery($searchParams['tags']);

            if (!empty($data['where'])) {
                $where[] = $data['where'];
                $binds = array_merge($binds, $data['binds']);
            }
        }

        // product permissions
        if (Products::usePermissions()) {
            // user
            $User = QUI::getUserBySession();

            $whereOr = [
                '`viewUsersGroups` IS NULL'
            ];

            if ($User->getId()) {
                $whereOr[] = '`viewUsersGroups` LIKE :permissionUser';

                $binds['permissionUser'] = [
                    'value' => '%,u' . $User->getId() . ',%',
                    'type' => PDO::PARAM_STR
                ];
            }

            // user groups
            $userGroupIds = $User->getGroups(false);
            $i = 0;

            foreach ($userGroupIds as $groupId) {
                $whereOr[] = '`viewUsersGroups` LIKE :permissionGroup' . $i;

                $binds['permissionGroup' . $i] = [
                    'value' => '%,g' . $groupId . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $i++;
            }

            $where[] = '(' . implode(' OR ', $whereOr) . ')';
        }

        $where[] = '`active` = 1';

        // retrieve query data for fields
        if (isset($searchParams['fields'])) {
            try {
                $queryData = $this->getFieldQueryData($searchParams['fields']);
                $where = array_merge($where, $queryData['where']);
                $binds = array_merge($binds, $queryData['binds']);
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::addError($Exception->getMessage(), $Exception->getContext());
            }
        }

        // build WHERE query string
        $sql .= " WHERE " . implode(" AND ", $where);

        if (!$countOnly) {
            $sql .= " " . $this->validateOrderStatement($searchParams);
        }

        if (!$countOnly) {
            if (!empty($searchParams['limit']) && isset($searchParams['sheet'])) {
                $Pagination = new QUI\Controls\Navigating\Pagination($searchParams);
                $paginationParams = $Pagination->getSQLParams();
                $queryLimit = QUI\Database\DB::createQueryLimit($paginationParams['limit']);

                foreach ($queryLimit['prepare'] as $bind => $value) {
                    $binds[$bind] = [
                        'value' => $value[0],
                        'type' => $value[1]
                    ];
                }

                $sql .= " " . $queryLimit['limit'];
            } elseif (isset($searchParams['limit'])) {
                $queryLimit = QUI\Database\DB::createQueryLimit($searchParams['limit']);

                foreach ($queryLimit['prepare'] as $bind => $value) {
                    $binds[$bind] = [
                        'value' => $value[0],
                        'type' => $value[1]
                    ];
                }

                $sql .= " " . $queryLimit['limit'];
            } else {
                $sql .= " LIMIT 20"; // @todo as settings
            }
        }

        $Stmt = $PDO->prepare($sql);

        // bind search values
        foreach ($binds as $var => $bind) {
            if (!str_contains($var, ':')) {
                $var = ':' . $var;
            }

            $Stmt->bindValue($var, $bind['value'], $bind['type']);
        }

        try {
            $Stmt->execute();
            $result = $Stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Exception $Exception) {
            QUI\System\Log::addError($Exception->getMessage());

            if ($countOnly) {
                return 0;
            }

            return [];
        }

        if ($countOnly) {
            return (int)current(current($result));
        }

        $productIds = [];

        foreach ($result as $row) {
            $productIds[] = $row['id'];
        }

        return $productIds;
    }

    /**
     * Return all fields that are used in the search
     *
     * @return array
     */
    public function getSearchFieldData(): array
    {
        return [];
    }

    /**
     * Return all fields that can be used in this search with search status (active/inactive)
     *
     * @return array
     * @throws Exception
     */
    public function getSearchFields(): array
    {
        $searchFields = [];
        $PackageCfg = Utils::getConfig();
        $searchFieldIdsFromCfg = $PackageCfg->get('search', 'freetext');

        if ($searchFieldIdsFromCfg === false) {
            $searchFieldIdsFromCfg = [];
        } else {
            $searchFieldIdsFromCfg = explode(',', $searchFieldIdsFromCfg);
            $searchFieldIdsFromCfg = array_unique($searchFieldIdsFromCfg);
        }

        $eligibleFields = self::getEligibleSearchFields();

        /** @var QUI\ERP\Products\Field\Field $Field */
        foreach ($eligibleFields as $Field) {
            if (!in_array($Field->getId(), $searchFieldIdsFromCfg)) {
                $searchFields[$Field->getId()] = false;
                continue;
            }

            $searchFields[$Field->getId()] = true;
        }

        return $searchFields;
    }

    /**
     * Set fields that are searchable
     *
     * @param array $searchFields
     * @return array - search fields
     * @throws Exception
     */
    public function setSearchFields(array $searchFields): array
    {
        $currentSearchFields = $this->getSearchFields();
        $newSearchFieldIds = [];

        foreach ($currentSearchFields as $fieldId => $search) {
            if (
                isset($searchFields[$fieldId])
                && $searchFields[$fieldId]
            ) {
                $newSearchFieldIds[] = $fieldId;
            } else {
                unset($currentSearchFields[$fieldId]);
            }
        }

        $PackageCfg = Utils::getConfig();

        $PackageCfg->set(
            'search',
            'freetext',
            implode(',', array_unique($newSearchFieldIds))
        );

        $PackageCfg->save();

        return $this->getSearchFields();
    }

    /**
     * Return all fields that are eligible for search
     *
     * Eligible Field = Is of a field type that is generally searchable +
     *                      field is public
     *
     * @return array
     */
    public function getEligibleSearchFields(): array
    {
        if (!is_null($this->eligibleFields)) {
            return $this->eligibleFields;
        }

        $fields = Fields::getStandardFields();
        $this->eligibleFields = $this->filterEligibleSearchFields($fields);

        return $this->eligibleFields;
    }
}
