<?php

/**
 * This file contains QUI\ERP\Products\Search\FrontendSearch
 */

namespace QUI\ERP\Products\Search;

use PDO;
use QUI;
use QUI\ERP\Products\Handler\Categories;
use QUI\ERP\Products\Handler\Fields;
use QUI\ERP\Products\Handler\Manufacturers;
use QUI\ERP\Products\Handler\Products;
use QUI\ERP\Products\Handler\Search as SearchHandler;
use QUI\ERP\Products\Product\Types\VariantChild;
use QUI\ERP\Products\Search\Cache as SearchCache;
use QUI\ERP\Products\Utils\ProductTypes;
use QUI\ERP\Products\Utils\Tables as TablesUtils;
use QUI\Exception;
use QUI\Interfaces\Users\User;

use function array_column;
use function array_merge;
use function array_search;
use function array_unique;
use function array_values;
use function array_walk;
use function boolval;
use function count;
use function explode;
use function implode;
use function in_array;
use function is_array;
use function is_null;
use function json_decode;
use function json_encode;
use function mb_strlen;
use function mb_substr;
use function md5;
use function str_replace;
use function trim;

/**
 * Class Search
 *
 * @package QUI\ERP\Products\Search
 */
class FrontendSearch extends Search
{
    const SITETYPE_SEARCH = 'quiqqer/productsearch:types/search';
    const SITETYPE_CATEGORY = 'quiqqer/products:types/category';
    const SITETYPE_LIST = 'quiqqer/productstags:types/list';

    /**
     * Flag how the search should handle variant children
     *
     * @var bool
     */
    protected bool $ignoreVariantChildren = true;
    protected bool $freeTextUsed = false;
    protected bool $freeTextSearchTermNoResults = false;

    protected ?array $searchFieldWeights = null;

    /**
     * All site types eligible for frontend search
     *
     * @var array
     */
    protected array $eligibleSiteTypes = [
        self::SITETYPE_CATEGORY => true,
        self::SITETYPE_SEARCH => true,
        self::SITETYPE_LIST => true,
        Manufacturers::SITE_TYPE_MANUFACTURER_LIST => true
    ];

    /**
     * The frontend Site where the search is conducted
     *
     * @var QUI\Interfaces\Projects\Site|null
     */
    protected QUI\Interfaces\Projects\Site | null $Site = null;

    /**
     * Site type of frontend search/category site
     */
    protected ?string $siteType = null;

    /**
     * ID of product category assigned to site
     */
    protected ?int $categoryId = null;

    /**
     * IDs of extra product categories, which the products have to assign
     *
     * @var array
     */
    protected array $categories = [];

    /**
     * Saves sql_mode setting string without "ONLY_FULL_GROUP_BY" directive
     *
     * @var ?string
     */
    protected ?string $sqlMode = null;

    /**
     * @var bool
     */
    protected bool $sqlModeOnlyFullGroupByDisabled = false;

    /**
     * @var ProductTypes
     */
    protected ProductTypes $ProductTypesUtils;

    /**
     * @var array
     */
    protected array $customSearchFields = [];

    /**
     * FrontendSearch constructor.
     *
     * @param QUI\Interfaces\Projects\Site|null $Site - Search Site or Category Site
     * @throws QUI\Exception
     */
    public function __construct(null | QUI\Interfaces\Projects\Site $Site = null)
    {
        $this->lang = QUI::getLocale()->getCurrent();

        if (!empty($Site)) {
            $type = $Site->getAttribute('type');

            if (!is_string($type) || !isset($this->eligibleSiteTypes[$type])) {
                throw new Exception([
                    'quiqqer/products',
                    'exception.frontendsearch.site.type.not.eligible',
                    [
                        'siteId' => $Site->getId()
                    ]
                ]);
            }

            $this->categoryId = (int)$Site->getAttribute('quiqqer.products.settings.categoryId');

            if ($Site->getAttribute('quiqqer.products.settings.extraProductCategories')) {
                $categories = explode(',', $Site->getAttribute('quiqqer.products.settings.extraProductCategories'));

                $this->categories = array_map(function ($categoryId) {
                    return (int)$categoryId;
                }, $categories);
            }

            $this->Site = $Site;
            $this->lang = $Site->getProject()->getLang();
            $this->siteType = $type;
        }

        // global variant settings
        if (QUI::getPackage('quiqqer/products')->getConfig()->get('variants', 'findChildrenInSearch')) {
            $this->ignoreVariantChildren = false;
        }

        $this->ProductTypesUtils = ProductTypes::getInstance();
    }

    /**
     * Execute product search
     *
     * @param array $searchParams - search parameters
     * @param bool $countOnly (optional) - return count of search results only [default: false]
     * @return array|int - product ids
     * @throws QUI\Exception
     */
    public function search(array $searchParams, bool $countOnly = false): array | int
    {
        QUI\Permissions\Permission::checkPermission(
            SearchHandler::PERMISSION_FRONTEND_EXECUTE
        );

        $SearchQueryCollector = new SearchQueryCollector($this, $searchParams);
        QUI::getEvents()->fireEvent('quiqqerProductsFrontendSearchExecute', [$SearchQueryCollector]);

        // Get search params that may have been modified during quiqqerProductsFrontendSearchStart event
        $searchParams = $SearchQueryCollector->getSearchParams();

        $binds = [];
        $where = [];
        $order = false;
        $tbl = TablesUtils::getProductCacheTableName();

        $ProductsConfig = QUI::getPackage('quiqqer/products')->getConfig();

        $findVariantParentsByChildValues = empty($searchParams['ignoreFindVariantParentsByChildValues'])
            && !!(int)$ProductsConfig->get('variants', 'findVariantParentByChildValues');

        // Main query
        $baseSql = "SELECT `id`, `parentId`";

        /**
         * Determine hierarchy used in freetext search.
         *
         * Products should be ordered by matching best with productNo -> title -> description in this order!
         */
        if (!empty($searchParams['freetext'])) {
            $searchTerm = trim($this->sanitizeString($searchParams['freetext']));

            $searchFieldWeights = $this->getFrontendSearchFieldWeights();

            $matchFieldSelects = [];
            $scoreFieldCounter = 1;

            foreach ($searchFieldWeights as $field => $searchField) {
                $selectField = "@score" . $scoreFieldCounter . " := CASE WHEN `$field` IS NOT NULL THEN";
                $selectField .= " (MATCH(`$field`) AGAINST (:searchTerm IN BOOLEAN MODE)";
                $selectField .= " * " . $searchField['priority'];

                if (
                    !empty($searchField['extraFieldWeights']) &&
                    is_array($searchField['extraFieldWeights'])
                ) {
                    foreach ($searchField['extraFieldWeights'] as $extraField => $extraFieldWeight) {
                        $selectField .= " + (`$extraField` * $extraFieldWeight)";
                    }
                }

                $selectField .= " -LENGTH(`$field`) * " . $searchField['lengthModifier'] . ")";
                $selectField .= " ELSE 0 END";
                $selectField .= " as score" . $scoreFieldCounter++;

                $matchFieldSelects[] = $selectField;

                if (mb_strlen($searchTerm) < 4) {
                    // Add custom score for all fields with value < 4 characters (the default MATCH AGAINST limit)
                    $selectFieldSmallFields = "@score" . $scoreFieldCounter . " := CASE";
                    $selectFieldSmallFields .= " WHEN LENGTH(`$field`) < 4";
                    $selectFieldSmallFields .= " AND `$field` = '$searchTerm' THEN 5000";
                    $selectFieldSmallFields .= " WHEN LENGTH(`$field`) < 4";
                    $selectFieldSmallFields .= " AND `$field` LIKE '%$searchTerm%' THEN 1000";
                    $selectFieldSmallFields .= " ELSE 0 END as score" . $scoreFieldCounter++;

                    $matchFieldSelects[] = $selectFieldSmallFields;
                }
            }

            /**
             * Matching hierarchy
             */
            $baseSql .= "," . implode(',', $matchFieldSelects);
            $scoreSelect = [];

            for ($i = 1; $i <= count($matchFieldSelects); $i++) {
                $scoreSelect[] = "@score" . $i;
            }

            $baseSql .= "," . implode('+', $scoreSelect) . ' as score';

            $binds['searchTerm'] = [
                'value' => '*' . $searchTerm . '*',
                'type' => PDO::PARAM_STR
            ];

            $order = "ORDER BY score DESC, id DESC";
        }

        // Table
        $baseSql .= " FROM " . $tbl;

        // WHERE constraints
        $where[] = 'lang = :lang';
        $binds['lang'] = [
            'value' => $this->lang,
            'type' => PDO::PARAM_STR
        ];

        // Filter by category
        if (!isset($searchParams['categoryProductSearchType'])) {
            $searchParams['categoryProductSearchType'] = 'OR';
        }

        $categoryProductSearchType = ' OR ';

        if ($searchParams['categoryProductSearchType'] == 'AND') {
            $categoryProductSearchType = ' AND ';
        }

        $whereCategories = [];
        $c = 0;

        // Determine if the search request should only be within a specific main category
        if ($searchParams['categoryProductSearchType'] === 'AND') {
            if (!empty($searchParams['category'])) {
                $where[] = '`category` LIKE :mainCategory';
                $binds['mainCategory'] = [
                    'value' => '%,' . (int)$searchParams['category'] . ',%',
                    'type' => PDO::PARAM_STR
                ];
            } elseif ($this->categoryId) {
                $where[] = '`category` LIKE :mainCategory';
                $binds['mainCategory'] = [
                    'value' => '%,' . $this->categoryId . ',%',
                    'type' => PDO::PARAM_STR
                ];
            }
        } else {
            $categoryId = false;

            if (!empty($searchParams['category'])) {
                $categoryId = $searchParams['category'];
            } elseif ($this->categoryId) {
                $categoryId = $this->categoryId;
            }

            if ($categoryId && !empty($searchParams['categories']) && is_array($searchParams['categories'])) {
                $searchParams['categories'][] = $categoryId;
            } elseif ($categoryId) {
                $this->categories[] = $categoryId;
            }
        }

        /*
         * If categories are provided in $searchParams, this means the user clicked
         * on a filter in the frontend. In this case, only these categories must be considered.
         * Additionally, categories are always searched with "OR" (instead of "AND", which *can* be the case
         * for the categories set in the product category Site).
         *
         * Otherwise, all categories associated with the product category Site are considered.
         */
        if (!empty($searchParams['categories']) && is_array($searchParams['categories'])) {
            $categoryProductSearchType = ' OR ';

            if ($this->categoryId) {
                if (($key = array_search($this->categoryId, $searchParams['categories'])) !== false) {
                    unset($searchParams['categories'][$key]);
                }
            }

            foreach ($searchParams['categories'] as $categoryId) {
                $whereCategories[] = '`category` LIKE :category' . $c;

                $binds['category' . $c] = [
                    'value' => '%,' . (int)$categoryId . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $c++;
            }
        } else {
            if (!empty($this->categories)) {
                foreach ($this->categories as $categoryId) {
                    $whereCategories[] = '`category` LIKE :extraCategory' . $c;

                    $binds['extraCategory' . $c] = [
                        'value' => '%,' . (int)$categoryId . ',%',
                        'type' => PDO::PARAM_STR
                    ];

                    $c++;
                }
            }
        }

        if (!empty($whereCategories)) {
            $where[] = '(' . implode($categoryProductSearchType, $whereCategories) . ')';
        }

        if (!isset($searchParams['fields']) && !isset($searchParams['freetext'])) {
            throw new Exception(
                'Wrong search parameters.',
                400
            );
        }

        // Freetext search
        if (!empty($searchParams['freetext'])) {
            $searchTerm = trim($this->sanitizeString($searchParams['freetext']));
            $whereFreeText = [];

            // split search value by space
            $freetextValues = [
                $searchTerm
            ];

            if ($this->freeTextSearchTermNoResults) {
                $freetextValues = array_merge(
                    $freetextValues,
                    explode(' ', $searchTerm)
                );
            }

            $valueCounter = 0;

            foreach ($freetextValues as $value) {
                // always search tags
                $whereFreeText[] = '`tags` LIKE :freetextTags';
                $binds['freetextTags'] = [
                    'value' => '%,' . $value . ',%',
                    'type' => PDO::PARAM_STR
                ];

                //$searchFields = $this->getSearchFields();
                $searchFields = Utils::getDefaultFrontendFreeTextFields();

                foreach ($searchFields as $Field) {
                    /* @var $Field QUI\ERP\Products\Field\Field */
                    // can only search fields with permission
                    if (!$this->canSearchField($Field)) {
                        continue;
                    }

                    // Special handling owhen sorting by price
                    $columnName = SearchHandler::getSearchFieldColumnName($Field);

                    $bindNo = $Field->getId() . '_' . $valueCounter++;

                    $whereFreeText[] = '`' . $columnName . '` LIKE :freetext' . $bindNo;
                    $binds['freetext' . $bindNo] = [
                        'value' => '%' . $value . '%',
                        'type' => PDO::PARAM_STR
                    ];
                }
            }

            $where[] = '(' . implode(' OR ', $whereFreeText) . ')';
            $this->freeTextUsed = true;
        }

        // tags search
        $siteTags = $this->Site ?
            $this->Site->getAttribute('quiqqer.products.settings.tags') : false;

        if (!empty($siteTags)) {
            $siteTags = explode(',', trim($siteTags, ','));
        }

        if (!is_array($siteTags)) {
            $siteTags = [];
        }

        if (!empty($searchParams['tags']) && is_array($searchParams['tags'])) {
            $siteTags = array_merge($siteTags, $searchParams['tags']);
        }

        if (!empty($siteTags)) {
            $data = $this->getTagQuery($siteTags);

            if (!empty($data['where'])) {
                $where[] = $data['where'];
                $binds = array_merge($binds, $data['binds']);
            }
        }

        // product permissions
        if (Products::usePermissions()) {
            // user
            $User = QUI::getUserBySession();

            $whereOr = [
                '`viewUsersGroups` IS NULL'
            ];

            if ($User->getId()) {
                $whereOr[] = '`viewUsersGroups` LIKE :permissionUser';

                $binds['permissionUser'] = [
                    'value' => '%,u' . $User->getId() . ',%',
                    'type' => PDO::PARAM_STR
                ];
            }

            // user groups
            $userGroupIds = $User->getGroups(false);
            $i = 0;

            foreach ($userGroupIds as $groupId) {
                $whereOr[] = '`viewUsersGroups` LIKE :permissionGroup' . $i;

                $binds['permissionGroup' . $i] = [
                    'value' => '%,g' . $groupId . ',%',
                    'type' => PDO::PARAM_STR
                ];

                $i++;
            }

            $where[] = '(' . implode(' OR ', $whereOr) . ')';
        }

        $where[] = '`active` = 1';

        // retrieve query data for fields
        if (isset($searchParams['fields'])) {
            try {
                $queryData = $this->getFieldQueryData($searchParams['fields']);
                $where = array_merge($where, $queryData['where']);
                $binds = array_merge($binds, $queryData['binds']);
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::addError($Exception->getMessage(), $Exception->getContext());
            }
        }

        // Add WHERE statements via event
        $where = array_merge($SearchQueryCollector->getWhereStatements(), $where);
        $binds = array_merge($SearchQueryCollector->getBinds(), $binds);

        // Build constraint statements
        $orderConstraint = '';
        $limitConstraint = '';
        $whereConstraint = " WHERE " . implode(" AND ", $where);

        if (!empty($order)) {
            $orderConstraint = " " . $order;
        } elseif (!$countOnly) {
            $orderConstraint = " " . $this->validateOrderStatement($searchParams);
        }

        if (!empty($searchParams['limit']) && !$countOnly) {
            $limit = explode(',', $searchParams['limit']);

            if (!empty($limit[1])) {
                $searchParams['sheet'] = (int)$limit[0] ?: 1;
                $searchParams['limit'] = (int)$limit[1];
            } else {
                if (empty($searchParams['sheet'])) {
                    $searchParams['sheet'] = 1;
                }

                if (empty($searchParams['limit'])) {
                    $searchParams['limit'] = (int)$limit[0];
                }
            }

            if (!empty($searchParams['limitOffset'])) {
                $limitConstraint = " LIMIT " . (int)$searchParams['limitOffset'] . "," . (int)$searchParams['limit'];
            } else {
                $Pagination = new QUI\Controls\Navigating\Pagination($searchParams);
                $sqlParams = $Pagination->getSQLParams();
                $limitConstraint = " LIMIT " . $sqlParams['limit'];
            }
        } else {
            if (!$countOnly) {
                $limitConstraint = " LIMIT " . 20; // @todo: standard-limit als setting auslagern
            }
        }

        // Subquery for variant children
        $queryFindParentsByChildren = false;
        $PDO = QUI::getPDO();

        if ($this->ignoreVariantChildren && $findVariantParentsByChildValues) {
            $binds['variantChildClass'] = [
                'value' => VariantChild::class,
                'type' => PDO::PARAM_STR
            ];

            $queryFindParentsByChildren = $baseSql;
            $queryFindParentsByChildren .= $whereConstraint;
            $queryFindParentsByChildren .= " AND `type` = :variantChildClass";
            $queryFindParentsByChildren .= " GROUP BY `parentId`";
        }

        // Subquery for all normal products (non variants)
        $queryNormalProducts = $baseSql;
        $queryNormalProducts .= $whereConstraint;

        if ($this->ignoreVariantChildren) {
            $binds['variantChildClass'] = [
                'value' => VariantChild::class,
                'type' => PDO::PARAM_STR
            ];

            $queryNormalProducts .= " AND `type` != :variantChildClass";
        }

        // If variant children should be listed in the search results, only
        // allow children whose parents are active!
        if (!$this->ignoreVariantChildren) {
            $variantParentTypes = $this->ProductTypesUtils->getVariantParentProductTypes();

            array_walk($variantParentTypes, function (&$type) {
                $type = mb_substr($type, 1);
                $type = str_replace("\\", "\\\\", $type);
            });

            $variantParentTypes = '\'' . implode('\',\'', $variantParentTypes) . '\'';

            $queryActiveParents = "SELECT `id` FROM " . $tbl . " WHERE `type` IN (" . $variantParentTypes . ")";
            $queryActiveParents .= " AND `active` = 1 AND `lang` = :lang";

            $queryNormalProducts .= " AND (`parentId` IS NULL OR `parentId` IN (" . $queryActiveParents . "))";
        }

        if ($queryFindParentsByChildren) {
            $queryNormalProducts .= " OR (`id`"
                . " IN (SELECT `parentId` FROM (" . $queryFindParentsByChildren . ") as children)";
            $queryNormalProducts .= " AND `active` = 1)";

            $queryNormalProducts .= " GROUP BY `id`";
        }

        // Temporarily disable ONLY_FULL_GROUP_BY mode in MySQL
        $this->disableOnlyFullGroupBySqlMode();

        // Build master search query
        $masterSqlQuery = $queryNormalProducts;
        $masterSqlQuery .= $orderConstraint;
        $masterSqlQuery .= $limitConstraint;

        $Stmt = $PDO->prepare($masterSqlQuery);

        // bind search values
        foreach ($binds as $var => $bind) {
            if (!str_contains($var, ':')) {
                $var = ':' . $var;
            }

            $Stmt->bindValue($var, $bind['value'], $bind['type']);
        }

        try {
            $Stmt->execute();
            $result = $Stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException(
                $Exception,
                QUI\System\Log::LEVEL_ERROR,
                [
                    '$masterSqlQuery' => $masterSqlQuery
                ]
            );

            if ($countOnly) {
                return 0;
            }

            return [];
        } finally {
            // Re-enable ONLY_FULL_GROUP_BY mode in MySQL
            $this->enableOnlyFullGroupBySqlMode();
        }

        // Repeat search if free text search with full search term has not found anything (and then try splitting the search term)
        if ($this->freeTextUsed && !$this->freeTextSearchTermNoResults && empty($result)) {
            $this->freeTextSearchTermNoResults = true;

            // Repeat search and split search term in single parts
            return $this->search($searchParams, $countOnly);
        }

        $productIds = array_column($result, 'id');
        $productIds = array_values(array_unique($productIds));

        if ($countOnly) {
            return count($productIds);
        }

        return $productIds;
    }

    /**
     * Get MySQL mode string without "ONLY_FULL_GROUP_BY" directive
     *
     * @return string|null
     *
     * @throws QUI\Database\Exception
     */
    protected function getCurrentSqlMode(): ?string
    {
        if (!is_null($this->sqlMode)) {
            return $this->sqlMode;
        }

        $sql = "SELECT @@sql_mode as mode";
        $result = QUI::getDataBase()->fetchSQL($sql);
        $this->sqlMode = $result[0]['mode'];

        return $this->sqlMode;
    }

    /**
     * (Re-)Enable "ONLY_FULL_GROUP_BY" MySQL mode for the current session.
     *
     * Only if successfully disabled before.
     *
     * @return void
     */
    protected function enableOnlyFullGroupBySqlMode(): void
    {
        if (!$this->sqlModeOnlyFullGroupByDisabled) {
            return;
        }

        try {
            $sqlMode = $this->getCurrentSqlMode();
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return;
        }

        $modes = explode(',', $sqlMode);
        $modes[] = 'ONLY_FULL_GROUP_BY';
        $mode = implode(',', array_unique($modes));

        if ($mode === '') {
            return;
        }

        try {
            QUI::getDataBase()->execSQL('SET SESSION sql_mode = \'' . $mode . '\'');
            $this->sqlModeOnlyFullGroupByDisabled = false;
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }
    }

    /**
     * Disable "ONLY_FULL_GROUP_BY" MySQL mode for the current session
     *
     * @return void
     */
    protected function disableOnlyFullGroupBySqlMode(): void
    {
        if ($this->sqlModeOnlyFullGroupByDisabled) {
            return;
        }

        try {
            $sqlMode = $this->getCurrentSqlMode();
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return;
        }

        $modes = explode(',', $sqlMode);
        $onlyFullGroupByModeFound = false;

        foreach ($modes as $k => $v) {
            if ($v === 'ONLY_FULL_GROUP_BY') {
                unset($modes[$k]);
                $onlyFullGroupByModeFound = true;
                break;
            }
        }

        if (!$onlyFullGroupByModeFound) {
            return;
        }

        $mode = implode(',', array_values($modes));

        if (empty($mode)) {
            return;
        }

        try {
            QUI::getDataBase()->execSQL('SET SESSION sql_mode = \'' . $mode . '\'');
            $this->sqlModeOnlyFullGroupByDisabled = true;
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }
    }

    /**
     * Return all fields that are used in the search
     *
     * @return array
     * @throws Exception
     */
    public function getSearchFieldData(): array
    {
        try {
            $cname = 'products/search/frontend/searchfielddata/';

            if ($this->Site) {
                $cname .= $this->Site->getId() . '/';
            }

            $cname .= $this->lang . '/';
            $cname .= $this->getGroupHashFromUser();
        } catch (QUI\Exception) {
            return [];
        }

        try {
            return SearchCache::get($cname);
        } catch (QUI\Exception) {
            // nothing, retrieve values
        }

        $searchFieldData = [];
        $parseFields = $this->getSearchFields();
        $restrictToCategoryIds = [];

        if ($this->siteType == self::SITETYPE_CATEGORY) {
            $categorySiteMainCatId = $this->Site->getAttribute(
                'quiqqer.products.settings.categoryId'
            );

            if (!empty($categorySiteMainCatId)) {
                $restrictToCategoryIds[] = (int)$categorySiteMainCatId;
            }

            $extraCategoryIds = $this->Site->getAttribute(
                'quiqqer.products.settings.extraProductCategories'
            );

            if (!empty($extraCategoryIds)) {
                $extraCategoryIds = explode(',', $extraCategoryIds);

                foreach ($extraCategoryIds as $extraCategoryId) {
                    $restrictToCategoryIds[] = (int)$extraCategoryId;
                }
            }
        }

        $Locale = new QUI\Locale();
        $Locale->setCurrent($this->lang);

        foreach ($parseFields as $fieldId => $search) {
            if (!$search) {
                continue;
            }
            try {
                $Field = Fields::getField($fieldId);
            } catch (QUI\ERP\Products\Field\Exception $Exception) {
                QUI\System\Log::addError($Exception->getMessage(), $Exception->getContext());
                continue;
            }

            if (!$this->canSearchField($Field)) {
                continue;
            }

            $searchFieldDataContent = [
                'id' => $Field->getId(),
                'searchType' => $Field->getSearchType(),
                'title' => $Field->getTitle($Locale),
                'description' => $Field->getTitle($Locale)
            ];

            if (in_array($Field->getSearchType(), $this->searchTypesWithValues)) {
                $searchValues = $this->getValuesFromField($Field, true, $restrictToCategoryIds);
                $searchParams = [];

                foreach ($searchValues as $val) {
                    try {
                        $Field->setValue($val);
                        $label = $Field->getValueByLocale($Locale);
                    } catch (QUI\Exception $Exception) {
                        QUI\System\Log::writeException(
                            $Exception,
                            QUI\System\Log::LEVEL_DEBUG
                        );
                        $label = $val;
                    }

                    $searchParams[] = [
                        'label' => $label,
                        'value' => $val
                    ];
                }

                $searchFieldDataContent['searchData'] = $searchParams;
            }

            $searchFieldData[] = $searchFieldDataContent;
        }

        SearchCache::set($cname, $searchFieldData);

        return $searchFieldData;
    }

    /**
     * Return all fields that can be used in this search with search status (active/inactive)
     *
     * @param array $options (optional) - Filter options
     * @return array
     */
    public function getSearchFields(array $options = []): array
    {
        $searchFields = $this->customSearchFields;
        $searchFieldsFromSite = $this->Site ?
            $this->Site->getAttribute('quiqqer.products.settings.searchFieldIds') : false;

        try {
            $eligibleFields = $this->getEligibleSearchFields();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::addError($Exception->getMessage(), $Exception->getContext());
            $eligibleFields = [];
        }

        if (!$searchFieldsFromSite || $searchFieldsFromSite === '{"":true}') {
            $searchFieldsFromSite = [];
        } else {
            $searchFieldsFromSite = json_decode($searchFieldsFromSite, true);
        }

        // quiqqer/erp#25 ... :-/
        if (empty($searchFieldsFromSite)) {
            $defaultFields = QUI\ERP\Products\Search\Utils::getDefaultFrontendFields();

            foreach ($defaultFields as $Field) {
                $searchFields[$Field->getId()] = true;
            }

            return $searchFields;
        }

        /** @var QUI\ERP\Products\Field\Field $Field */
        foreach ($eligibleFields as $Field) {
            $available = true;

            if (!empty($options['showSearchableOnly']) && !$Field->isSearchable()) {
                $available = false;
            }

            if (!isset($searchFieldsFromSite[$Field->getId()])) {
                $available = false;
            }

            if ($available) {
                $searchFields[$Field->getId()] = boolval(
                    $searchFieldsFromSite[$Field->getId()]
                );
            } else {
                $searchFields[$Field->getId()] = false;
            }
        }

        return $searchFields;
    }

    /**
     * Set fields that are searchable
     *
     * @param array $searchFields
     * @return array - search fields
     */
    public function setSearchFields(array $searchFields): array
    {
        $currentSearchFields = $this->getSearchFields();

        foreach ($currentSearchFields as $fieldId => $search) {
            if (isset($searchFields[$fieldId])) {
                $currentSearchFields[$fieldId] = boolval($searchFields[$fieldId]);
            }
        }

        foreach ($searchFields as $fieldId => $search) {
            if (!isset($currentSearchFields[$fieldId])) {
                $this->customSearchFields[$fieldId] = boolval($search);
                $currentSearchFields[$fieldId] = boolval($search);
            }
        }

        if ($this->Site && method_exists($this->Site, 'getEdit')) {
            try {
                $Edit = $this->Site->getEdit();

                $Edit->setAttribute(
                    'quiqqer.products.settings.searchFieldIds',
                    json_encode($currentSearchFields)
                );

                $Edit->save();
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $currentSearchFields;
    }

    /**
     * Set the global default / fallback fields that are searchable
     *
     * @param array $searchFields
     * @return array - search fields
     */
    public static function setGlobalSearchFields(array $searchFields): array
    {
        $GlobalSearch = new QUI\ERP\Products\Search\GlobalFrontendSearch();
        $currentSearchFields = $GlobalSearch->getSearchFields();
        $newSearchFieldIds = [];

        foreach ($currentSearchFields as $fieldId => $search) {
            if (isset($searchFields[$fieldId]) && $searchFields[$fieldId]) {
                $newSearchFieldIds[] = $fieldId;
            } else {
                unset($currentSearchFields[$fieldId]);
            }
        }

        $PackageCfg = Utils::getConfig();

        $PackageCfg->set(
            'search',
            'frontend',
            implode(',', array_unique($newSearchFieldIds))
        );

        try {
            $PackageCfg->save();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }


        // field result
        $searchFields = [];
        $PackageCfg = Utils::getConfig();
        $searchFieldIdsFromCfg = $PackageCfg->get('search', 'frontend');

        if ($searchFieldIdsFromCfg === false) {
            $searchFieldIdsFromCfg = [];
        } else {
            $searchFieldIdsFromCfg = explode(',', $searchFieldIdsFromCfg);
            $searchFieldIdsFromCfg = array_unique($searchFieldIdsFromCfg);
        }

        try {
            $eligibleFields = $GlobalSearch->getEligibleSearchFields();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            $eligibleFields = [];
        }


        /** @var QUI\ERP\Products\Field\Field $Field */
        foreach ($eligibleFields as $Field) {
            if (!in_array($Field->getId(), $searchFieldIdsFromCfg)) {
                $searchFields[$Field->getId()] = false;
                continue;
            }

            $searchFields[$Field->getId()] = true;
        }

        return $searchFields;
    }

    /**
     * Return all fields that are eligible for search
     *
     * Eligible Field = Is of a field type that is generally searchable +
     *                      field is public
     *
     * @return array
     * @throws QUI\Exception
     */
    public function getEligibleSearchFields(): array
    {
        if (!is_null($this->eligibleFields)) {
            return $this->eligibleFields;
        }

        switch ($this->siteType) {
            case self::SITETYPE_CATEGORY:
                $categoryId = $this->Site->getAttribute(
                    'quiqqer.products.settings.categoryId'
                );

                if ($categoryId === false || $categoryId === '') {
                    QUI::getMessagesHandler()->addAttention(
                        QUI::getLocale()->get(
                            'quiqqer/products',
                            'attention.frontendsearch.category.site.no.category',
                            [
                                'siteId' => $this->Site->getId()
                            ]
                        )
                    );

                    $fields = Fields::getStandardFields();
                } else {
                    $Category = Categories::getCategory($categoryId);
                    $fields = $Category->getFields();
                }
                break;

            default:
                $fields = Fields::getStandardFields();
        }

        $this->eligibleFields = $this->filterEligibleSearchFields($fields);

        return $this->eligibleFields;
    }

    /**
     * Gets a unique hash for a user
     *
     * @param User|null $User (optional) - If omitted, User is
     * @return string - md5 hash
     */
    protected function getGroupHashFromUser(null | QUI\Interfaces\Users\User $User = null): string
    {
        if (is_null($User)) {
            $User = QUI::getUserBySession();
        }

        $groups = $User->getGroups(false);
        $groups = implode(',', $groups);

        return md5($groups);
    }

    /**
     * The search considers variant children
     */
    public function considerVariantChildren(): void
    {
        $this->ignoreVariantChildren = false;
    }

    /**
     * The search ignores variant children
     * - Children are therefore not displayed in the search.
     */
    public function ignoreVariantChildren(): void
    {
        $this->ignoreVariantChildren = true;
    }

    protected function getFrontendSearchFieldWeights(): array
    {
        if ($this->searchFieldWeights !== null) {
            return $this->searchFieldWeights;
        }

        try {
            $conf = QUI::getPackage('quiqqer/productsearch')->getConfig();
            $searchFieldWeightsConfigValue = $conf->getSection('frontend_field_weights');
            $searchFieldWeights = [];

            foreach ($searchFieldWeightsConfigValue as $field => $weight) {
                $searchFieldWeights[$field] = json_decode($weight, true);
            }

            $this->searchFieldWeights = $searchFieldWeights;
        } catch (\Exception $exception) {
            QUI\System\Log::writeException($exception);
            $this->searchFieldWeights = [];
        }

        return $this->searchFieldWeights;
    }
}
