/**
 * Suggest search für produkte
 */
define('package/quiqqer/productsearch/bin/controls/frontend/search/MobileSuggest', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/utils/Background',
    'qui/controls/loader/Loader',
    'qui/controls/windows/Popup',
    'qui/utils/System',
    'Ajax',
    'Locale',
    'Mustache',

    'text!package/quiqqer/productsearch/bin/controls/frontend/search/MobileSuggest.html',
    'css!package/quiqqer/productsearch/bin/controls/frontend/search/MobileSuggest.css'

], function (
    QUI,
    QUIControl,
    QUIBackground,
    QUILoader,
    QUIPopup,
    QUISystemUtils,
    QUIAjax,
    QUILocale,
    Mustache,
    template
) {
    "use strict";

    let ios = QUISystemUtils.iOSversion(),
        lg = 'quiqqer/products',
        project = false,
        lang = false,
        siteid = false;

    if (typeof QUIQQER_PROJECT !== 'undefined' && 'name' in QUIQQER_PROJECT) {
        project = QUIQQER_PROJECT.name;
    }

    if (typeof QUIQQER_PROJECT !== 'undefined' && 'lang' in QUIQQER_PROJECT) {
        lang = QUIQQER_PROJECT.lang;
    }

    if (typeof QUIQQER_SITE !== 'undefined' && 'id' in QUIQQER_SITE) {
        siteid = QUIQQER_SITE.id;
    }

    return new Class({

        Extends: QUIPopup,
        Type: 'package/quiqqer/productsearch/bin/controls/frontend/search/MobileSuggest',

        Binds: [
            '$onOpen',
            '$onOpenBegin',
            '$onClose',
            '$onCreate',
            '$keyup',
            '$search',
            '$renderSearch',
            '$showResults',
            '$hideResults',
            '$showLoader',
            '$focusAndOpenKeyboard',
            '$showNoResultsInfo',
            '$hideNoResultsInfo',
            '$setCssAnimationToEntries'
        ],

        options: {
            siteid: siteid,
            project: project,
            lang: lang,
            delay: 500,
            globalsearch: false,
            limit: false,
            showlinktosearchsite: false,
            searchurl: false,
            backgroundopacity: 0.05,
            backgroundstyles: {backgroundColor: 'black'},
            initialanimation: 'slidein' // slidein, fadein, slideInWithFadeIn
        },

        initialize: function (options) {
            this.parent(options);

            this.setAttributes({
                buttons: false,
                header: false,
                backgroundClosable: false
            });

            this.$Background = null;
            this.$Close = null;
            this.$Input = null;
            this.$Result = null;
            this.$ResultCtn = null;
            this.$NoResultsInfo = null;
            this.$searchTerm = '';

            this.$created = false;
            this.$timer = null;
            this.$page = 1;

            this.addEvents({
                onOpen: this.$onOpen,
                onOpenBegin: this.$onOpenBegin,
                onCloseBegin: this.$onClose
            });

            this.Background = new QUIBackground({
                opacity: this.getAttribute('backgroundopacity'),
                styles: this.getAttribute('backgroundstyles')
            });
        },

        /**
         * Create the domnode element
         */
        $onOpenBegin: function () {
            this.getElm().setStyles({
                display: 'none'
            });

            this.$Container = new Element('div', {
                'class': 'quiqqer-products-mobileSuggest',
                html: Mustache.render(template, {
                    placeholder: QUILocale.get('quiqqer/productsearch', 'control.search.suggest.mobile.placeholder'),
                    searchUrl: this.getAttribute('searchurl')
                })
            }).inject(document.body);

            this.$Container.setStyles({
                opacity: 0,
                top: -50
            });

            this.$Close = this.$Container.getElement('.quiqqer-products-mobileSuggest-close');
            this.$Form = this.$Container.getElement('.quiqqer-products-mobileSuggest-form');
            this.$Label = this.$Container.getElement('.quiqqer-products-mobileSuggest-form label');
            this.$Input = this.$Container.getElement('.quiqqer-products-mobileSuggest-form input');
            this.$Result = this.$Container.getElement('.quiqqer-products-mobileSuggest-results');
            this.$ResultCtn = this.$Container.getElement('.quiqqer-products-mobileSuggest-results-container');

            this.$CloseBackground = new Element('div', {
                'class': 'quiqqer-products-mobileSuggest-closeBackground',
                events: {
                    click: function () {
                        this.close();
                    }.bind(this)
                }
            });

            this.$showCloseBg();

            this.$Form.addEvent('submit', function (event) {
                event.stop();
            });

            this.$Close.addEvents({
                click: function () {
                    this.close();
                }.bind(this)
            });

            this.$Input.addEvents({
                keyup: this.$keyup
            });

            if (!ios) {
                this.$Input.set('autofocus', null);
            } else {
                let scrollPos = document.body.getScroll().y;

                if (scrollPos === 0) {
                    const altScrollPos = document.body.getStyle('top').toInt() * -1;

                    if (altScrollPos > 0) {
                        scrollPos = altScrollPos;
                    }
                }

                this.$Container.setStyles({
                    opacity: 0,
                    top: scrollPos
                });
            }

            this.Loader.getElm().setStyle('background', 'transparent');
            this.Loader.inject(this.$ResultCtn);

            this.$created = true;

            return this.$Elm;
        },

        /**
         * event : on open
         */
        $onOpen: function () {
            this.$Container.setStyles({
                zIndex: this.Background.getElm().getStyle('zIndex') + 1
            });

            return new Promise(function (resolve) {
                let top = 0;

                if (ios) {
                    top = this.$Container.getStyle('top');
                }

                moofx(this.$Container).animate({
                    opacity: 1,
                    top: top
                }, {
                    duration: 250,
                    callback: resolve
                });
            }.bind(this)).then(function () {
                if (!ios) {
                    this.$Input.focus();
                } else {
                    this.$focusAndOpenKeyboard();
                }
            }.bind(this));
        },

        /**
         * Close the search
         *
         * @return {Promise}
         */
        $onClose: function () {
            return new Promise(function (resolve) {
                moofx(this.$Container).animate({
                    opacity: 0,
                    top: -50
                }, {
                    duration: 200,
                    callback: function () {
                        this.$Container.destroy();
                        this.$Container = null;
                        resolve();
                    }.bind(this)
                });
            }.bind(this));
        },

        /**
         * Key up event
         *
         * @param event
         */
        $keyup: function (event) {
            const self = this;
            this.$page = 1;

            if (event.key === 'enter') {
                return;
            }

            if (event.key === 'esc') {
                this.close();
                return;
            }

            // no action if there is no text in input
            if (this.$Input.value === '') {
                this.$hideResults();
                return;
            }

            // prevents the search from being executed
            // after action-less keys (alt, shift, ctrl, etc.)
            if (this.$searchTerm === this.$Input.value) {
                return;
            }

            this.$searchTerm = this.$Input.value;

            if (this.$timer) {
                clearTimeout(this.$timer);
            }

            const ShowLoaderAndHideResults = this.$showLoader().then(this.$hideResults);
            this.$showCloseBg();

            this.$hideoResultsInfo().then(() => {
                this.$timer = (function () {
                    if (self.$Input.value === '') {
                        return self.Loader.hide().then(self.$hideResults);
                    }

                    ShowLoaderAndHideResults.then(self.$search).then(self.$renderSearch);
                }).delay(this.getAttribute('delay'), self);
            });
        },

        /**
         * Execute search
         */
        $search: function () {
            return new Promise(function (resolve) {
                QUIAjax.get('package_quiqqer_productsearch_ajax_frontend_suggestRendered', resolve, {
                    'package': 'quiqqer/products',
                    siteId: this.getAttribute('siteid'),
                    project: JSON.encode({
                        name: this.getAttribute('project'),
                        lang: this.getAttribute('lang')
                    }),
                    searchParams: JSON.encode({
                        freetext: this.$Input.value,
                        page: this.$page
                    }),
                    globalsearch: this.getAttribute('globalsearch'),
                    limit: this.getAttribute('limit'),
                    showLinkToSearchSite: this.getAttribute('showlinktosearchsite'),
                    searchUrl: this.getAttribute('searchurl'),
                    showPrice: this.getAttribute('showprice'),
                    showDesc: this.getAttribute('showdesc')
                });
            }.bind(this));
        },

        /**
         * set the results to the dropdown
         *
         * @param {string} data
         * @return {Promise}
         */
        $renderSearch: function (data) {
            if (data === '') {
                this.$Form.appendChild(this.$getNoResultsInfo());

                return this.Loader.hide().then(this.$showNoResultsInfo);
            }

            this.$hideCloseBg();

            this.$Result.set('html', data);

            this.$Result.getElements('li').forEach((LiElm) => {
                LiElm.addEvents({
                    mousedown: function (event) {
                        event.stop();
                    },
                    click: function (event) {
                        let Target = event.target;

                        if (Target.nodeName !== 'LI') {
                            Target = Target.getParent('li');
                        }

                        window.location = Target.get('data-url');
                    }
                });
            });

            this.$Result.getElements('form').addEvent('submit', function (e) {
                e.stop();
            });

            this.$Result
                .getElements('select[name="quiqqer-products-search-sheets"]')
                .addEvent('change', function (e) {
                    this.$page = e.target.value;
                    this.$hideResults(false).then(this.$showLoader).then(this.$search).then(this.$renderSearch);
                }.bind(this));

            const ShowAllResults = this.$Result.getElement('.quiqqer-products-suggestSimple__showAllResults > a');

            if (ShowAllResults) {
                ShowAllResults.addEvent('click', (e) => {
                    e.stop();
                    this.$Form.submit();
                });
            }

            // hide elements
            this.$Result.getElements('.quiqqer-products-suggestSimple--animation').forEach((Elm) => {
                Elm.classList.add('hide');
            })

            return this.$showResults().then(this.$setCssAnimationToEntries);
        },

        /**
         * Show the results dropdown
         *
         * @returns {Promise}
         */
        $showResults: function () {
            return this.Loader.hide().then(function () {
                return new Promise(function (resolve) {
                    moofx(this.$Result).animate({
                        opacity: 1
                    }, {
                        duration: 200,
                        callback: resolve
                    });
                }.bind(this));
            }.bind(this));
        },

        $setCssAnimationToEntries: function () {
            const elms = this.$Result.getElements('.quiqqer-products-suggestSimple--animation'),
                len = elms.length;
            let i = 0;

            for (i; i < len; i++) {
                elms[i].style.setProperty('--delay', (100 * i) + 'ms');
                elms[i].classList.add('fade-in');
            }
        },

        /**
         * Hide the results
         *
         * @returns {Promise}
         */
        $hideResults: function (hideBg = true) {
            return new Promise(function (resolve) {
                if (hideBg) {
                    moofx(this.$Result).animate({
                        opacity: 0
                    }, {
                        duration: 200,
                        callback: function () {
                            resolve();
                        }.bind(this)
                    });
                } else {
                    resolve();
                }
            }.bind(this));
        },

        /**
         * Show the loader
         *
         * @returns {Promise}
         */
        $showLoader: function () {
            return this.Loader.show();
        },

        /**
         * Get no results info HTMLNode
         *
         * @returns {Element}
         */
        $getNoResultsInfo: function () {
            if (this.$NoResultsInfo) {
                return this.$NoResultsInfo;
            }

            this.$NoResultsInfo = new Element('div', {
                'class': 'quiqqer-products-mobileSuggest-noResult',
                text: QUILocale.get(lg, 'message.product.search.empty'),
                styles: {
                    transform: 'translateY(-100%)'
                }
            });

            return this.$NoResultsInfo;
        },

        /**
         * Show no results info
         *
         * @returns {Promise}
         */
        $showNoResultsInfo: function () {
            if (!this.$NoResultsInfo) {
                return Promise.resolve();
            }

            return new Promise((resolve) => {
                moofx(this.$NoResultsInfo).animate({
                    transform: 'translateY(99%)'
                }, {
                    duration: 300,
                    callback: resolve
                })
            })
        },

        /**
         * Hide no results info
         *
         * @returns {Promise}
         */
        $hideoResultsInfo: function () {
            if (!this.$NoResultsInfo) {
                return Promise.resolve();
            }

            return new Promise((resolve) => {
                moofx(this.$NoResultsInfo).animate({
                    transform: 'translateY(0)'
                }, {
                    duration: 300,
                    callback: resolve
                })
            })
        },

        /**
         * Show close bg
         *
         * By clicking on this element (hide bg) the entire search will be closed
         */
        $showCloseBg: function () {
            this.$ResultCtn.appendChild(this.$CloseBackground);
        },

        /**
         * Hide close bg
         *
         * By clicking on this element (hide bg) the entire search will be closed
         */
        $hideCloseBg: function () {
            if (!this.$ResultCtn.contains(this.$CloseBackground)) {
                return;
            }

            this.$ResultCtn.removeChild(this.$CloseBackground);
        },

        $focusAndOpenKeyboard: function () {
            const timeout = 100;
            const self = this;

            // Align temp input element approximately where the input element is
            // so the cursor doesn't jump around
            let TemSearchInput = document.createElement('input');
            TemSearchInput.classList.add('quiqqer-products-mobileSuggest-isFixPlaceholder');
            TemSearchInput.style.opacity = "0";
//            TemSearchInput.setAttribute('autofocus', 'autofocus');
            TemSearchInput.setAttribute('type', 'text');

            // Put this temp element as a child of the page <body> and focus on it
            this.$Label.appendChild(TemSearchInput);
//            document.body.appendChild(TemSearchInput);
            TemSearchInput.focus();
            TemSearchInput.click();

            // The keyboard is open. Now do a delayed focus on the target element
            setTimeout(function () {
                self.$Input.focus();
                self.$Input.click();

                // Remove the temp element
                self.$Label.removeChild(TemSearchInput);
//                document.body.removeChild(TemSearchInput);
            }, timeout);
        }
    });
});
