/**
 * Backend search settings
 */
define('package/quiqqer/productsearch/bin/controls/products/search/Settings', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/buttons/Switch',
    'qui/controls/loader/Loader',
    'controls/grid/Grid',
    'Locale',
    'Ajax',
    'package/quiqqer/products/bin/Fields'

], function (QUI, QUIControl, QUISwitch, QUILoader, Grid, QUILocale, Ajax, Fields) {
    "use strict";

    const lg = 'quiqqer/products';

    if (typeof Array.insert === 'undefined') {
        Array.prototype.insert = function (index) {
            index = Math.min(index, this.length);
            arguments.length > 1 &&
            this.splice.apply(this, [
                index,
                0
            ].concat([].pop.call(arguments))) &&
            this.insert.apply(this, arguments);

            return this;
        };
    }


    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/productsearch/bin/controls/products/search/Settings',

        Binds: [
            '$onImport',
            '$onSwitchChange',
            '$onProductSearchSwitchChange',
            'refresh'
        ],

        options: {
            ids: []
        },

        initialize: function (options) {
            this.parent(options);

            this.$Input = null;
            this.$confGroup = false;
            this.$Loader = new QUILoader();

            this.addEvents({
                onImport: this.$onImport
            });
        },

        /**
         * event : on create
         */
        $onImport: function () {
            this.$Input = this.getElm();
            this.$Input.type = 'hidden';

            this.$confGroup = this.$Input.name;

            // create
            this.$Elm = new Element('div', {
                styles: {
                    'float': 'left',
                    width: '100%'
                }
            }).wraps(this.$Input);

            if (this.$Elm.getParent('.field-container')) {
                new Element('div', {
                    'class': 'field-container-field field-container-field-no-padding'
                }).wraps(this.$Elm);
            }

            this.$Loader.inject(this.$Elm);
            this.$Loader.show();

            // label
            const Row = this.$Input.getParent('.qui-xml-panel-row-item');
            const Label = document.getElement('[for="' + this.$Input.id + '"]');

            if (Label) {
                Label.setStyle('width', '100%');
            }

            if (Row) {
                Row.setStyle('width', '100%');
            }


            // size
            const size = this.$Elm.getSize();

            // grid container
            const Container = new Element('div', {
                styles: {
                    'float': 'left',
                    width: size.x
                }
            }).inject(this.$Elm);

            const columnModel = [
                {
                    header: QUILocale.get(lg, 'settings.window.products.grid.searchstatus'),
                    dataIndex: 'status',
                    dataType: 'QUI',
                    width: 60
                },
                {
                    header: QUILocale.get('quiqqer/system', 'id'),
                    dataIndex: 'id',
                    dataType: 'number',
                    width: 60
                },
                {
                    header: QUILocale.get('quiqqer/system', 'title'),
                    dataIndex: 'title',
                    dataType: 'text',
                    width: 200
                },
                {
                    header: QUILocale.get(lg, 'fieldtype'),
                    dataIndex: 'fieldtype',
                    dataType: 'text',
                    width: 200
                },
                {
                    header: QUILocale.get(lg, 'searchtype'),
                    dataIndex: 'search_type',
                    dataType: 'text',
                    width: 200
                }
            ];

            if (this.$confGroup === 'search.backend') {
                columnModel.insert(1, {
                    header: QUILocale.get('quiqqer/productsearch', 'settings.window.products.grid.productSearch'),
                    dataIndex: 'productSearch',
                    dataType: 'QUI',
                    width: 60
                });

                this.$ProductOverview = new Element('input', {
                    type: 'hidden',
                    name: 'productOverview'
                });
            }

            this.$Grid = new Grid(Container, {
                height: 300,
                width: size.x - 100,
                perPage: 150,
                columnModel: columnModel
            });

            this.$Grid.addEvents({
                refresh: this.refresh
            });

            if (this.$confGroup !== 'search.backend') {
                this.$Loader.show();
                this.$Grid.refresh();
                return;
            }

            Ajax.get('package_quiqqer_products_ajax_search_backend_getProductSearchFields', (result) => {
                let productSearchFieldIds = [];

                for (let i in result) {
                    if (!result.hasOwnProperty(i)) {
                        continue;
                    }

                    if (result[i]) {
                        productSearchFieldIds.push(i);
                    }

                    this.$ProductOverview.value = productSearchFieldIds.join(',');
                }

                this.$Loader.show();
                this.$Grid.refresh();
            }, {
                'package': 'quiqqer/products'
            });
        },

        resize: function () {

        },

        /**
         * Get the available search fields
         *
         * @returns {Promise}
         */
        refresh: function () {
            this.$Loader.show();

            return new Promise((resolve, reject) => {
                Ajax.get('package_quiqqer_products_ajax_search_backend_getSearchFields', (result) => {
                    const fieldIds = Object.keys(result),
                        ids = [],
                        values = this.$Input.value.split(',');

                    let productSearchFields = [];

                    if (this.$confGroup === 'search.backend') {
                        productSearchFields = this.$ProductOverview.value.split(',').map(function (e) {
                            return parseInt(e);
                        });
                    }

                    values.each(function (value) {
                        ids.push(parseInt(value));
                    });

                    this.setAttribute('ids', ids);

                    Fields.getChildren(fieldIds).then((fieldlist) => {
                        const fieldsData = fieldlist.map((entry) => {
                            const Switch = new QUISwitch({
                                status: ids.contains(entry.id),
                                fieldId: entry.id,
                                events: {
                                    onChange: this.$onSwitchChange
                                }
                            });

                            let result = {
                                status: Switch,
                                id: entry.id,
                                title: entry.title,
                                fieldtype: entry.type,
                                search_type: entry.search_type
                            };

                            if (this.$confGroup === 'search.backend') {
                                result.productSearch = new QUISwitch({
                                    type: 'productSearch',
                                    status: productSearchFields.indexOf(entry.id) === -1 ? 0 : 1,
                                    fieldId: entry.id,
                                    events: {
                                        onChange: this.$onProductSearchSwitchChange
                                    }
                                });
                            }

                            return result;
                        });

                        this.$Grid.setData({
                            data: fieldsData
                        });

                        this.$Loader.hide();
                        resolve();
                    });

                }, {
                    'package': 'quiqqer/products',
                    onError: reject
                });
            });
        },

        /**
         * Set fields to the config
         *
         * @param {Array} searchFields
         * @param {boolean} productSearch
         * @returns {Promise}
         */
        setFields: function (searchFields, productSearch) {
            return new Promise((resolve, reject) => {
                let call = '';

                if (typeof productSearch === 'undefined') {
                    productSearch = false;
                }

                switch (this.$confGroup) {
                    case 'search.frontend':
                        call = 'package_quiqqer_products_ajax_search_frontend_setGlobalSearchFields';
                        break;

                    case 'search.backend':
                        call = 'package_quiqqer_products_ajax_search_backend_setSearchFields';
                        break;

                    case 'search.freetext':
                        call = 'package_quiqqer_products_ajax_search_global_setSearchFields';
                        break;

                    default:
                        return reject();
                }

                if (productSearch) {
                    call = 'package_quiqqer_products_ajax_search_backend_setProductSearchFields';
                }

                Ajax.post(call, resolve, {
                    'package': 'quiqqer/products',
                    onError: reject,
                    searchFields: JSON.encode(searchFields)
                });
            });
        },

        /**
         * event : switch status change
         *
         * @param {Object} CurrentSwitch
         */
        $onSwitchChange: function (CurrentSwitch) {
            const controls = QUI.Controls.getControlsInElement(this.$Elm);
            const switches = controls.filter(function (Control) {
                if (Control.getAttribute('type') === 'productSearch') {
                    return false;
                }

                return Control.getType() === 'qui/controls/buttons/Switch';
            });

            let i, len, fieldId;
            const values = {};

            for (i = 0, len = switches.length; i < len; i++) {
                fieldId = switches[i].getAttribute('fieldId');

                values[fieldId] = switches[i].getStatus();
            }

            CurrentSwitch.disable();

            this.setFields(values).then((result) => {
                const keys = [];

                for (let i in result) {
                    if (result.hasOwnProperty(i) && result[i]) {
                        keys.push(i);
                    }
                }

                this.$Input.value = keys.join(',');
                return this.refresh();
            });
        },

        $onProductSearchSwitchChange: function () {
            const controls = QUI.Controls.getControlsInElement(this.$Elm);
            const switches = controls.filter(function (Control) {
                return Control.getAttribute('type') === 'productSearch';
            });

            let i, len, fieldId;
            const values = {};

            for (i = 0, len = switches.length; i < len; i++) {
                fieldId = switches[i].getAttribute('fieldId');

                values[fieldId] = switches[i].getStatus();
            }

            this.setFields(values, true).then((result) => {
                const keys = [];

                for (let i in result) {
                    if (result.hasOwnProperty(i) && result[i]) {
                        keys.push(i);
                    }
                }

                this.$ProductOverview.value = keys.join(',');
                return this.refresh();
            });
        }
    });
});
