/**
 *
 * Ein Feld für die Suche, hohlt sich die search types rein
 *
 * @event onReady [this]
 * @event onChange [this]
 *
 * self::SEARCHTYPE_TEXT,
 * - Input
 *
 * self::SEARCHTYPE_SELECTRANGE,
 * - 2 Select (von bis)
 *
 * self::SEARCHTYPE_SELECTSINGLE,
 * - 1 Select
 *
 * self::SEARCHTYPE_SELECTMULTI,
 * - Select multi
 *
 * self::SEARCHTYPE_BOOL,
 * - Input checkbox (oder Select)
 *
 * self::SEARCHTYPE_HASVALUE,
 * - Select (ja / nein)
 *
 * self::SEARCHTYPE_DATE,
 * - Input date
 *
 * self::SEARCHTYPE_DATERANGE,
 * - 2 Input date (von bis)
 *
 * self::SEARCHTYPE_INPUTSELECTRANGE,
 * - 2 Select (von bis - mit zusätzlicher eingabe -> wie zb mobile.de)
 *
 * self::SEARCHTYPE_INPUTSELECTSINGLE
 * - 1 Select (von bis - mit zusätzlicher eingabe -> wie zb mobile.de)
 */
define('package/quiqqer/productsearch/bin/controls/search/SearchField', [

    'qui/QUI',
    'qui/controls/Control',

    'css!package/quiqqer/productsearch/bin/controls/search/SearchField.css'

], function (QUI, QUIControl) {
    'use strict';

    return new Class({

        Extends: QUIControl,
        Type: 'package/quiqqer/productsearch/bin/controls/search/SearchField',

        Binds: [
            '$onInject',
            'setSearchData'
        ],

        options: {
            searchtype: 'text',
            fieldid: false,
            searchdata: null
        },

        initialize: function (options) {
            this.$Elm = null;
            this.$Input = null;
            this.$Type = null;

            this.$ready = false;
            this.$searchData = null;
            this.$searchValue = null;

            this.parent(options);

            this.addEvents({
                onInject: this.$onInject
            });

            if (this.getAttribute('searchdata')) {
                this.$searchData = this.getAttribute('searchdata');
            }
        },

        /**
         * Return the Field-ID
         *
         * @returns {Number}
         */
        getFieldId: function () {
            return this.getAttribute('fieldid');
        },

        /**
         * Set the focus to the element
         */
        focus: function () {
            if (!this.$Type) {
                if (!this.$ready) {
                    this.addEvent('ready', this.focus);
                }

                return;
            }

            try {
                this.$Type.focus();
            } catch (e) {
                console.error(this.$Type.getType, 'Has no focus method?');
                console.error(e);
            }
        },

        /**
         * Create the domnode element
         *
         * @return {HTMLDivElement}
         */
        create: function () {
            this.$Elm = new Element('div', {
                'class': 'quiqqer-products-search-field',
                'data-quiid': this.getId()
            });

            if (!this.$Input) {
                this.$Input = new Element('input', {
                    type: 'hidden'
                }).inject(this.$Elm);
            }

            return this.$Elm;
        },

        /**
         * event : on inject
         */
        $onInject: function () {
            require([
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/Bool',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/Checkbox',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/CheckboxList',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/Date',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/DateRange',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/HasValue',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/InputSelectRange',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/InputSelectSingle',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectMulti',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectRange',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectSingle',
                'package/quiqqer/productsearch/bin/controls/search/searchtypes/Text'
            ], (Bool, Checkbox, CheckboxList, Date, DateRange, HasValue,
                InputSelectRange, InputSelectSingle, SelectMulti, SelectRange, SelectSingle, Text
            ) => {

                switch (this.getAttribute('searchtype')) {
                    case 'text':
                        this.$Type = new Text().inject(this.getElm());
                        break;

                    case 'checkbox':
                        this.$Type = new Checkbox().inject(this.getElm());
                        break;

                    case 'checkboxList':
                        this.$Type = new CheckboxList().inject(this.getElm());
                        break;

                    case 'selectRange':
                        this.$Type = new SelectRange().inject(this.getElm());
                        break;

                    case 'inputSelectRange':
                        this.$Type = new InputSelectRange().inject(this.getElm());
                        break;

                    case 'selectSingle':
                        this.$Type = new SelectSingle().inject(this.getElm());
                        break;

                    case 'inputSelectSingle':
                        this.$Type = new InputSelectSingle().inject(this.getElm());
                        break;

                    case 'selectMulti':
                        this.$Type = new SelectMulti().inject(this.getElm());
                        break;

                    case 'bool':
                        this.$Type = new Bool().inject(this.getElm());
                        break;

                    case 'hasValue':
                        this.$Type = new HasValue().inject(this.getElm());
                        break;

                    case 'date':
                        this.$Type = new Date().inject(this.getElm());
                        break;

                    case 'dateRange':
                        this.$Type = new DateRange().inject(this.getElm());
                        break;
                }

                if (!this.$Type) {
                    console.error('missing', this.getAttribute('searchtype'));

                    this.$ready = true;
                    this.fireEvent('ready', [this]);
                    return;
                }

                this.$Type.addEvent('onChange', function () {
                    this.fireEvent('change', [this]);
                }.bind(this));

                if (this.$searchValue) {
                    this.$Type.setSearchValue(this.$searchValue);
                }

                if (this.$searchData) {
                    this.$Type.setSearchData(this.$searchData);
                }

                this.$ready = true;
                this.fireEvent('ready', [this]);

            });
        },

        /**
         * Is control ready?
         *
         * @returns {boolean}
         */
        isReady: function () {
            return this.$ready;
        },

        /**
         * Reset the field and set the default value
         *
         * @return {Promise}
         */
        reset: function () {
            if (this.$Type && 'reset' in this.$Type) {
                this.$Type.reset();
            }
        },

        /**
         * Set the search data for the fields
         *
         * @param {object|array} data
         */
        setSearchData: function (data) {
            if (this.$Type) {
                this.$Type.setSearchData(data);
            } else {
                this.$searchData = data;
            }
        },

        /**
         * Set the search value for the fields
         *
         * @param {object|array} data
         */
        setSearchValue: function (data) {
            if (this.$Type) {
                this.$Type.setSearchValue(data);
            } else {
                this.$searchValue = data;
            }
        },

        /**
         * Set the default value for the field
         * -> reset the field
         */
        setSearchDefault: function () {
            if (this.$Type) {
                this.$Type.reset();
            }
        },

        /**
         * Return the search value
         *
         * @returns {Object|Boolean|String|Number}
         */
        getSearchValue: function () {
            if (!this.getAttribute('fieldid')) {
                return false;
            }

            if (!this.$Type) {
                return false;
            }

            let value = this.$Type.getSearchValue();

            if (!value) {
                return value;
            }

            if (typeof this.$Type.$Select !== 'undefined' && this.$Type.$Select) {
                let min = this.$Type.$Select.getAttribute('min');
                let max = this.$Type.$Select.getAttribute('max');

                if (parseFloat(value.from) === parseFloat(min) &&
                    parseFloat(value.to) === parseFloat(max)) {
                    return false;
                }
            }

            return this.$Type.getSearchValue();
        },

        /**
         * Return the value formatted
         *
         * @returns {String}
         */
        getSearchValueFormatted: function () {
            if ('getSearchValueFormatted' in this.$Type) {
                return this.$Type.getSearchValueFormatted();
            }

            return this.getSearchValue().toString();
        }
    });
});
