/**
 * @event onChange [this]
 *
 * mit eingabe vom benutzer
 */
define('package/quiqqer/productsearch/bin/controls/search/searchtypes/InputSelectRange', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/input/Range',
    'Locale'

], function (QUI, QUIControl, QUIRange, QUILocale) {
    "use strict";

    var NumberFormatter = QUILocale.getNumberFormatter({
        style: 'currency',
        currency: window.DEFAULT_CURRENCY || 'EUR'
    });

    return new Class({
        Extends: QUIControl,
        Type: 'package/quiqqer/productsearch/bin/controls/search/searchtypes/InputSelectRange',

        Binds: [
            '$onImport'
        ],

        options: {
            range: false,
            snap: true,
            connect: true,
            value: false
        },

        initialize: function (options) {
            this.$Elm = null;
            this.$Select = null;
            this.$data = {};

            this.parent(options);
        },

        /**
         * Create the domnode element
         *
         * @return {HTMLDivElement}
         */
        create: function () {
            var self = this;

            this.$Select = new QUIRange({
                range: this.getAttribute('range'),
                snap: this.getAttribute('snap'),
                connect: this.getAttribute('connect'),
                styles: {
                    width: '100%'
                },
                Formatter: function () {
                    return self.getSearchValueFormatted();
                },
                events: {
                    change: function () {
                        self.fireEvent('change', [self]);
                    }
                }
            });

            this.$Elm = this.$Select.create();
            this.$Elm.addClass('quiqqer-products-searchtype-selectrange');

            this.refresh();

            return this.$Elm;
        },

        /**
         * Refresh the control
         */
        refresh: function () {
            if (typeOf(this.$data) === 'array') {
                var i, pc;
                var values = this.$data.map(function (entry) {
                    return parseFloat(entry.value);
                });

                values.sort(function (a, b) {
                    return a - b;
                });

                var len = values.length;

                var range = {
                    min: values[0],
                    max: values[len - 1]
                };

                // percent
                var percentStep = 100 / len;

                for (i = 1; i < len - 1; i++) {
                    pc = Math.round(percentStep * i);

                    range[pc + '%'] = values[i];
                }

                if (range.min !== range.max) {
                    this.$Select.setRange(range);
                }

                return;
            }


            if ('from' in this.$data && 'to' in this.$data) {
                if (this.$data.from === this.$data.to) {
                    return;
                }

                this.$Select.setValue([
                    this.$data.from,
                    this.$data.to
                ]);
                return;
            }

            if ('from' in this.$data) {
                this.$Select.setFrom(this.$data.from);
                return;
            }

            if ('to' in this.$data) {
                this.$Select.setTo(this.$data.to);
            }
        },

        /**
         * Reset the field
         */
        reset: function () {
            var min = this.$Select.getAttribute('min'),
                max = this.$Select.getAttribute('max');

            this.setSearchValue({
                from: min,
                to: max
            });
        },

        /**
         * set the search data
         *
         * @param {Object|Array} data
         */
        setSearchData: function (data) {
            if (typeOf(data) !== 'object' && typeOf(data) !== 'array') {
                return;
            }

            this.$data = data;
            this.refresh();

            // value
            if (this.getAttribute('value') !== false) {
                return;
            }

            var values = this.$data.map(function (entry) {
                return parseFloat(entry.value);
            });

            this.$Select.setValue([
                values[0],
                values[values.length - 1]
            ]);

            this.$Select.setAttribute('min', values[0]);
            this.$Select.setAttribute('max', values[values.length - 1]);
            this.refresh();
        },

        /**
         * Set the input select value
         *
         * @param {Array|String|Object} value
         */
        setSearchValue: function (value) {
            if (typeOf(value) === 'object') {
                var from = null, to = null;

                if ("from" in value) {
                    from = value.from;
                }

                if ("to" in value) {
                    to = value.to;
                }

                value = [from, to];
            }

            this.setAttribute('value', value);

            if (this.$Select) {
                this.$Select.setValue(value);
            }
        },

        /**
         * Return the search value
         *
         * @returns {Object|Boolean}
         */
        getSearchValue: function () {
            var value = this.$Select.getValue();

            if (parseFloat(value.from) <= this.$Select.getAttribute('min') &&
                parseFloat(value.to) >= this.$Select.getAttribute('max')) {
                return false;
            }

            return value;
        },

        /**
         * Return the value formatted
         *
         * @returns {string}
         */
        getSearchValueFormatted: function () {
            var value = this.getSearchValue();

            if (!value) {
                value = {
                    from: false,
                    to: false
                };
            }

            if (!value.from) {
                value.from = this.$Select.getAttribute('min');
            }

            if (!value.to) {
                value.to = this.$Select.getAttribute('max');
            }

            return NumberFormatter.format(value.from) +
                ' bis ' + NumberFormatter.format(value.to);
        }
    });
});
