define('package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectRange', [

    'qui/QUI',
    'qui/controls/Control',
    'qui/controls/input/Range',
    'Locale',

    'css!package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectRange.css'

], function (QUI, QUIControl, QUIRange, QUILocale) {
    "use strict";

    const NumberFormatter = QUILocale.getNumberFormatter({
        style: 'currency',
        currency: window.DEFAULT_CURRENCY || 'EUR'
    });

    return new Class({
        Extends: QUIControl,
        Type: 'package/quiqqer/productsearch/bin/controls/search/searchtypes/SelectRange',

        Binds: [
            '$onImport'
        ],

        options: {
            snap: true,
            connect: true
        },

        initialize: function (options) {
            this.$Elm = null;
            this.$Select = null;
            this.$data = {};

            this.parent(options);
        },

        /**
         * Create the domnode element
         *
         * @return {HTMLDivElement}
         */
        create: function () {
            let ignoreFirstFireEventBecauseOfLoading = true;

            this.$Select = new QUIRange({
                range: {
                    min: 0,
                    max: 100
                },

                snap: this.getAttribute('snap'),
                connect: this.getAttribute('connect'),

                styles: {
                    width: '100%'
                },

                Formatter: function () {
                    return this.getSearchValueFormatted();
                }.bind(this),

                events: {
                    change: function () {
                        if (ignoreFirstFireEventBecauseOfLoading) {
                            ignoreFirstFireEventBecauseOfLoading = false;
                            return;
                        }

                        this.fireEvent('change', [this]);
                    }.bind(this)
                }
            });

            this.$Elm = this.$Select.create();
            this.$Elm.addClass('quiqqer-products-searchtype-selectrange');

            this.refresh();

            return this.$Elm;
        },

        /**
         * Refresh the control
         */
        refresh: function () {
            if (typeOf(this.$data) === 'array') {
                let i, pc;
                const values = this.$data.map(function (entry) {
                    return parseFloat(entry.value);
                });

                values.sort(function (a, b) {
                    return a - b;
                });

                let len = values.length;

                const range = {
                    min: values[0],
                    max: values[len - 1]
                };

                // percent
                const percentStep = 100 / len;

                for (i = 1; i < len - 1; i++) {
                    pc = Math.round(percentStep * i);

                    range[pc + '%'] = values[i];
                }

                this.$Select.setRange(range);
                return;
            }


            if ('from' in this.$data && 'to' in this.$data) {
                this.$Select.setValue([
                    this.$data.from,
                    this.$data.to
                ]);
                return;
            }

            if ('from' in this.$data) {
                this.$Select.setFrom(this.$data.from);
                return;
            }

            if ('to' in this.$data) {
                this.$Select.setTo(this.$data.to);
            }
        },

        /**
         * Reset the field
         */
        reset: function () {
            const min = this.$Select.getAttribute('min'),
                max = this.$Select.getAttribute('max');

            this.setSearchValue({
                from: min,
                to: max
            });
        },

        /**
         * Set the input select value
         * @param value
         */
        setSearchValue: function (value) {
            this.setAttribute('value', value);

            this.$Select.setValue([
                value.from,
                value.to
            ]);
        },

        /**
         * set the search data
         *
         * @param {Object|Array} data
         */
        setSearchData: function (data) {
            if (typeOf(data) === 'array') {
                let min = null;
                let max = null;

                for (let i = 0, len = data.length; i < len; i++) {
                    if (typeof data[i].value === 'undefined') {
                        continue;
                    }

                    if (min === null) {
                        min = data[i].value;
                    }

                    if (max === null) {
                        max = data[i].value;
                    }

                    if (min > data[i].value) {
                        min = data[i].value;
                    }

                    if (max < data[i].value) {
                        max = data[i].value;
                    }
                }

                this.$Select.setAttribute('min', min);
                this.$Select.setAttribute('max', max);

                this.$data = data;
                this.refresh();
                this.reset();
                return;
            }

            if (typeOf(data) !== 'object') {
                return;
            }

            this.$data = data;
            this.refresh();
        },

        /**
         * Return the search value
         *
         * @returns {Object}
         */
        getSearchValue: function () {
            return this.$Select.getValue();
        },

        /**
         * Return the value formatted
         *
         * @returns {string}
         */
        getSearchValueFormatted: function () {
            const value = this.getSearchValue();

            return NumberFormatter.format(value.from) +
                ' bis ' + NumberFormatter.format(value.to);
        }
    });
});
