<?php

/**
 * This file contains QUI\ERP\Products\Search\Cache
 */

namespace QUI\ERP\Products\Search;

use DateTime;
use QUI;
use QUI\ERP\Products\Utils\Package as PackageUtils;
use QUI\Exception;
use Stash;

use function file_exists;
use function is_dir;
use function is_int;
use function is_null;
use function md5;

/**
 * Class Search
 *
 * Cache class for product search
 */
class Cache extends QUI\QDOM
{
    /**
     * Cache stash for search cache
     *
     * @var ?Stash\Pool
     */
    protected static ?Stash\Pool $Stash = null;

    /**
     * Set data to product search cache
     *
     * @param string $name
     * @param mixed $data
     * @param DateTime|int|null $time -> sekunden oder datetime
     *
     * @return Stash\Interfaces\ItemInterface
     * @throws Exception
     */
    public static function set(
        string $name,
        mixed $data,
        null | DateTime | int $time = null
    ): Stash\Interfaces\ItemInterface {
        $Item = self::getStashItem($name);
        $Item->set($data);

        if ($time instanceof DateTime) {
            $Item->expiresAt($time);
        } elseif (is_int($time)) {
            $Item->expiresAfter($time);
        }

        return $Item;
    }

    /**
     * Get data from product search cache
     *
     * @param string $name
     * @return string|array<mixed>|object|boolean
     * @throws QUI\Cache\Exception
     */
    public static function get(string $name): object | bool | array | string
    {
        try {
            $Item = self::getStashItem($name);
            $data = $Item->get();
            $isMiss = $Item->isMiss();
        } catch (\Exception $Exception) {
            throw new QUI\Cache\Exception(
                QUI::getLocale()->get(
                    'quiqqer/system',
                    'exception.lib.cache.manager.not.exist'
                ),
                404
            );
        }

        if ($isMiss) {
            throw new QUI\Cache\Exception(
                QUI::getLocale()->get(
                    'quiqqer/system',
                    'exception.lib.cache.manager.not.exist'
                ),
                404
            );
        }

        return $data;
    }

    /**
     * Empty cache
     *
     * @param string|null $key (optional) - if no key given, cache is cleared completely
     * @throws Exception
     */
    public static function clear(?string $key = null): void
    {
        self::getStashItem($key)->clear();

        try {
            self::getStash()->getItem('quiqqer/products')->clear();
        } catch (\Exception $Exception) {
            QUI\System\Log::addError($Exception->getMessage());
        }
    }

    /**
     * Return a specific cache item
     *
     * @param string|null $key (optional) - cache name / cache key
     * @return Stash\Interfaces\ItemInterface
     * @throws Exception
     */
    protected static function getStashItem(null | string $key = null): Stash\Interfaces\ItemInterface
    {
        if (is_null($key)) {
            $key = md5(__FILE__) . '/products/';
        } else {
            $key = md5(__FILE__) . '/products/' . $key;
        }

        return self::getStash()->getItem($key);
    }

    /**
     * Get product cache stash
     *
     * @return Stash\Pool
     * @throws QUI\Exception
     */
    protected static function getStash(): Stash\Pool
    {
        if (!is_null(self::$Stash)) {
            return self::$Stash;
        }

        $cacheDir = self::getCacheDir();

        try {
            $Stash = new Stash\Pool(
                QUI\Cache\Manager::getDriver([
                    'path' => $cacheDir
                ])
            );
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            throw new Exception([
                'quiqqer/products',
                'exception.searchcache.initialize.error',
                [
                    'error' => $Exception->getMessage()
                ]
            ]);
        }

        self::$Stash = $Stash;

        return self::$Stash;
    }

    /**
     * Get base cache dir
     *
     * @return string
     */
    protected static function getCacheDir(): string
    {
        $cacheDir = PackageUtils::getVarDir() . 'cache/products/search/';

        if (!file_exists($cacheDir) || !is_dir($cacheDir)) {
            QUI\Utils\System\File::mkdir($cacheDir);
        }

        return $cacheDir;
    }
}
