<?php

/**
 * This file contains QUI\ERP\Products\Search\Controls\Suggest
 */

namespace QUI\ERP\Products\Search\Controls;

use QUI;
use QUI\Database\Exception;
use QUI\ERP\Products\Search\FrontendSearch;
use QUI\Interfaces\Projects\Site;
use QUI\Projects\Project;

use function dirname;

/**
 * Class Suggest
 */
class Suggest extends QUI\Control
{
    /**
     * constructor
     *
     * @param array<string, mixed> $attributes
     */
    public function __construct(array $attributes = [])
    {
        $this->setAttributes([
            'Site' => false,
            'Project' => false,
            'data-qui' => 'package/quiqqer/productsearch/bin/controls/frontend/search/Suggest',
            'hideOnProductSearch' => false,
            'globalsearch' => false,
            'limit' => 0, // int
            'showPrice' => '', // true / false
            'showDesc' => '', // true / false
            'showLinkToSearchSite' => '' // true / false
        ]);

        $this->addCSSFile(dirname(__FILE__) . '/Suggest.css');
        $this->addCSSClass('quiqqer-products-search-suggest');

        parent::__construct($attributes);
    }

    /**
     * (non-PHPdoc)
     *
     * @throws QUI\Exception
     * @see \QUI\Control::create()
     */
    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();
        $Site = $this->getSite();
        $Search = $this->getSite();
        $Config = QUI::getPackage('quiqqer/productsearch')->getConfig();

        if ($Config === null) {
            throw new QUI\Exception('Config "quiqqer/productsearch" could not be loaded.');
        }


        if ($Site->getAttribute('quiqqer.products.settings.showFreeText')) {
            return '';
        }

        if ($this->getAttribute('globalsearch')) {
            $this->setAttribute('data-qui-options-globalsearch', 1);
            $Search = $this->getSearch();
        }

        // config
        $limit = intval($Config->get('frontendSuggestSearch', 'limit'));
        $showPrice = $Config->get('frontendSuggestSearch', 'showPrice');
        $showDesc = $Config->get('frontendSuggestSearch', 'showDesc');
        $showLinkToSearchSite = $Config->get('frontendSuggestSearch', 'showLinkToSearchSite');

        if (intval($this->getAttribute('limit')) > 0) {
            $limit = intval($this->getAttribute('limit'));
        }

        if ($limit <= 0) {
            $limit = 10;
        }

        if ($this->getAttribute('showPrice') !== '') {
            $showPrice = $this->getAttribute('showPrice');
        }

        if ($this->getAttribute('showDesc') !== '') {
            $showDesc = $this->getAttribute('showDesc');
        }

        if ($this->getAttribute('showLinkToSearchSite') !== '') {
            $showLinkToSearchSite = $this->getAttribute('showLinkToSearchSite');
        }

        $this->setJavaScriptControlOption('searchurl', $Search->getUrlRewritten());
        $this->setJavaScriptControlOption('limit', $limit);
        $this->setJavaScriptControlOption('showlinktosearchsite', intval($showLinkToSearchSite));
        $this->setJavaScriptControlOption('showprice', intval($showPrice));
        $this->setJavaScriptControlOption('showdesc', intval($showDesc));

        $Engine->assign([
            'this' => $this,
            'Site' => $this->getSite(),
            'Search' => $Search
        ]);

        return $Engine->fetch(dirname(__FILE__) . '/Suggest.html');
    }

    /**
     * Return the current site
     *
     * @return QUI\Interfaces\Projects\Site
     *
     * @throws QUI\Exception
     */
    protected function getSite(): QUI\Interfaces\Projects\Site
    {
        $Site = $this->getAttribute('Site');

        if ($Site) {
            switch ($Site->getAttribute('type')) {
                case FrontendSearch::SITETYPE_SEARCH:
                case FrontendSearch::SITETYPE_CATEGORY:
                    return $Site;
            }
        }

        $Project = $this->getProject();

        $search = $Project->getSites([
            'where' => [
                'type' => FrontendSearch::SITETYPE_SEARCH
            ],
            'limit' => 1
        ]);

        if (isset($search[0])) {
            $this->setAttribute('Site', $search[0]);
        } else {
            $this->setAttribute('Site', $Project->firstChild());
        }

        return $this->getAttribute('Site');
    }

    /**
     * Return the global search
     *
     * @return Site
     *
     * @throws Exception
     * @throws QUI\Exception
     */
    protected function getSearch(): QUI\Interfaces\Projects\Site
    {
        $Project = $this->getProject();

        $search = $Project->getSites([
            'where' => [
                'type' => FrontendSearch::SITETYPE_SEARCH
            ],
            'limit' => 1
        ]);

        if (isset($search[0])) {
            return $search[0];
        }

        return $this->getSite();
    }

    /**
     * Return the current project
     *
     * @return Project
     *
     * @throws QUI\Exception
     */
    protected function getProject(): QUI\Projects\Project
    {
        $ProjectAttribute = $this->getAttribute('Project');

        if ($ProjectAttribute instanceof Project) {
            return $ProjectAttribute;
        }

        $Project = QUI::getRewrite()->getProject();

        if ($Project === null) {
            throw new QUI\Exception('Project could not be resolved from rewrite context.');
        }

        return $Project;
    }
}
