<?php

namespace QUI\ERP\Products\Search;

/**
 * Class SearchQueryCollector
 *
 * Dynamically add search query instructions to a product search
 */
class SearchQueryCollector
{
    /**
     * WHERE statements
     *
     * @var array<int, string>
     */
    protected array $where = [];

    /**
     * PDO binds
     *
     * @var array<string, mixed>
     */
    protected array $binds = [];

    /**
     * The search params that are applied to the search the
     * SearchQueryCollector collects statements for
     *
     * @var array<string, mixed>
     */
    protected array $searchParams = [];

    /**
     * @var Search
     */
    protected Search $Search;

    /**
     * SearchQueryCollector constructor.
     *
     * @param Search $Search
     * @param array<string, mixed> $searchParams (optional) - The search params that are applied to the search the
     * SearchQueryCollector collects statements for
     */
    public function __construct(Search $Search, array $searchParams = [])
    {
        $this->Search = $Search;
        $this->searchParams = $searchParams;
    }

    /**
     * Add a WHERE statement
     *
     * @param string $statement - WHERE statement (e.g. "`productId` IS NOT NULL"); do NOT add "WHERE"
     * @param array<string, mixed> $binds (optional)
     *
     * $binds example:
     * [
     *      'varName' => [
     *          'value' => 'Patrick',
     *          'type' => \PDO::PARAM_STR
     *      ]
     * ]
     */
    public function addWhere(string $statement, array $binds = []): void
    {
        $this->where[] = $statement;
        $this->binds = array_merge($this->binds, $binds);
    }

    /**
     * @return array<int, string>
     */
    public function getWhereStatements(): array
    {
        return $this->where;
    }

    /**
     * @return array<string, mixed>
     */
    public function getBinds(): array
    {
        return $this->binds;
    }

    /**
     * @return array<string, mixed>
     */
    public function getSearchParams(): array
    {
        return $this->searchParams;
    }

    /**
     * @param array<string, mixed> $searchParams
     */
    public function setSearchParams(array $searchParams): void
    {
        $this->searchParams = $searchParams;
    }

    /**
     * @return Search
     */
    public function getSearch(): Search
    {
        return $this->Search;
    }
}
