<?php

namespace QUI\ERP\Products\Search;

use QUI;
use QUI\Exception;

use function array_filter;
use function array_unique;
use function explode;
use function is_array;
use function is_null;
use function is_string;
use function json_decode;
use function mb_substr;

/**
 * Class Utils
 */
class Utils
{
    /**
     * List of default frontend category fields
     *
     * @var array<int, QUI\ERP\Products\Field\Field>|null
     */
    protected static ?array $defaultFrontendFields = null;

    /**
     * List of default free text fields
     *
     * @var array<int, QUI\ERP\Products\Field\Field>|null
     */
    protected static ?array $defaultFreeTextFields = null;

    /**
     * @return array<string, mixed>
     * @throws Exception
     */
    public static function getSearchParameterFromRequest(): array
    {
        $search = QUI::getRequest()->get('search');
        $fields = QUI::getRequest()->get('f');
        $tags = QUI::getRequest()->get('t');
        $sortBy = QUI::getRequest()->get('sortBy');
        $sortOn = QUI::getRequest()->get('sortOn');

        if (isset($_REQUEST['search'])) {
            $search = $_REQUEST['search'];
        }

        $categories = QUI::getRequest()->get('c');

        $Site = QUI::getRewrite()->getSite();
        $defaultSorting = $Site?->getAttribute('quiqqer.products.settings.defaultSorting');

        if (empty($defaultSorting)) {
            $Package = QUI::getPackage('quiqqer/products');
            $defaultSorting = $Package->getConfig()?->get('products', 'defaultSortField');
        }

        if ($categories) {
            $categories = explode(',', $categories);
        }

        if (!is_array($categories)) {
            $categories = [];
        }

        // look for default site settings
        if (!$sortOn && !empty($defaultSorting)) {
            $sorting = explode(' ', $defaultSorting);
            $sortOn = $sorting[0];

            if (str_starts_with($sortOn, 'S')) {
                $sortOn = mb_substr($sortOn, 1);
            }
        }

        if (!$sortBy && !empty($defaultSorting)) {
            $sorting = explode(' ', $defaultSorting);

            if (isset($sorting[1])) {
                switch ($sorting[1]) {
                    case 'DESC':
                    case 'ASC':
                        $sortBy = $sorting[1];
                }
            }
        }

        $searchParams = array_filter([
            'freetext' => $search,
            'fields' => $fields,
            'tags' => $tags,
            'sortBy' => $sortBy,
            'sortOn' => $sortOn,
        ]);

        if (!empty($categories)) {
            $searchParams['categories'] = $categories;
        }

        if (isset($searchParams['fields'])) {
            if (is_string($searchParams['fields'])) {
                $searchParams['fields'] = json_decode($searchParams['fields'], true);
            }

            if (
                is_null($searchParams['fields'])
                && !empty($_REQUEST['f'])
                && is_string($_REQUEST['f'])
            ) {
                $searchParams['fields'] = json_decode($_REQUEST['f'], true);
            }

            if (is_null($searchParams['fields'])) {
                unset($searchParams['fields']);
            }
        }

        if (isset($searchParams['tags'])) {
            if (is_string($searchParams['tags'])) {
                $searchParams['tags'] = explode(',', $searchParams['tags']);
            } else {
                unset($searchParams['tags']);
            }
        }

        return $searchParams;
    }

    /**
     * @return mixed|string
     */
    public static function getViewParameterFromRequest(): mixed
    {
        $Session = QUI::getSession();
        $productView = $Session?->get('productView');

        if ($productView) {
            return $productView;
        }

        if (QUI::getRequest()->get('v')) {
            return QUI::getRequest()->get('v');
        }

        return '';
    }

    /**
     * Return the default frontend fields
     *
     * @return array<int, QUI\ERP\Products\Field\Field>|null
     * @throws Exception
     */
    public static function getDefaultFrontendFields(): ?array
    {
        if (self::$defaultFrontendFields !== null) {
            return self::$defaultFrontendFields;
        }

        $Package = QUI::getPackage('quiqqer/productsearch');
        $defaultIds = $Package->getConfig()?->get('search', 'frontend');
        $fields = [];

        if (is_string($defaultIds) && $defaultIds !== '') {
            $defaultIds = explode(',', $defaultIds);
            $defaultIds = array_unique($defaultIds);

            foreach ($defaultIds as $fieldId) {
                try {
                    $fields[] = QUI\ERP\Products\Handler\Fields::getField((int)$fieldId);
                } catch (QUI\Exception) {
                }
            }
        }

        self::$defaultFrontendFields = $fields;

        return $fields;
    }


    /**
     * Return the default frontend fields
     *
     * @return array<int, QUI\ERP\Products\Field\Field>|null
     * @throws Exception
     */
    public static function getDefaultFrontendFreeTextFields(): ?array
    {
        if (self::$defaultFreeTextFields !== null) {
            return self::$defaultFreeTextFields;
        }

        $Package = QUI::getPackage('quiqqer/productsearch');
        $defaultIds = $Package->getConfig()?->get('search', 'freetext');
        $fields = [];

        if (is_string($defaultIds) && $defaultIds !== '') {
            $defaultIds = explode(',', $defaultIds);
            $defaultIds = array_unique($defaultIds);

            foreach ($defaultIds as $fieldId) {
                try {
                    $fields[] = QUI\ERP\Products\Handler\Fields::getField((int)$fieldId);
                } catch (QUI\Exception) {
                }
            }
        }

        self::$defaultFreeTextFields = $fields;

        return $fields;
    }

    /**
     * Get quiqqer/productsearch config
     *
     * @return QUI\Config
     * @throws Exception
     */
    public static function getConfig(): QUI\Config
    {
        $Config = QUI::getPackage('quiqqer/productsearch')->getConfig();

        if ($Config === null) {
            throw new Exception('Config "quiqqer/productsearch" could not be loaded.');
        }

        return $Config;
    }
}
